#define sm_service_t SmService
#define sm_request_t SmRequest
#define sm_exc_t     SmExcpt

#include <l4/syscalls.h>
#include <services/server/decodelf.h>
#include <services/client/fileprov.h>
#include <services/client/rm.h>
#include <services/client/genericdm.h>
#include <services/lib/rm.h>
#include "elfexe.h"
#include <services/lib/decodelf.h>
#include <services/lib/utils.h>
#include <services/lib/cow.h>
#include <services/lib/nameservice.h>

typedef struct elfclient_s {
  l4_threadid_t task;
  unsigned elffile_index;
} elfclient_t;

typedef struct elffiledesc_s {
  segmentDesc_t elfSegments[16];
  dataspace_t ds;
  int refcount;
  int numElfSegments;
  unsigned entrypt;
} elffiledesc_t;

typedef int page_t[1024];
#define MAX_ELFFILE_PAGESIZE 4096 /* allow 16Mb of vaddrspace per elf file */
typedef page_t elffile_t[MAX_ELFFILE_PAGESIZE];

#define MAX_ELF_CLIENTS 64
#define MAX_ELF_FILES 16

elfclient_t elfclients[MAX_ELF_CLIENTS];
elffiledesc_t elffiledescs[MAX_ELF_FILES];
const elffile_t *elffiles = (const elffile_t *) 0xb0000000;
page_t initpages_mempool[MAX_ELF_CLIENTS+1];
page_t *initpages;

static l4_threadid_t myid;

void
elfDecoderMain (void)
{
  sm_request_t request;
  l4_ipc_buffer_t requestbuf;
  l4_msgdope_t msgdope;
  int i;

  myid = l4_myself();
  hprintf ("decodelf: alive, id = 0x%x\n", myid.dw);
  nameservice_init();
  nameservice_register_service("decodelf");
  
  (unsigned) initpages = ((unsigned)&initpages_mempool + 4095) &~ 4095;

  flick_init_request (&request, &requestbuf);

  msgdope = flick_server_wait(&request);
  if (L4_IPC_IS_ERROR(msgdope)) {
    hprintf("decodelf: flick_server_get_request failed, result=0x%x\n", msgdope);
    enter_kdebug ("decodelf: wait failed");
  }

  while (3) {
#if 0
    dbgprintf ("decodelf: got request from %x.%x, msgdope = 0x%x\n", 
            request.client_tid.id.task, request.client_tid.id.lthread, 
            msgdope);
    dbgprintf ("decodelf: msgbuf: %08x %08x %08x %08x\n",
      request.msgbuf->buffer[0], request.msgbuf->buffer[1],
      request.msgbuf->buffer[2], request.msgbuf->buffer[3]);
#endif
    if (msgdope.msgdope & 0xf0)
      enter_kdebug ("decodelf: ipc error");
    i = decodelf_dm_server(&request);

    switch(i) {
    case DISPATCH_ACK_SEND:
      msgdope = flick_server_reply_and_wait (&request);
      break;
    case DISPATCH_NO_REPLY:
      msgdope = flick_server_wait(&request);
      break;
    default:
      hprintf("decodelf: confused by return from decodelf_dm_server\n");
      enter_kdebug ("decodelf: confused by decodelf_dm_server");
    }
    /* should check msgdope here */
  }
}

void
decodelf_dm_server_transfer(sm_request_t *request, dsid_t ds,
                            threadid_t desttask, sm_exc_t *_ex)
{
  if ((unsigned)(ds >> 8) > MAX_ELF_CLIENTS ||
      TASKID(request->client_tid).dw != elfclients[ds>>8].task.dw) {
    hprintf ("decodelf: invalid dsid 0x%x for client %x.%x\n", ds,
             request->client_tid.id.task, request->client_tid.id.lthread);
    hprintf ("decodelf: elfclients[ds>>8].task.dw == 0x%x\n",
             elfclients[ds>>8].task.dw);
    enter_kdebug("decodelf: invalid dsid");
    /* throw exception here */
    return;
  }
  if ((ds & 0xff) != 0xff) 
    return;
  elfclients[ds>>8].task = TASKID((l4_threadid_t)desttask);
}

void
decodelf_dm_server_decode (sm_request_t *request, const dataspace_t *exesource,
                           dsid_t *init_dsid, sm_exc_t *_ev)
{
  int ret, i, client_index, file_index;
  segmentInfo_t *infopage;
  dbgprintf ("decodelf: in decodelf_dm_decode\n");
  
  /* find a free client descriptor struct */
  for (client_index = 0; client_index < MAX_ELF_CLIENTS; client_index++)
    if (elfclients[client_index].task.dw == L4_NIL_ID.dw)
      break;
  if (client_index == MAX_ELF_CLIENTS) {
    hprintf ("decodelf: too many clients\n");
    enter_kdebug ("decodelf: too many clients");
    /* throw exception here */
    return;
  }

  elfclients[client_index].task = TASKID(request->client_tid);
  
  for (file_index = 0; file_index < MAX_ELF_FILES; file_index++)
    if ((elffiledescs[file_index].ds.dsm  == exesource->dsm) &&
        (elffiledescs[file_index].ds.dsid == exesource->dsid)) {
      dbgprintf ("decodelf: file already open");
      break;
    }
  if (file_index == MAX_ELF_FILES) {
    for (file_index = 0; file_index < MAX_ELF_FILES; file_index++)
      if (elffiledescs[file_index].ds.dsm == L4_NIL_ID.dw)
        break;
    if (file_index == MAX_ELF_FILES) {
      hprintf ("decodelf: cannot open any more elf files\n");
      enter_kdebug ("decodelf: cannot open any more elf files");
      /* throw exception here */
      return;
    }
    dbgprintf ("decodelf: opening file {dsm:%x.%x,dsid:%x}\n",
               ((l4_threadid_t)exesource->dsm).id.task,
               ((l4_threadid_t)exesource->dsm).id.lthread,
               exesource->dsid);
    elffiledescs[file_index].ds = *exesource;
    elffiledescs[file_index].refcount = 0;
    attach_region (exesource, (unsigned)&elffiles[file_index], 
                   MAX_ELFFILE_PAGESIZE*4096, 0, PF_R);
    ret = verifyElfFormat (&elffiles[file_index]);
    if (ret) {
      hprintf ("decodelf: bad elf file format, errcode %d\n", ret);
      enter_kdebug("decodelf: bad elf file format");
      /* throw exception here */
      /* clean up here too */
      return;
    }
    buildSegmentList (&elffiles[file_index], 
                      elffiledescs[file_index].elfSegments,
                      &elffiledescs[file_index].numElfSegments,
                      &elffiledescs[file_index].entrypt);

    for (i = 0; i < elffiledescs[file_index].numElfSegments; i++)
      dbgprintf ("decodelf: segment 0x%x->0x%x size 0x%x %c%c%c\n", 
                 elffiledescs[file_index].elfSegments[i].srcbase, 
                 elffiledescs[file_index].elfSegments[i].destbase, 
                 elffiledescs[file_index].elfSegments[i].memsize,
                (elffiledescs[file_index].elfSegments[i].access &PF_R)?'r':' ',
                (elffiledescs[file_index].elfSegments[i].access &PF_W)?'w':' ',
                (elffiledescs[file_index].elfSegments[i].access &PF_X)?'x':' ');

  }
  
  infopage = (segmentInfo_t *) &initpages[client_index];

  /* prepare the segment info page for our client */
  for (i = 0; i < elffiledescs[file_index].numElfSegments; i++) {
    infopage[i].destbase = elffiledescs[file_index].elfSegments[i].destbase;
    infopage[i].size = elffiledescs[file_index].elfSegments[i].memsize;
    infopage[i].access = elffiledescs[file_index].elfSegments[i].access;
    infopage[i].dsid = i | (client_index << 8);
  }
  infopage[i].destbase = (unsigned) -1;
  infopage[i].size = elffiledescs[file_index].entrypt;

  elfclients[client_index].elffile_index = file_index;
  elffiledescs[file_index].refcount++;
  *init_dsid = (client_index << 8) | 0xff;
}

void
decodelf_dm_server_decode_4debug (sm_request_t *request,
                                  const dataspace_t *exesource,
                                  dsid_t *init_dsid, sm_exc_t *_ev)
{
  int ret, i, client_index, file_index;
  segmentInfo_t *infopage;
  dbgprintf ("decodelf: in decodelf_dm_decode\n");
  
  /* find a free client descriptor struct */
  for (client_index = 0; client_index < MAX_ELF_CLIENTS; client_index++)
    if (elfclients[client_index].task.dw == L4_NIL_ID.dw)
      break;
  if (client_index == MAX_ELF_CLIENTS) {
    hprintf ("decodelf: too many clients\n");
    enter_kdebug ("decodelf: too many clients");
    /* throw exception here */
    return;
  }

  elfclients[client_index].task = TASKID(request->client_tid);
  
  for (file_index = 0; file_index < MAX_ELF_FILES; file_index++)
    if ((elffiledescs[file_index].ds.dsm  == exesource->dsm) &&
        (elffiledescs[file_index].ds.dsid == exesource->dsid)) {
      dbgprintf ("decodelf: file already open");
      break;
    }
  if (file_index == MAX_ELF_FILES) {
    for (file_index = 0; file_index < MAX_ELF_FILES; file_index++)
      if (elffiledescs[file_index].ds.dsm == L4_NIL_ID.dw)
        break;
    if (file_index == MAX_ELF_FILES) {
      hprintf ("decodelf: cannot open any more elf files\n");
      enter_kdebug ("decodelf: cannot open any more elf files");
      /* throw exception here */
      return;
    }
    dbgprintf ("decodelf: opening file {dsm:%x.%x,dsid:%x}\n",
               ((l4_threadid_t)exesource->dsm).id.task,
               ((l4_threadid_t)exesource->dsm).id.lthread,
               exesource->dsid);
    elffiledescs[file_index].ds = *exesource;
    elffiledescs[file_index].refcount = 0;
    attach_region (exesource, (unsigned)&elffiles[file_index], 
                   MAX_ELFFILE_PAGESIZE*4096, 0, PF_R);
    ret = verifyElfFormat (&elffiles[file_index]);
    if (ret) {
      hprintf ("decodelf: bad elf file format, errcode %d\n", ret);
      enter_kdebug("decodelf: bad elf file format");
      /* throw exception here */
      /* clean up here too */
      return;
    }
    buildSegmentList (&elffiles[file_index], 
                      elffiledescs[file_index].elfSegments,
                      &elffiledescs[file_index].numElfSegments,
                      &elffiledescs[file_index].entrypt);

    for (i = 0; i < elffiledescs[file_index].numElfSegments; i++) {
      elffiledescs[file_index].elfSegments[i].access |= PF_W; /* force */
      dbgprintf ("decodelf: segment 0x%x->0x%x size 0x%x %c%c%c\n", 
                 elffiledescs[file_index].elfSegments[i].srcbase, 
                 elffiledescs[file_index].elfSegments[i].destbase, 
                 elffiledescs[file_index].elfSegments[i].memsize,
                (elffiledescs[file_index].elfSegments[i].access &PF_R)?'r':' ',
                (elffiledescs[file_index].elfSegments[i].access &PF_W)?'w':' ',
                (elffiledescs[file_index].elfSegments[i].access &PF_X)?'x':' ');
    }
  }
  
  infopage = (segmentInfo_t *) &initpages[client_index];

  /* prepare the segment info page for our client */
  for (i = 0; i < elffiledescs[file_index].numElfSegments; i++) {
    infopage[i].destbase = elffiledescs[file_index].elfSegments[i].destbase;
    infopage[i].size = elffiledescs[file_index].elfSegments[i].memsize;
    infopage[i].access = elffiledescs[file_index].elfSegments[i].access;
    infopage[i].dsid = i | (client_index << 8);
  }
  infopage[i].destbase = (unsigned) -2;
  infopage[i].size = elffiledescs[file_index].entrypt;

  elfclients[client_index].elffile_index = file_index;
  elffiledescs[file_index].refcount++;
  *init_dsid = (client_index << 8) | 0xff;
}
 
void
decodelf_dm_server_close(sm_request_t *request, 
                         dsid_t ds, 
                         sm_exc_t *_ev)
{
  int file_index;
  sm_exc_t exc;
  dbgprintf ("decodelf: close 0x%x from %x.%x\n", ds, request->client_tid.id.task,
           request->client_tid.id.lthread);
  if ((unsigned)(ds >> 8) > MAX_ELF_CLIENTS ||
      TASKID(request->client_tid).dw != elfclients[ds>>8].task.dw) {
    hprintf ("decodelf: invalid dsid 0x%x for client %x.%x\n", ds,
             request->client_tid.id.task, request->client_tid.id.lthread);
    hprintf ("decodelf: elfclients[ds>>8].task.dw == 0x%x\n",
             elfclients[ds>>8].task.dw);
    enter_kdebug("decodelf: invalid dsid");
    /* throw exception here */
    return;
  }
  if ((ds & 0xff) != 0xff)
    return;
  file_index = elfclients[ds>>8].elffile_index;
  if (--elffiledescs[file_index].refcount == 0) {
    detach_dataspace (&elffiledescs[file_index].ds);
    genericdm_dm_close ((sm_service_t)elffiledescs[file_index].ds.dsm,
                        elffiledescs[file_index].ds.dsid, &exc);
    elffiledescs[file_index].ds.dsm = L4_NIL_ID.dw;
  }
  elfclients[ds>>8].task = L4_NIL_ID;
  dealloc_copies(TASKID(request->client_tid).dw);
}
 
void
decodelf_dm_server_map_page(sm_request_t *request, dsid_t ds, 
                            sdword_t offset, l4_snd_fpage_t *mapping,
                            sm_exc_t *_ev)
{
  unsigned srcaddr, destaddr;
  int in_file, writeable, num_bytes_to_copy;
  char ch;

  /* ds >> 8 is client index, ds & 0xff is segment index */

  if ((unsigned)(ds >> 8) > MAX_ELF_CLIENTS ||
      TASKID(request->client_tid).dw != elfclients[ds>>8].task.dw) {
    hprintf ("decodelf: invalid dsid 0x%x for client %x.%x\n", ds,
             request->client_tid.id.task, request->client_tid.id.lthread);
    hprintf ("decodelf: elfclients[ds>>8].task.dw == 0x%x\n",
             elfclients[ds>>8].task.dw);
    enter_kdebug("decodelf: invalid dsid");
    /* throw exception here */
    return;
  }
  if ((ds & 0xff) == 0xff) {
    destaddr = offset;
    srcaddr = (unsigned) &initpages[ds>>8];
    writeable = 0;
    in_file = 0;
  } else {
    int fileindex;
    fileindex = elfclients[ds>>8].elffile_index;
    if (((ds & 0xff) >= elffiledescs[fileindex].numElfSegments) || 
        (offset >= elffiledescs[fileindex].elfSegments[ds&0xff].memsize)) {
      hprintf ("decodelf: bad dsid or segment offset\n");
      enter_kdebug ("decodelf: bad dsid or seg offset");
      /* throw exception here */
      return;
    }
    srcaddr = (unsigned)elffiledescs[fileindex].elfSegments[ds&0xff].srcbase +
              (offset &~ 4095);
    destaddr = elffiledescs[fileindex].elfSegments[ds&0xff].destbase + 
               (offset &~ 4095);
    writeable = !!(elffiledescs[fileindex].elfSegments[ds&0xff].access & PF_W);
    num_bytes_to_copy = elffiledescs[fileindex].elfSegments[ds&0xff].filesize
                        - (offset &~ 4095);
    if (num_bytes_to_copy > 4096)  num_bytes_to_copy = 4096;
    else if (num_bytes_to_copy <= 0) num_bytes_to_copy = 0;
  }
  if (writeable) {
    l4_threadid_t copy_id;
    dbgprintf ("decodelf: dest=%x, numbytes=%x\n", destaddr, num_bytes_to_copy);
    copy_id = request->client_tid;
    /* we need next line to distinguish between 2 writeable segments sharing
       the same page in the elf file */
    copy_id.id.lthread = ds & 0xff;
    srcaddr = (num_bytes_to_copy == 0) ? 
      (unsigned) get_zeroed_copy ((void*)destaddr, copy_id.dw):
      (unsigned) get_partial_page_copy ((void*)srcaddr, copy_id.dw,
                                        num_bytes_to_copy);
    ch = *(volatile char *) srcaddr;
    *(volatile char *) srcaddr = ch;
  } else
    *(volatile char *) srcaddr;
  mapping->snd_base = destaddr; /* this should ignored by client rm */
  mapping->fpage = l4_fpage (srcaddr, 12, writeable, 0);
  dbgprintf ("decodelf: sending fpage=0x%x\n", mapping->fpage);
} 

static int elfDecoderStack [1024];

int
main (void)
{
  create_thread (2, elfDecoderMain, &elfDecoderStack[1022], l4_myself());
  rm_main_thread(1);
  return 0; /* should never return */
}

