#include "elfexe.h"
#include <services/lib/utils.h>

int
verifyElfFormat (void *elfFile)
     /* returns nonzero if *elfFile is not an elf file appropriate for 
      * loading */
{
  Elf32_Ehdr *ehdr;

  ehdr = (Elf32_Ehdr *) elfFile;
  if (ehdr->e_ident[0] != 0x7f
      || ehdr->e_ident[1] != 'E'
      || ehdr->e_ident[2] != 'L'
      || ehdr->e_ident[3] != 'F')
    return 1; /* not elf */
  if (ehdr->e_ident[4] != 1)
    return 2; /* not 32-bit */
  if (ehdr->e_ident[5] != 1)
    return 3; /* not lsb */
  if (ehdr->e_ident[6] != 1 || ehdr->e_version != 1)
    return 4; /* not elf version 1 */
  if (ehdr->e_type != ET_EXEC)
    return 5; /* not an executable */
  if (ehdr->e_machine != EM_386)
    return 6; /* wrong CPU architecture */
  return 0; /* all checks succeeded */
}

segmentDesc_t *
buildSegmentList (void *elfFile, segmentDesc_t *elfSegments, 
                  int *p_numElfSegments, unsigned *entry)
     /* returns a linked list of program segment descriptors */
{
  int numElfSegments = 0;
  Elf32_Ehdr *ehdr;
  Elf32_Phdr *phdr;
  int segIndex, i;

  ehdr = (Elf32_Ehdr *) elfFile;
  phdr = (Elf32_Phdr *) (ehdr->e_phoff + (unsigned) elfFile);
  dbgprintf ("elfdm: ehdr->e_phnum == %d\n", ehdr->e_phnum);
  for (segIndex = 0; segIndex < ehdr->e_phnum; segIndex++)
    if (phdr[segIndex].p_type == PT_DYNAMIC
	|| phdr[segIndex].p_type == PT_INTERP
	|| phdr[segIndex].p_type == PT_SHLIB)
      return NULL;
    else if (phdr[segIndex].p_type == PT_LOAD
	     || phdr[segIndex].p_type == PT_NOTE
	     || phdr[segIndex].p_type == PT_PHDR)
      numElfSegments++;

  for (segIndex = 0, i = 0; segIndex < ehdr->e_phnum; segIndex++) {
    if (phdr[segIndex].p_type == PT_LOAD
	|| phdr[segIndex].p_type == PT_NOTE
	|| phdr[segIndex].p_type == PT_PHDR) {
      elfSegments[i].srcbase = (void *) ((phdr[segIndex].p_offset &~ 0xfff) + 
				       (unsigned) elfFile);
      elfSegments[i].destbase = phdr[segIndex].p_vaddr &~ 0xfff;
      elfSegments[i].memsize = phdr[segIndex].p_memsz + 
        (phdr[segIndex].p_vaddr & 0xfff);
      elfSegments[i].memsize = (elfSegments[i].memsize + 4095) &~ 0xfff;
      elfSegments[i].filesize = phdr[segIndex].p_filesz +
        (phdr[segIndex].p_offset & 0xfff);
      /*elfSegments[i].filesize = (elfSegments[i].filesize + 4095) &~ 0xfff;*/

      elfSegments[i].access = phdr[segIndex].p_flags & ~PF_MASKPROC;
      if (i++ == 14) {
        hprintf ("elfdm: too many loadable segments\n");
        enter_kdebug ("elfdm: too many segs");
        break;
      }
    }
  }
  *p_numElfSegments = (numElfSegments <= 16) ? numElfSegments : 16;
  *entry = ehdr->e_entry;
  return elfSegments;
}
