#include <l4/types.h>
#include <l4/syscalls.h>
#include <l4/ipc.h>
#include <l4/librmgr.h>
#include <flux/lmm.h>
#include <malloc.h>
#include <stdio.h>
#include <services/lib/queue.h>
#include <services/server/dm_anon.h>
#include <services/lib/nameservice.h>
#include <services/lib/sm_handle.h>
#include <services/sys/services.h>

/**** test_dm_anon: switch to 1 ****/
int debug = 0;
int debug_msgs = 0;	/* Dump msgs to/from clients */

/**** test_dm_anon defines ****/
//int dm_test = 1;
//char test_memory[0x40000];        /* 256 KB */


#define TRUNC_PAGE(x)	(dword_t)(((dword_t)(x)) & ~(L4_PAGESIZE - 1))
#define ROUND_PAGE(x)	(dword_t)(TRUNC_PAGE(x) + L4_PAGESIZE)
#define MEM_MAX		(128L*1024L*1024L)	/* From rmgr memmap.h */

#define MALLOC_POOL_SIZE 0x40000	/* 256 KB */
static char pool[MALLOC_POOL_SIZE];
static lmm_region_t region;

void _start, _end;

typedef struct page_s {
    queue_chain_t pageq;
    queue_head_t contig_page_queue;
    dword_t addr;			/* Phys addr */
    dword_t offset;			/* Offset within dataspace */
} *page_t;
queue_head_t free_page_queue;

static __inline__ l4_threadid_t
TASKID(l4_threadid_t tid)
{
    l4_threadid_t newtid;

    newtid = tid;
    newtid.id.lthread = 0;
    return newtid;
}

typedef struct dataspace_s {
  queue_chain_t dataspaceq;
  int descriptor;
  l4_threadid_t owner;
  queue_head_t alloc_page_queue;
  int log2_pagesize;
} *dataspace_t;

#define MAX_HANDLES 1000


/* defines the malloc function for the sm_handle lib */
char *sm_malloc(int s) {
  return malloc(s);
}

void  
dump(dword_t *buf, int len)
{
    int i;

    for (i = 0; i < len; i++) {
        printf("%08x ", buf[i]);
        if (!((i + 1) % 8)) printf("\n");
    }       
    if (!((i + 1) % 8)) printf("\n");
}

void
setup_malloc_mem(void)
{
    lmm_init(&malloc_lmm);
    lmm_add_region(&malloc_lmm, &region, (void *)0, (vm_size_t) - 1, 0, 0);
    lmm_add_free(&malloc_lmm, pool, MALLOC_POOL_SIZE);
}

void
destroy_malloc_mem(void)
{
    lmm_init(&malloc_lmm);
}


/* NOTE: Another library?  */
/* Store owner as thread id */
sm_handle_t safe_new_sm_handle(l4_threadid_t owner, unsigned perms, void
			       *attached_obj, int table_incr) {
  sm_handle_t h;
  LnThread o = (unsigned) owner.dw;

  h = new_sm_handle(o, perms, attached_obj);
  if (debug) {
    printf("safe_new: owner: 0x%x; obj: 0x%x; perms: 0x%x\n", 
	   (unsigned) o, (unsigned) attached_obj, perms);
    printf("safe_new: next free spot: %d\n", min_max_handles_sm_handle());
  }

  if (h == SM_HANDLE_INVALID) {
    printf("Table full\n");
    printf("Handle used = %d\n", used_handles_sm_handle());
    printf("Handle table size = %d\n", max_handles_sm_handle());
      
    if (table_incr > 0) {
      if (reinitialize_sm_handle_system(max_handles_sm_handle() 
				       + table_incr)) {
	printf("Error: reinitialization problem\n");
	return h;
      }
      printf("ReInitialized: max handles = %d\n", max_handles_sm_handle());
      h = new_sm_handle(o, perms, attached_obj);
    }
  }
  if (debug) dump_sm_handle(h);
  return h;
}


void
dm_anon_ds_server_open(sm_request_t *request, dsid_t *ds, sm_exc_t *_ev)
{
    dataspace_t ds_struct;

    /* NOTE: Need to authorize open */

    /* Create dataspace */
    ds_struct = (dataspace_t)malloc(sizeof(struct dataspace_s));
    if (!ds_struct) {
	printf("dm_anon: open, no memory\n");
	flick_throw_exception(exc_dm_anon_no_memory);
	return;
    }
    queue_init(&ds_struct->alloc_page_queue);
    /* Set the default page size, a single page */
    ds_struct->log2_pagesize = L4_LOG2_PAGESIZE;
    /* Set all perms on for now */
    *ds = (dsid_t) safe_new_sm_handle(request->client_tid, 0xffffffff, 
				      ds_struct, MAX_HANDLES);

    if (debug) printf("dm_anon: open 0x%x; at 0x%x for object 0x%x\n", 
		      *ds, SM_HANDLE_PTR(*ds), ds_struct);
    if (!SM_HANDLE_ACTIVE((sm_handle_t)*ds)) {
      	printf("dm_anon: bad handle\n");
	/*
	printf("dm_anon: table: 0x%x; index: %d; hptr: 0x%x\n", 
	       (unsigned) sm_ht,
	       ((sm_handle_t)*ds & SM_HANDLE_MASK),
	       SM_HANDLE_PTR((sm_handle_t) *ds));
	*/
	enter_kdebug("handle problem");
	*ds = 0;
	/* Do we need to free the ds_struct? */
	flick_throw_exception(exc_dm_anon_no_descriptor);
    }
}

void
dm_anon_ds_server_open_contig(sm_request_t *request, dsid_t *ds,
			      sword_t log2_pagesize, sm_exc_t *_ev)
{
    dataspace_t ds_struct;

    /* NOTE: Need to authorize open */

    /* Check the page size, must be between 1 KB and 1 MB */
    if (log2_pagesize < 12 || log2_pagesize > 20) {
	printf("dm_anon: open_contig, invalid page size\n");
	flick_throw_exception(exc_dm_anon_invalid_pagesize);
	return;
    }

    /* Create dataspace */
    ds_struct = (dataspace_t)malloc(sizeof(struct dataspace_s));
    if (!ds_struct) {
	printf("dm_anon: open, no memory\n");
	flick_throw_exception(exc_dm_anon_no_memory);
	return;
    }
    queue_init(&ds_struct->alloc_page_queue);
    ds_struct->log2_pagesize = log2_pagesize;
    /* Set all perms on for now */
    *ds = (dsid_t) safe_new_sm_handle(request->client_tid, 0xffffffff, 
				      ds_struct, MAX_HANDLES);

    if (debug) printf("dm_anon: open 0x%x; at 0x%x for object 0x%x\n", 
		      *ds, SM_HANDLE_PTR(*ds), ds_struct);
    if (!SM_HANDLE_ACTIVE((sm_handle_t)*ds)) {
      	printf("dm_anon: bad handle\n");
	/*
	printf("dm_anon: table: 0x%x; index: %d; hptr: 0x%x\n", 
	       (unsigned) sm_ht,
	       ((sm_handle_t)*ds & SM_HANDLE_MASK),
	       SM_HANDLE_PTR((sm_handle_t) *ds));
	*/
	enter_kdebug("handle problem");
	*ds = 0;
	/* Do we need to free the ds_struct? */
	flick_throw_exception(exc_dm_anon_no_descriptor);
    }
}

int
page_count(int log2_pagesize)
{
    int i, count = 1;

    for (i = 0; i < log2_pagesize - L4_LOG2_PAGESIZE; i++) count *= 2;
    return count;
}

void
dm_anon_ds_server_close(sm_request_t *request, dsid_t ds, sm_exc_t *_ev)
{
    dataspace_t ds_struct;
    page_t p, first_p;
    sm_handle smh, *hptr = &smh;

    /* Find the dataspace */
    if ((read_sm_handle(ds, hptr) != SM_OK) || 
	(!AUTHORIZE_SM_HANDLE_BITOP(hptr, request->client_tid, 
				   req_dm_anon_ds_close))) { 
      /* !Legit || !authed handle */
      printf("dm_anon: close, invalid handle-%d;\nhptr: 0x%x; hptr->perms 0x%x; h 0x%x; h->perms 0x%x\n", 
	     ds, (unsigned) hptr, hptr->perms, 
	     (unsigned) SM_HANDLE_PTR(ds), SM_HANDLE_PTR(ds)->perms);
      flick_throw_exception(exc_dm_anon_invalid_handle);
      return;
    }
    
    ds_struct = (dataspace_t)attached_obj_sm_handle(hptr);

    /* Unmap and free all pages */
    while (!queue_empty(&ds_struct->alloc_page_queue)) {
        queue_remove_first(&ds_struct->alloc_page_queue, first_p,
			   page_t, pageq);
        l4_fpage_unmap(l4_fpage(first_p->addr, ds_struct->log2_pagesize, 0, 0),
                       L4_FP_OTHER_SPACES);
	first_p->offset = -1;
	queue_enter(&free_page_queue, first_p, page_t, pageq);
	while (!queue_empty(&first_p->contig_page_queue)) {
	    /* Keep pages contiguous! */
	    queue_remove_first(&first_p->contig_page_queue, p, page_t, pageq);
	    queue_enter(&free_page_queue, p, page_t, pageq);
	}
    }

    /* Remove dataspace from queue, free dataspace data structure */
    free_sm_handle(ds);
    free(ds_struct);

    if (debug) printf("dm_anon: close %d\n", ds);
}

void
dm_anon_ds_server_map_page(sm_request_t *request, dsid_t ds,
			   sdword_t offset, l4_snd_fpage_t *page,
			   sm_exc_t *_ev)
{
    dataspace_t ds_struct;
    page_t p, first_p, last_p;
    sm_handle smh, *hptr = &smh;
    int i;


    /* Find the dataspace */
    offset &=~ 0x4095;

    /* NOTE: More authorization than simply map/not required? */
    if ((read_sm_handle(ds, hptr) != SM_OK) || 
	(!AUTHORIZE_SM_HANDLE_BITOP(hptr, request->client_tid, 
				   req_dm_anon_ds_map_page))) { 
      /* !Legit || !authed handle */
      printf("dm_anon: map_page, invalid handle: %d\n", ds);
      flick_throw_exception(exc_dm_anon_invalid_handle);
      return;
    }

    ds_struct = (dataspace_t)attached_obj_sm_handle(hptr);

    /* Has the page been mapped from a previous attach? */
    queue_iterate(&ds_struct->alloc_page_queue, p, page_t, pageq) {
	if (p->offset == offset) break;
    }

    /* If the page has not been mapped, map a free page */
    if (queue_empty(&ds_struct->alloc_page_queue) ||
	p->offset != offset) {
	/* Make sure we have free pages */
	if (queue_empty(&free_page_queue)) {
	    printf("dm_anon: map_page, no free page\n");
	    flick_throw_exception(exc_dm_anon_no_free_page);
	    return;
	}

	/* Try to get the necessary contiguous free pages */
	queue_iterate(&free_page_queue, first_p, page_t, pageq) {
	    last_p = first_p;
	    for (i = 1; i < page_count(ds_struct->log2_pagesize); i++) {
		p = (page_t)queue_next(&last_p->pageq);
		if (p->addr - last_p->addr != L4_PAGESIZE) break;
		last_p = p;
	    }
	    if (i == page_count(ds_struct->log2_pagesize)) goto found;
	}

	printf("dm_anon: map_page, no contig page\n");
	flick_throw_exception(exc_dm_anon_no_contig_page);
	return;

	/*
	 * Get free contiguous pages, zero them.  Add the first to the
	 * alloc page queue.  Add the others to the contig page queue.
	 */
found:	queue_remove_first(&free_page_queue, first_p, page_t, pageq);
	memset((void *)first_p->addr, 0, L4_PAGESIZE);
	queue_enter(&ds_struct->alloc_page_queue, first_p, page_t, pageq);
	first_p->offset = offset;
	p = (page_t)queue_next(&first_p->pageq);
	for (i = 1; i < page_count(ds_struct->log2_pagesize); i++) {
	    queue_remove_first(&free_page_queue, p, page_t, pageq);
	    memset((void *)p->addr, 0, L4_PAGESIZE);
	    queue_enter(&first_p->contig_page_queue, p, page_t, pageq);
	    /* Offset is not valid for these pages */
	    p = (page_t)queue_next(&p->pageq);
	}
    }

    page->snd_base = first_p->addr;
    page->fpage = l4_fpage(first_p->addr, ds_struct->log2_pagesize,
			   L4_FPAGE_RW, L4_FPAGE_MAP);

    if (debug) printf("dm_anon: map_page, 0x%x to <%d, 0x%x>, size %d\n",
		      first_p->addr, ds, offset, ds_struct->log2_pagesize);
}

void
dm_anon_ds_server_paddr(sm_request_t *request, dsid_t ds,
			sdword_t offset, sdword_t *paddr, sm_exc_t *_ev)
{
    dataspace_t ds_struct;
    page_t p;
    sm_handle smh, *hptr = &smh;


    /* Find the dataspace */
    if ((read_sm_handle(ds, hptr) != SM_OK) || 
	(!AUTHORIZE_SM_HANDLE_BITOP(hptr, request->client_tid, 
				   req_dm_anon_ds_paddr))) { 
      /* !Legit || !authed handle */
      printf("dm_anon: map_page, invalid handle-%d\n", ds);
      flick_throw_exception(exc_dm_anon_invalid_handle);
      return;
    }

    ds_struct = (dataspace_t)attached_obj_sm_handle(hptr);

    /* Find page */
    queue_iterate(&ds_struct->alloc_page_queue, p, page_t, pageq) {
	if (p->offset == offset) break;
    }
    if (p->offset != offset) {
	printf("dm_anon: paddr, invalid offset-0x%x\n", offset);
	flick_throw_exception(exc_dm_anon_invalid_offset);
	return;
    }

    *paddr = p->addr;

    if (debug) printf("dm_anon: paddr <%d, 0x%x> = 0x%x\n",
		      ds, p->offset, p->addr);
}

int
main(void)
{
    dword_t p;
    sm_request_t request;
    l4_ipc_buffer_t buf;
    dword_t dummy;
    l4_threadid_t my_preempter, my_pager;
    int count = 0, free_pages = 0;

    if (debug) printf("dm_anon: Initializing malloc mem\n");
    setup_malloc_mem();

    /* Initialize dataspace queue, descriptor count */
    if (debug) printf("dm_anon: Initializing dataspace queue\n");
    if ((initialize_sm_handle_system(MAX_HANDLES)) != SM_OK) {
      printf("Error: initialization problem\n");
      exit(-1);
    }

    /* Initialize free page queue, physical memory dataspaces */
    if (debug) printf("dm_anon: Grabbing free pages from rmgr\n");
    queue_init(&free_page_queue);
    my_preempter = L4_INVALID_ID;
    my_pager = L4_INVALID_ID;
    l4_thread_ex_regs(l4_myself(), -1, -1, &my_preempter, &my_pager,
		      &dummy, &dummy, &dummy);
    for (p = 0; p < MEM_MAX; p += L4_PAGESIZE) {
        page_t page;
	l4_snd_fpage_t sfpage;
	l4_msgdope_t result;

	/* Skip the kernel info page, adapter pages, dm_anon image */
	if (p <= 0x100000 ||
	    p >= TRUNC_PAGE(&_start) && p < ROUND_PAGE(&_end))
	    continue;

	count++;
	l4_i386_ipc_call(my_pager, L4_IPC_SHORT_MSG,
			 p | 1, L4_LOG2_PAGESIZE << 2, dummy,
			 L4_IPC_MAPMSG(0, L4_WHOLE_ADDRESS_SPACE),
			 &sfpage.snd_base, &sfpage.fpage.fpage, &dummy,
			 L4_IPC_NEVER, &result);
	if (l4_ipc_fpage_received(result) &&
	    l4_ipc_is_fpage_writable(sfpage.fpage)) {
	    page = (page_t)malloc(sizeof(struct page_s));
	    if (!page) {
		printf("dm_anon: No memory\n");
		exit(-1);
	    }
	    queue_init(&page->contig_page_queue);
	    page->addr = p;
	    page->offset = -1;
	    queue_enter(&free_page_queue, page, page_t, pageq);
	    free_pages++;
	}
    }

    /* Use back-up page grabbing plan: for test program only */
    /**** For test_dm_anon ****/
    /*
    if ((free_pages == 0) && (dm_test)) {
      page_t page;

      printf("dm_anon: Trying backup page grab...\n");
      p = ((unsigned) (&test_memory) + 0x1000)  & 0xfffff000; 
      printf("dm_anon: memory range 0x%x to 0x%x\n", p, p + 0x39000);

      for (; p < ((unsigned) &test_memory + 0x39000); p += L4_PAGESIZE) {      
	count++;
	
	page = (page_t) malloc(sizeof(struct page_s));
	if (!page) {
	  printf("dm_anon: No memory\n");
	  exit(-1);
	}
	page->addr = p;
	page->offset = -1;
	queue_enter(&free_page_queue, page, page_t, pageq);
	free_pages++;
      }
    }
    */
    /* end: **** test_dm_anon ****/

    printf("dm_anon: Tried %d pages (%d KB), grabbed %d pages (%d KB)\n",
           count, count * L4_PAGESIZE / 1024,
	   free_pages, free_pages * L4_PAGESIZE / 1024);

    /* Register */
    nameservice_init();
    /**** test_dm_anon ****
    nameservice_register_service("handle_dm_anon");
    ****/
    nameservice_register_service(DM_ANON);

    /* Service requests */
    if (debug) printf("dm_anon: Servicing requests\n");
    flick_init_request(&request, &buf);
    if (L4_IPC_IS_ERROR(flick_server_wait(&request))) {
	printf("dm_anon: flick_server_wait failed\n");
	exit(-1);
    }
    while (1) {
	if (debug_msgs) {
	    printf("dm_anon: from, ");
	    dump(request.msgbuf->buffer, 7);
	}
	dm_anon_ds_server(&request);
	if (debug_msgs) {
	    printf("dm_anon: to, ");
	    dump(request.msgbuf->buffer, 7);
	}
	if (L4_IPC_IS_ERROR(flick_server_reply_and_wait(&request))) {
	    printf("dm_anon: flick_server_replay_and_wait failed\n");
	    exit(-1);
	}
    }

    enter_kdebug("dm_anon: Exited service loop");
}
