#define sm_service_t SmService
#define sm_request_t SmRequest
#define sm_exc_t     SmExcpt

#include <l4/syscalls.h>
#include <services/server/fileprov.h>
#include <services/lib/utils.h>
#include <stdio.h>
#include <assert.h>
#include <sys/stat.h>
#include <unistd.h>
#include <services/lib/nameservice.h>

typedef struct cache_page_descr_s cache_page_descr_t;
struct cache_page_descr_s {
  int file_page_num;
  int handle;
  void *mem_buf;
};

#define MAX_HANDLES 32
typedef struct filehandle_s {
  FILE *file;
  /* following information is used to tell whether to open the file again */
  __dev_t device;
  __ino_t inode;
  __off_t size;
  __time_t mtime;
} filehandle_t;

filehandle_t filehandles[MAX_HANDLES];

#define NUM_CACHE_PAGES 512

cache_page_descr_t cache_page_descrs[NUM_CACHE_PAGES];
unsigned char cache_pages_array[(NUM_CACHE_PAGES+1) * 4096];
unsigned char *cache_pages = 0;

int next_flushable; /* no point doing a decent page replacement
		     * policy in this interim code */

#define D_OPEN 0
#define D_CLOSE 1
#define D_PAGEFAULT 2

void *
find_page_in_cache (int handle, unsigned int fault_addr)
{
  int pgnum;
  unsigned fault_page;
  fault_page = fault_addr / 4096;
  for (pgnum = 0; pgnum < NUM_CACHE_PAGES; pgnum++)
    if (cache_page_descrs[pgnum].file_page_num == fault_page
        && cache_page_descrs[pgnum].handle == handle
	&& cache_page_descrs[pgnum].mem_buf != NULL)
      break;
  if (pgnum == NUM_CACHE_PAGES)
    return NULL;
  else {
    return cache_page_descrs[pgnum].mem_buf;
  }
}

void *
get_page (int handle, unsigned int fault_addr)
{
  int ret;
  void *mem_buf;
  FILE *f;
  if (((unsigned)handle >= MAX_HANDLES) || (filehandles[handle].file == NULL))
    return NULL; /* bad handle */
  mem_buf = find_page_in_cache (handle, fault_addr);
  if (mem_buf != NULL) {
    dbgprintf ("found [handle:%x,offs:%x] in cache at addr %x\n", handle,
             fault_addr, mem_buf);
    return mem_buf;
  }

  /* now flush the next page (no fancy page replacement policy) */
  if (cache_page_descrs[next_flushable].mem_buf != NULL) {
    l4_fpage_t doomed_fpage;
    dbgprintf("fileprov: flushing cache page %d\n", next_flushable);
    doomed_fpage = l4_fpage ((unsigned long)&cache_pages[4096*next_flushable], 
			     12, 0, 0);
    l4_fpage_unmap (doomed_fpage, L4_FP_FLUSH_PAGE);
  }

  /* now read 4kb from the file into the cache page */
  f = filehandles[handle].file;
  ret = fseek (f, fault_addr & (~4095), SEEK_SET);
  assert (ret == 0); /* do proper error handling later */
  memset (&cache_pages[4096*next_flushable], 0, 4096);
  ret = fread (&cache_pages[4096*next_flushable], 4096, 1, f);
  /* assert (ret == 1); */

  /* hprintf("runl4: read from file: %x %x %x %x\n",
          cache_pages[4096*next_flushable+0],
          cache_pages[4096*next_flushable+1],
          cache_pages[4096*next_flushable+2],
          cache_pages[4096*next_flushable+3]); */

  /* fix up the cache page descriptor and return */
  cache_page_descrs[next_flushable].file_page_num = fault_addr / 4096;
  cache_page_descrs[next_flushable].mem_buf = &cache_pages[4096*next_flushable];
  cache_page_descrs[next_flushable].handle = handle;
  dbgprintf("fileprov: cache page %d->0x%x\n", next_flushable, fault_addr);
  mem_buf = &cache_pages[4096*next_flushable];
  dbgprintf("fileprov: page contents: %02x %02x %02x %02x\n",
          ((unsigned char*)mem_buf)[0], ((unsigned char*)mem_buf)[1],
          ((unsigned char*)mem_buf)[2], ((unsigned char*)mem_buf)[3]);
  next_flushable++;
  next_flushable %= NUM_CACHE_PAGES;
  return mem_buf;
}

void
fileProviderMain (void)
{
  sm_request_t request;
  l4_ipc_buffer_t requestbuf;
  l4_threadid_t myid;
  l4_msgdope_t msgdope;
  int i;

  myid = l4_myself();
  hprintf ("fileprov: alive, id = 0x%x\n", myid.dw);
  nameservice_init();
  nameservice_register_service("fileprov");

  (unsigned) cache_pages = ((unsigned)cache_pages_array + 4095) & (~4095);

  flick_init_request (&request, &requestbuf);

  msgdope = flick_server_wait(&request);
  if (L4_IPC_IS_ERROR(msgdope)) {
    hprintf("fileprov: flick_server failed, msgdope=%x\n",msgdope);
    enter_kdebug ("fileprov: wait failed");
  }

  while (3) {
    i = fileprov_dm_server(&request);

    switch(i) {
    case DISPATCH_ACK_SEND:
      msgdope = flick_server_reply_and_wait (&request);
      break;
    case DISPATCH_NO_REPLY:
      msgdope = flick_server_wait(&request);
      break;
    default:
      printf("fileprov: confused by return from fileprovider_server\n");
      exit(1);
    }
    /* should check msgdope here */
  }
}

extern int errno;

void
fileprov_dm_server_open(sm_request_t *_desc, filename_t filename, 
                         dsid_t *ds, sm_exc_t *_ev)
{
  int handle, ret;
  FILE *f;
  struct stat buf;
  f = fopen (filename, "rb");
  if (f == NULL) {
    printf ("fileprovider_open: couldn't open file, errno=%d\n", errno);
    /* throw exception here */
    _ev->_type = 42;
    enter_kdebug ("fileprov: couldn't open file");
    return;
  }

  /* see if file is already open */
  ret = fstat(f->_fileno, &buf);
  assert (ret == 0);
  for (handle = 0; handle < MAX_HANDLES; handle++)
    if ((filehandles[handle].file != NULL) && 
        (filehandles[handle].inode == buf.st_ino) &&
        (filehandles[handle].device == buf.st_dev) &&
        (filehandles[handle].size == buf.st_size) &&
        (filehandles[handle].mtime == buf.st_mtime)) {
      break;
    }

  if (handle == MAX_HANDLES) { /* file wasn't already open */
    for (handle = 0; handle < MAX_HANDLES; handle++)
      if (filehandles[handle].file == NULL)
        break;
    if (handle == MAX_HANDLES) {
      printf ("fileprovider_open: too many open files\n");
      enter_kdebug ("fileprov: too many open files");
      return;
    }
    filehandles[handle].file = f;
    filehandles[handle].inode = buf.st_ino;
    filehandles[handle].device = buf.st_dev;
    filehandles[handle].size = buf.st_size;
    filehandles[handle].mtime = buf.st_mtime;
  }

  *ds = 0xe1f0000 | handle;
  dbgprintf ("fileprovider_open(\"%s\"): successful, handle=0x%x\n", filename,
             handle);
}
 
void
fileprov_dm_server_close(sm_request_t *_desc, dsid_t ds,
                          sm_exc_t *_ev)
{
  int ret, handle, i;
  handle = ds & 0xffff;
  if (((unsigned)handle >= MAX_HANDLES) || (filehandles[handle].file == NULL)
      || (ds >> 16) != 0xe1f) {
    printf ("fileprovider_close(0x%x): bad handle\n", ds);
    enter_kdebug("fileprov: bad handle");
    return;
  }
  ret = fclose (filehandles[handle].file);
  if (ret != 0) {
    printf ("fileprovider_close: fclose() failed\n");
  }
  filehandles[handle].file = NULL;
  for (i = 0; i < NUM_CACHE_PAGES; i++)
    if (cache_page_descrs[i].handle == handle)
      cache_page_descrs[i].mem_buf = NULL; /* invalidate stale cache pages */
}
 
void
fileprov_dm_server_map_page(sm_request_t *request, dsid_t ds, sdword_t offset,
                            l4_snd_fpage_t *mapping, sm_exc_t *_ev)
{
  unsigned long srcpage_address;
  int handle;
  handle = ds & 0xffff;
  if (((unsigned)handle >= MAX_HANDLES) || (filehandles[handle].file == NULL)
      || (ds >> 16) != 0xe1f) {
    printf ("fileprovider_ds_fault(0x%x): bad handle\n", ds);
    enter_kdebug("fileprov: bad ds handle");
    return;
  }
  srcpage_address = (unsigned long) get_page (handle, offset);
  if (srcpage_address == 0) {
    printf ("fileprovider_ds_fault(0x%x): trouble getting cache page\n", ds);
    enter_kdebug("fileprov: problem in ds_fault");
    return;
  }
  *(volatile char *) srcpage_address;
  mapping->fpage = l4_fpage (srcpage_address, 12, 0, 0);
  mapping->snd_base = offset;
  dbgprintf ("fileprov: fulfilling ds fault for %x.%x\n",
           request->client_tid.id.task, request->client_tid.id.lthread);
}

int
main(void)
{
  fileProviderMain();
  return 0;
}

