/* This is a library for establishing a character pipe style of communication
 * with another task */

#ifndef __lib_ipcpipe_h__
#define __lib_ipcpipe_h__

#include <l4/types.h>
#include <l4/syscalls.h>
#include <services/lib/utils.h>

#define QUEUESIZE 256

typedef struct ipcpipe_s {
  l4_threadid_t remote_listener, remote_speaker, local_listener, local_speaker,
                local_notify_break;
  volatile short inqhead,  /* written only by getchar  */
                 inqtail,  /* written only by listener */
                 outqhead, /* written only by speaker  */
                 outqtail; /* written only by putchar  */
  volatile char inqueue[QUEUESIZE], outqueue[QUEUESIZE]; /* circular arrays */
  cp_resource_t inqcp, outqcp; /* for consumer/producer synchronization */
} ipcpipe_t;

extern l4_threadid_t mytask;

int
ipcpipe_open (ipcpipe_t *port,
              l4_threadid_t remote_listener, l4_threadid_t remote_speaker,
              int local_listener_lthread, int local_speaker_lthread,
              int local_notify_break_lthread,
              void *local_listener_stack, void *local_speaker_stack);

static __inline__ int
ipcpipe_putchar (ipcpipe_t *port, int ch)
{
  dbgprintf("p%x'%c'\n", mytask.id.task, ch);
  port->outqueue[port->outqtail] = (char)ch;
  port->outqtail = (port->outqtail + 1) % QUEUESIZE;
  produce(&port->outqcp);
}


static __inline__ int
ipcpipe_getchar (ipcpipe_t *port)
{
  int ch;
  consume(&port->inqcp);
  ch = port->inqueue[port->inqhead];
  port->inqhead = (port->inqhead + 1) % QUEUESIZE;
  dbgprintf("g%x'%c'\n", mytask.id.task, ch);
  return ch;
}

static __inline__ int
ipcpipe_flushoutput (ipcpipe_t *port)
{
  while (port->outqtail != port->outqhead) {
    unsigned dum, count = 0;
    l4_msgdope_t result;
    l4_i386_ipc_receive (L4_NIL_ID, L4_IPC_SHORT_MSG, &dum, &dum, &dum,
                         L4_IPC_TIMEOUT(128,12,128,12,9,9), &result);
  }
}

#endif
