#include <l4/ipc.h>

#include "config.h"
#include "kdb.h"
#include "thread.h"
#include "thread_list.h"
#include "irq.h"
#include "time.h"

#include "ipc.h"

// the rendezvous signal handler: sent by the IPC sender to the receiver
unsigned long
signal_send_regs_t::func(thread_t *t)
{
  if (t->state() == Thread_invalid)
    return L4_IPC_ENOT_EXISTENT;

  unsigned ipc_state = t->state() & (Thread_receiving|Thread_waiting
				     | Thread_send_in_progress
				     | Thread_ipc_in_progress);

  // Thread_send_in_progress must not be set
  if (! (ipc_state == (Thread_waiting | Thread_ipc_in_progress))
      && (! ((ipc_state == (Thread_receiving | Thread_ipc_in_progress)) 
	     && t->my_partner == sender())))
    return 0x80000000;		// transient error -- partner not ready

  if (my_sender_regs->eax & ~2)
    panic("long ipc");		// XXX unimplemented

  unsigned long ret;

  ret = t->my_receive_regs->eax = 0; // error code: IPC successful
  sender()->sender_dequeue(t);	// dequeue from sender queue if enqueued

  // this is a fast (registers-only) receive operation
  t->my_receive_regs->edx = my_sender_regs->edx;
  t->my_receive_regs->ebx = my_sender_regs->ebx;

  *t->my_receive_regs->thread_esi_edi() = sender()->id(); // copy sender ID

  if (my_sender_regs->eax & 2)
    {
      t->my_receive_regs->eax = 
	fpage_map(sender()->space(), 
		  (l4_fpage_t){fpage: my_sender_regs->ebx},
		  t->space(),
		  (l4_fpage_t){fpage: t->my_receive_regs->ebp},
		  my_sender_regs->edx & ~PAGE_MASK);

      if (t->my_receive_regs->eax & L4_IPC_REMAPFAILED)
	ret = L4_IPC_SEMAPFAILED;
    }

  sender()->state_del(Thread_polling | Thread_send_in_progress);

  t->state_change(~(Thread_waiting|Thread_receiving|Thread_ipc_in_progress),
		  Thread_running);

  return ret;
}

// the "receiver is ready" signal handler: sent by IPC receiver to sender
unsigned long
signal_receiver_ready_t::func(thread_t *t)
{
  if (! t->state_del(Thread_polling))
    return Sig_fail;

  t->state_add(Thread_running);

  return Sig_success;
}

bool
thread_t::ipc_receiver_ready()
{
  signal_receiver_ready_t s;

  return current()->send_signal(this, &s) == signal_t::Sig_success;
}

// the send operation
unsigned 
thread_t::do_send(thread_t *partner, l4_timeout_t t, thread_user_regs_t *regs)
{
  if (!partner || partner->state() == Thread_invalid // partner nonexistent?
      || partner == nil_thread) // special case: must not send to L4_NIL_ID
    return L4_IPC_ENOT_EXISTENT;
      
  unsigned ret;

  // register partner so that we can be dequeued by kill()
  my_send_partner = partner;	

  // XXX the cancel flag should be reset upon return to user
  state_change(~Thread_cancel, 
	       Thread_polling | Thread_ipc_in_progress 
	       | Thread_send_in_progress);
  sender_enqueue(partner);
  
  // try a rendezvous with the partner
  ret = partner->ipc_send_regs(regs);
      
  if (ret & 0x80000000)		// transient error
    {
      // in case of a transient error, try sleeping
      
      timeout_t timeout;
      
      if (t.to.snd_exp != 0) // not sleep forever?
	{
	  if (t.to.snd_man == 0) // timeout = zero?
	    {
	      state_del(Thread_polling | Thread_ipc_in_progress
			| Thread_send_in_progress);
	      sender_dequeue(partner);
	      return L4_IPC_SETIMEOUT; // give up
	    }      
	  
	  // enqueue a timeout
	  my_timeout = &timeout;
	  timeout.set(kmem::info()->clock 
		      + timeout_to_microsec(t.to.snd_man, t.to.snd_exp));
	  
	}
      
      do 
	{
	  // XXX need to find a way to detect a reset thread here
	  // (before scheduling)
	  if (state_change_safely(~(Thread_running|Thread_polling), 
				  Thread_polling))
	    schedule();
	  
	  state_add(Thread_polling);
	  
	  // detect if we have been reset meanwhile
	  if (state() & Thread_cancel)
	    {
	      // we've presumably been reset!
	      state_del(Thread_cancel);
	      sender_dequeue(partner);
	      ret = L4_IPC_SECANCELED;
	      break;
	    }
	  
	  // detect if we have timed out
	  if (timeout.has_hit())
	    {
	      sender_dequeue(partner);
	      ret = L4_IPC_SETIMEOUT;
	      break;
	    }
	  
	  // OK, it's our turn to try again to send the message
	  ret = partner->ipc_send_regs(regs);	      
	}
      while ((ret & 0x80000000)); // transient error
				// re-enter loop even if !(state&polling)
				// for error handling
      if (timeout.is_set())
	timeout.reset();

      my_timeout = 0;
    }

  assert(! in_sender_list());

  state_del(Thread_polling | Thread_send_in_progress
	    | ((state() & (Thread_receiving | Thread_waiting))
	       ? 0 : Thread_ipc_in_progress));

  return ret;      
}

// preparation for the receive operation
void 
thread_t::prepare_receive(sender_t *partner,
			  thread_user_regs_t *regs)
{
  if (partner && partner->state() == Thread_invalid) // partner nonexistent?
    return;			// just return
  // don't need to signal error here -- it will be detected later

  unsigned ipc_state;

  my_receive_regs = regs;	// message should be poked in here

  if (partner)
    {
      my_partner = partner;
      ipc_state = Thread_receiving;
    }
  else
    ipc_state = Thread_waiting;

  // if we don't have a send, get us going here; otherwise, the send
  // operation will set the Thread_ipc_in_progress flag after it has
  // initialized
  if (regs->eax == 0xffffffff)
    state_add(ipc_state | Thread_ipc_in_progress);
  else
    state_add(ipc_state);
}

// the receive operation
unsigned 
thread_t::do_receive(sender_t *partner, l4_timeout_t t, 
		     thread_user_regs_t *regs)
{
  unsigned ipc_state = partner ? Thread_receiving : Thread_waiting;

  timeout_t timeout;
  my_timeout = &timeout;

  if (t.to.rcv_exp != 0)	// not sleep forever?
    {
      // enqueue a timeout
      timeout.set(kmem::info()->clock 
		  + timeout_to_microsec(t.to.rcv_man, t.to.rcv_exp));

    }	  

  while (state() & ipc_state)
    {
      // if a sender has been queued, wake it up
      sender_t *first = sender_first; // find a sender
      if (first && (!partner || partner == first))
	first->ipc_receiver_ready(); // wake it up
      
      if (! state_change_safely(~(ipc_state | Thread_running), ipc_state))
	break;
      
      schedule();
    }

  if (timeout.is_set())
    timeout.reset();

  my_timeout = 0;
  
  if (state() & Thread_ipc_in_progress) // abnormal termination
    {
      // the IPC has not been finished.  could be timeout or cancel
      state_del(Thread_ipc_in_progress);

      if (state() & Thread_cancel)
	{
	  // we've presumably been reset!
	  state_del(Thread_cancel);
	  regs->eax = L4_IPC_RECANCELED;
	}
      else
	regs->eax = L4_IPC_RETIMEOUT;
    }

  return regs->eax;		// sender puts return code here
}

// the IPC system call
unsigned
thread_t::sys_ipc(thread_regs_t *regs)
{
  unsigned ret = 0;
  l4_timeout_t t;
  t.timeout = regs->ecx;

  // find the ipc partner thread belonging to the destination thread
  threadid_t dst_id(regs->thread_esi_edi());
  thread_t *partner = dst_id.lookup(); // XXX no clans & chiefs for now!

  // XXX the cancel flag should be reset upon return to user
  state_del(Thread_cancel);

  // first, before starting a send, we must prepare a pending receive
  // so that we can atomically switch from send mode to receive mode
  
  bool have_receive = (regs->ebp != 0xffffffff);
  bool have_sender = false;
  sender_t *sender = 0;
  irq_t *interrupt = 0;
  
  if (have_receive)
    {
      if (regs->ebp & 1)	// open wait ?
	{
	  sender = 0; 
	  have_sender = true;
	}

      // not an open wait
      else if (! dst_id.is_nil()) // not nil thread, and not irq id?
	{
	  if (! partner)
	    return L4_IPC_ENOT_EXISTENT;

	  sender = partner; 
	  have_sender = true;
	}

      // nil thread or interrupt thread
      else if (regs->esi == 0)	// nil thread?
	{
	  // only prepare IPC for timeout != 0
	  if (t.to.rcv_exp == 0 || t.to.rcv_man != 0)
	    {
	      sender = partner; 
	      have_sender = true;
	    }
	}
      // must be interrupt thread
      else 
	{
	  if (regs->esi > irq_t::irq_max)
	    return L4_IPC_ENOT_EXISTENT;

	  interrupt = irq_t::lookup(regs->esi - 1);

	  if (my_irq != interrupt)
	    {
	      // a receive with a timeout != 0 from a non-associated
	      // interrupt is illegal
	      if (t.to.rcv_exp == 0 || t.to.rcv_man != 0) // t/o != 0
		return L4_IPC_ENOT_EXISTENT;
	    }

	  if (my_irq)
	    {
	      sender = my_irq;	// we always try to receive from the
				// assoc'd irq first, not from the spec. one
	      have_sender = true;
	    }
	}

      if (have_sender)
	prepare_receive(sender, regs);
    }

  // the send part

  if (regs->eax != 0xffffffff)	// do we do a send operation?
    {
      ret = do_send(partner, t, regs);
    }

  // send done, receive operation follows

  if (have_receive		// do we do a receive operation?
      && (ret & L4_IPC_ERROR_MASK) == 0) // no send error
    {
      if (have_sender)
	{
	  // do the receive operation
	  ret = do_receive(sender, t, regs);

	  // if this was a receive from an interrupt and the timeout was 0, 
	  // this is a re-associate operation
	  if (ret != L4_IPC_RETIMEOUT
	      || ! interrupt
	      || (t.to.rcv_exp == 0 || t.to.rcv_man != 0)) // t/o != 0
	    {
	      return ret;
	    }
	}

      else if (! interrupt)	// rcv from nil; this also means t/o == 0
	{
	  //
	  // disassociate from irq
	  //
	  if (my_irq)
	    {
	      my_irq->free(this);
	      my_irq = 0;
	    }

	  return L4_IPC_RETIMEOUT;
	}

      //
      // associate with an interrupt
      //
      if (interrupt->alloc(this))
	{
	  // succeeded -- free previously allocated irq
	  if (my_irq)
	    my_irq->free(this);

	  my_irq = interrupt;

	  return L4_IPC_RETIMEOUT; // success
	}

      return L4_IPC_ENOT_EXISTENT; // failed -- could not associate new irq
    }

  // skipped receive operation
  state_del(Thread_receiving|Thread_waiting|Thread_ipc_in_progress);

  return ret;
}

// the page fault handler
bool 
thread_t::handle_page_fault_pager(vm_offset_t pfa, unsigned error_code)
{
  // do not handle user space page faults from kernel mode if we're
  // already handling a signal
  if (!(error_code & 4) && switching_in.test())
    {
      panic("page fault in signal handler");
    }

  // set up a register block used as an IPC parameter block for the
  // page fault IPC

  thread_user_regs_t r;
  
  r.edx = (pfa & ~3) | (error_code & 3); // dword 1
  r.ebx = (error_code & 4) ? regs()->eip : 0xffffffff; // dword 2
  r.eax = 0;			// snd descriptor = short msg
  r.ebp = static_cast<unsigned>
    (L4_IPC_MAPMSG(0, L4_WHOLE_ADDRESS_SPACE)); // rcv descriptor = map msg
  
  prepare_receive(my_pager, &r);

  unsigned err = do_send(my_pager, L4_IPC_NEVER, &r);

  if (err & L4_IPC_ERROR_MASK)
    {
      if (config::conservative)
	{
	  printf("page fault send error = 0x%x\n", err);
	  kdb::ke("send to pager failed");
	}

      // skipped receive operation
      state_del(Thread_receiving|Thread_waiting|Thread_ipc_in_progress);

      return false;
    }

  err = do_receive(my_pager, L4_IPC_NEVER, &r);
  
  if (err & L4_IPC_ERROR_MASK)
    {
      if (config::conservative)
	{
	  printf("page fault rcv error = 0x%x\n", err);
	  kdb::ke("rcv from pager failed");
	}
      
      return false;
    }

  return true;
}
