// utility functions for mapping flexpages from one address space into another

#include <l4/types.h>
#include <l4/ipc.h>

#include <flux/machine/types.h>
#include <flux/machine/paging.h>
#include <stdio.h>
#include <assert.h>

#include "config.h"
#include "kmem.h"
#include "space.h"
#include "mapping.h"
#include "kdb.h"

#include "ipc.h"


#define MAPDB_RAM_ONLY

static mapdb_t the_mapdb;
static mapdb_t * const mapdb = &the_mapdb;

// map the region described by "fp_from " of address space "from" into
// region "fp_to" at offset "offs" of address space "to", updating the
// mapping database as we do so
unsigned
fpage_map(space_t *from, l4_fpage_t fp_from, space_t *to, l4_fpage_t fp_to,
	  vm_offset_t offs)
{
  assert((offs & PAGE_MASK) == 0);

  unsigned condition = 0;

  // compute virtual address space regions for this operation
  vm_offset_t snd_size = fp_from.fp.size;
  snd_size = (snd_size >= 32) ? kmem::mem_user_max
    : (1L << snd_size) & ~PAGE_MASK;
  vm_offset_t snd_start = config::backward_compatibility
    ? fp_from.fpage & (snd_size - 1) // size aligment
    : fp_from.fpage & ~PAGE_MASK;
  
  vm_offset_t rcv_size = fp_to.fp.size;
  rcv_size = (rcv_size >= 32) ? kmem::mem_user_max
    : (1L << rcv_size) & ~PAGE_MASK;
  vm_offset_t rcv_start = config::backward_compatibility
    ? fp_to.fpage & (rcv_size - 1) // size aligment
    : fp_to.fpage & ~PAGE_MASK;
  
  // loop variables
  vm_offset_t rcv_addr = rcv_start + (offs & (rcv_size - 1));
  vm_offset_t snd_addr = snd_start;

  bool is_sigma0 = (from->space() == config::sigma0_taskno);

  // We now loop through all the pages we want to send from the
  // sender's address space, looking up appropriate parent mappings in
  // the mapping data base, and entering a child mapping and a page
  // table entry for the receiver.

  // Special care is taken for 4MB page table entries we find in the
  // sender's address space: If what we will create in the receiver is
  // not a 4MB-mapping, too, we have to find the correct parent
  // mapping for the new mapping database entry: This is the sigma0
  // mapping for all addresses != the 4MB page base address.

  // verify sender and receiver virtual addresses are still within
  // bounds; if not, bail out.  Sigma0 may send from any address (even
  // from an out-of-bound one)
  while (snd_size		// pages left for sending?
	 && rcv_addr < kmem::mem_user_max // addresses OK?
	 && (snd_addr < kmem::mem_user_max || is_sigma0))
    {
      // first, look up the page table entry in the sender address
      // space -- except for sigma0, which is special-cased because it
      // doesn't need to have a fully-contructed page table

      vm_offset_t phys, size;
      bool read_only;

      if ((is_sigma0 
	   || from->v_lookup(snd_addr, &phys, &size, &read_only)))
	{
	  // we have a mapping in the sender's address space
	  // locate the corresponding entry in the mapping data base

	  if (is_sigma0)	// special-cased because we don't do pt lookup
	    {			// for sigma0
	      phys = trunc_superpage(snd_addr);
	      size = SUPERPAGE_SIZE;
	      read_only = false;

	      // for IPC mapping purposes, sigma0 addresses from
	      // 0x40000000 .. 0xC0000000-1 are shifted to 0x80000000
	      // .. 0xffffffff -- ugly but true!
	      if (phys >= 0x40000000 && phys < 0xC0000000)
		phys += 0x40000000;
	    }
	  
	  // The owner of the parent mapping.  By default, this is the
	  // sender task.  However, if we create a 4K submapping from
	  // a 4M submapping, and the 4K mapping doesn't have the same
	  // physical address (i.e., the 4K subpage starts somewhere
	  // in the middle of the 4M page), then the owner of the
	  // parent mapping is sigma0.
	  space_t *parent = from;

	  // if the sender page table entry is a superpage (4mb-page),
	  // check if the receiver can take a superpage
	  if (size == SUPERPAGE_SIZE)
	    {
	      if (size > snd_size // want to send less that a superpage?
		  || !superpage_aligned(rcv_addr) // rcv page not aligned?
		  || rcv_addr + SUPERPAGE_SIZE > rcv_start + rcv_size)
				// rcv area to small?
		{
		  // we map a 4K mapping from a 4MB mapping

		  // XXX we should check here (via some flag or a
		  // "magic" submapping) if we're allowed to do so,
		  // because we currently can only give out 4K
		  // mappings from a single 4M mapping.

		  if (fp_from.fp.grant)
		    {
		      printf("KERNEL: XXX can't grant from 4M page");
		      fp_from.fp.grant = 0;
		    }

		  size = PAGE_SIZE;

		  vm_offset_t super_offset = snd_addr & SUPERPAGE_MASK;

		  if (super_offset)
		    {
		      phys += super_offset;
		      parent = sigma0;
		    }
		}

	    }

	  // find an already existing page table entry in the receiver
	  vm_offset_t r_phys, r_size;
	  bool r_ro;
	  mapping_t *pm = 0;

	  if (to->v_lookup(rcv_addr, &r_phys, &r_size, &r_ro))
	    {
	      // we have something mapped.  check if we can do an
	      // upgrade mapping, otherwise skip operation

	      // Don't check here if an r/w upgrade makes sense (i.e.,
	      // we pass on r/w permission, or the receiver has a r/o
	      // mapping).  We must go through v_insert() in any case
	      // because we need its return value to signal success
	      // (condition |= L4_IPC_FPAGE_MASK).

	      if (r_size == size)
		{
		  if (phys != r_phys) // different pg frame
		    goto skip;
		}
	      else if ((r_size == SUPERPAGE_SIZE 
			&& r_phys + (rcv_addr & SUPERPAGE_MASK) != phys)
		       || phys + (snd_addr & SUPERPAGE_MASK) != r_phys)
		goto skip;	// different pg frame

	      // OK, so we can do an r/w upgrade.  see if existing
	      // mapping is a child of our's

#ifdef MAPDB_RAM_ONLY
	      if (phys < kmem::info()->main_memory.high)// XXX mapdb limitation
		{
#endif
		  pm = mapdb->lookup(to, rcv_addr, Map_mem);

		  if (! pm)
		    goto skip;	// someone deleted this mapping in the meantime
		  
		  if (pm->parent()->space() != parent->space())
		    {
		      // not a child of ours
		      mapdb->free(pm);
		      goto skip;
		    }

		  pm = pm->parent();
#ifdef MAPDB_RAM_ONLY
		}
#endif
	    }
	  else
	    {
#ifdef MAPDB_RAM_ONLY
	      if (phys < kmem::info()->main_memory.high)// XXX mapdb limitation
		{
#endif
		  // look up the parent mapping in the mapping data base; this
		  // also locks the mapping tree for this page frame
		  pm = mapdb->lookup(parent, snd_addr, Map_mem);
		  
		  if (! pm)
		    goto skip;	// someone deleted this mapping in the meantime
#ifdef MAPDB_RAM_ONLY
		}
#endif
	    }


	  space_t::status_t status =
	    to->v_insert(phys, rcv_addr, size, 
			 read_only || ! fp_from.fp.write);

	  switch (status)
	    {
	    case space_t::Insert_warn_exists:
	    case space_t::Insert_warn_rw_upgrade:
	    case space_t::Insert_ok:
#ifdef MAPDB_RAM_ONLY
	      if (phys < kmem::info()->main_memory.high)// XXX mapdb limitation
		{
#endif
		  if (fp_from.fp.grant)
		    {
		      mapdb->grant(pm, to, Map_mem);
		      
		      from->v_delete(snd_addr, size);
		    }
		  else if (status == space_t::Insert_ok)
		    check( mapdb->insert(pm, to, rcv_addr, size, Map_mem) );
#ifdef MAPDB_RAM_ONLY
		}
#endif
	      condition |= L4_IPC_FPAGE_MASK;	     
	      break;

	    case space_t::Insert_err_nomem:
	      condition |= L4_IPC_REMAPFAILED;
	      break;

	    case space_t::Insert_err_exists:
	      
	      ; // do nothing
	    }

	  if (pm)
	    mapdb->free(pm);

	  if (condition & L4_IPC_REMAPFAILED)
	    return condition;
	}
      else			// have no pgtable entry in sender -- skip
	size = PAGE_SIZE;
      
    skip:

      rcv_addr += size;
      if (rcv_addr >= rcv_start + rcv_size) rcv_addr = rcv_start; // wrap
      
      snd_size -= size;
      snd_addr += size;
    }

  return condition;
}

// unmap the mappings in the region described by "fp" from the address
// space "space" and/or the address spaces the mappings have been
// mapped into.
bool
fpage_unmap(space_t *space, l4_fpage_t fp, bool me_too, bool only_read_only)
{
  vm_offset_t size = fp.fp.size;
  size = (size >= 32) ? kmem::mem_user_max : (1L << size) & ~PAGE_MASK;
  vm_offset_t start = config::backward_compatibility
    ? fp.fpage & (size - 1)	// size-alignment
    : fp.fpage & ~PAGE_MASK;

  vm_offset_t phys, phys_size;
  bool read_only;

  bool need_flush = false;

  // iterate over all pages in "space"'s page table that are mapped
  // into the specified region
  for (vm_offset_t address = start;
       address < start + size
	 && address < kmem::mem_user_max;
       )
    {
      // find a matching page table entry
      if (! space->v_lookup(address, &phys, &phys_size, &read_only))
	{
	  address += phys_size;
	  continue;
	}

#ifdef MAPDB_RAM_ONLY
      if (phys >= kmem::info()->main_memory.high)
	{
	  address += PAGE_SIZE;
	  continue;
	}
#endif

      // XXX we can't split 4MB mappings yet
      if (phys_size == SUPERPAGE_SIZE
	  && me_too
	  && (address + SUPERPAGE_SIZE > start + size
	      || ! superpage_aligned(address)))
	{
	  kdb::ke("can't split superpage in unmap");
	  address = round_superpage(address + 1); // cont. at next superpage
	  continue;
	}

      // find all "space" mappings belonging to this page table entry
      for (vm_offset_t phys_address = phys;
	   address < start + size
	     && phys_address < phys + phys_size;
	   address += PAGE_SIZE,
	     phys_address += PAGE_SIZE)
	{
	  mapping_t *parent;
	  
	  // find the corresponding mapping for "address"
	  if (phys_size == SUPERPAGE_SIZE
	      && ! superpage_aligned(address))
	    {
	      parent = mapdb->lookup(sigma0, address, Map_mem);
	    }
	  else
	    {
	      parent = mapdb->lookup(space, address, Map_mem);
	    }

	  assert(parent);
	  
	  // first delete from this address space
	  if (me_too)
	    {
	      space->v_delete(address, phys_size, only_read_only);
	      need_flush = true;
	    }

	  // now delete from the other address spaces
	  for (mapping_t *m = parent->next_child(parent);
	       m;
	       m = m->next_child(parent))
	    {
	      m->space().lookup()->v_delete(m->vaddr(), m->size(),
					    only_read_only);
	    }

	  mapping_t *locked = 
	    (parent->space() == config::sigma0_taskno) 
	    ? parent : parent->parent();

	  if (!only_read_only)
	    {
	      mapdb->flush(parent, 
			   me_too && !(parent->space() 
				       == config::sigma0_taskno));
	    }

	  mapdb->free(locked);

	}
    }

  if (need_flush)
    kmem::tlb_flush();

  return true;
}
