#include "atomic.h"
#include "thread.h"
#include "thread_list.h"
#include "irq.h"

static irq_t irq00(0);
static irq_t irq01(1);
static irq_t irq02(2);
static irq_t irq03(3);
static irq_t irq04(4);
static irq_t irq05(5);
static irq_t irq06(6);
static irq_t irq07(7);
static irq_t irq08(8);
static irq_t irq09(9);
static irq_t irq10(10);
static irq_t irq11(11);
static irq_t irq12(12);
static irq_t irq13(13);
static irq_t irq14(14);
static irq_t irq15(15);

irq_t * const irq_t::irqs[] = { &irq00, &irq01, &irq02, &irq03, &irq04, 
				&irq05, &irq06, &irq07, &irq08, &irq09, 
				&irq10, &irq11, &irq12, &irq13, &irq14, 
				&irq15 };

// constructor
irq_t::irq_t(unsigned irqnum)
  : _irq_thread(0), _queued(0)
{ 
  my_id = L4_NIL_ID;
  my_id.lh.low = irqnum + 1;

  my_state = Thread_dead;

  // these initializations should be part of sender_t (from which we
  // inherit), but aren't done there with respect to special needs of
  // thread_t

  sender_next = 0;
}

unsigned long
signal_send_irq_t::func(thread_t *t)
{
  if (t->state() == Thread_invalid)
    return L4_IPC_ENOT_EXISTENT;

  if (! (t->state() & Thread_waiting)
      && (! ((t->state() & Thread_receiving) && t->my_partner == my_irq)))
    return 0x80000000;		// transient error -- partner not ready

  t->state_change(~(Thread_waiting|Thread_receiving|Thread_ipc_in_progress),
		  Thread_running);
  // XXX receiver should also get a fresh timeslice

  int o;
  do {
    o = my_irq->_queued;
  } while (! compare_and_swap(&my_irq->_queued, o, o - 1));

  if (o <= 1)			// remain queued if more interrupts are left
    my_irq->sender_dequeue(t);

  *t->my_receive_regs->thread_esi_edi() = my_irq->id(); // copy sender ID
  t->my_receive_regs->eax = 0;	// state = OK

  return 0;			// OK
}

// the receiver's activation function
bool 
irq_t::ipc_receiver_ready()
{
  assert(current() == _irq_thread);

  if (_queued)
    _irq_thread->ipc_send_irq(this);
  
  return true;
}

inline void
irq_t::hit()
{
  // we're entered with disabled irqs

  if (! _irq_thread)
    {
      pic_disable_irq(my_id.lh.low - 1);
    }

  else if (_queued++ == 0)	// increase hit counter
    {
      sender_enqueue(_irq_thread); // XXX should enqueue at position 1
      
      // if the thread is waiting for this interrupt, make it running;
      // this will cause it to run irq->receiver_ready(), which
      // handles the rest

      unsigned ipc_state = 
	_irq_thread->state() & (Thread_receiving|Thread_waiting
				| Thread_send_in_progress
				| Thread_ipc_in_progress);
      
      // Thread_send_in_progress must not be set
      if (ipc_state == (Thread_waiting | Thread_ipc_in_progress)
	  || (ipc_state == (Thread_receiving | Thread_ipc_in_progress)
	      && _irq_thread->my_partner == this))
	{
	  // we don't need to manipulate the state in a safe way
	  // because we are still running with interrupts turned off
	  _irq_thread->state_change(~ipc_state, 
				    ipc_state | Thread_running);
	  if (current() != _irq_thread
	      && _irq_thread->my_prio >= current()->my_prio)
	    {
	      // sti() is done in switch_to()  
	      
	      // switch to the interrupt thread	 
	      current()->switch_to(_irq_thread);
	  
	      return;
	    }
	  else
	    _irq_thread->ready_enqueue();
	}
    }

  sti();			// enable interrupts
}

// hardware interrupt entry point
extern "C" void
irq_interrupt(unsigned irqnum)
{
  // we're entered with disabled irqs
#if 0
  irq_ack(irqnum);		// XXX user thread should do this
#endif

#if 0				// screen spinner for debugging purposes
  (*(unsigned char*)(kmem::phys_to_virt(0xb8000 + irqnum*2)))++;
#endif

  irq_t::lookup(irqnum)->hit();
}
