// -*- c++ -*-
#ifndef IRQ_H
#define IRQ_H

#include <flux/machine/pc/pic.h>

#include "atomic.h"
#include "thread.h"

class signal_send_irq_t;

class irq_t : public sender_t
{
public:
  static const irq_max = 16;

  explicit irq_t(unsigned irqnum); // constructor

  // activation functions
  virtual bool ipc_receiver_ready();

  void hit();

  // other stuff
  bool alloc(thread_t *t);
  bool free(thread_t *t);

  // irq registry
  static irq_t *lookup(unsigned irqnum);

  // initialization
  static void init(unsigned char master_base, unsigned char slave_base);

private:
  irq_t();			// default constructors remain undefined
  irq_t(irq_t&);

  // data layout
  thread_t *_irq_thread;
  int _queued;

  // global irq registry
  static irq_t * const irqs[irq_max];

  friend class signal_send_irq_t;
};

// inlines

inline irq_t *
irq_t::lookup(unsigned irqnum)
{
  assert (irqnum < irq_max);
  return irqs[irqnum];
}

inline bool
irq_t::alloc(thread_t *t)
{
  bool ret = compare_and_swap(&_irq_thread, static_cast<thread_t*>(0), t);

  if (ret) 
    {
      pic_disable_irq(my_id.lh.low - 1);
      _queued = 0;
    }

  return ret;
}

inline bool
irq_t::free(thread_t *t)
{
  bool ret = compare_and_swap(&_irq_thread, t, static_cast<thread_t*>(0));

  if (ret) 
    {
      pic_disable_irq(my_id.lh.low - 1);
      sender_dequeue(t);
    }

  return ret;
}

//
// helpers 
//

// acknowledge an interrupt to the PIC; do this in the preferred way for 
// the special fully nested mode we've programmed
inline void
irq_ack(char irqnum)
{
  if (irqnum >= 8)
    {
      outb_p(SLAVES_ICW, NON_SPEC_EOI); // EOI slave
      outb_p(SLAVES_ICW, OCW_TEMPLATE | READ_NEXT_RD | READ_IS_ONRD);
      if (inb_p(SLAVES_ICW))	// slave still active?
	return;			// -- don't EOI master
    }
  outb_p(MASTER_ICW, NON_SPEC_EOI); // EOI master
}

#endif
