#include <string.h>
#include <assert.h>

#include "globals.h"
#include "kdb.h"
#include "kmem.h"

//
// class kmem
//

// static class variables

vm_offset_t kmem::mem_max, kmem::himem;
vm_offset_t kmem::zero_page;
pd_entry_t *kmem::kdir;
pd_entry_t  kmem::cpu_global;
l4_kernel_info_t *kmem::kinfo;

multiboot_info kmem::kmbi;
char kmem::kcmdline[256];

x86_gate *kmem::idt;
x86_tss *kmem::tss;
x86_desc *kmem::gdt;

lmm_t kmem::lmm;
lmm_region_t kmem::lmm_region_all;

vm_offset_t 
kmem::virt_to_phys(const void *addr)
{
  vm_offset_t a = reinterpret_cast<vm_offset_t>(addr);

  if ((a & 0xff000000) == mem_phys) return a - mem_phys;

  pd_entry_t p = kdir[(a >> PDESHIFT) & PDEMASK];
  if (! (p & INTEL_PDE_VALID)) return 0xffffffff;
  if (p & INTEL_PDE_SUPERPAGE) return (p & ~SUPERPAGE_MASK) 
				 | (a & SUPERPAGE_MASK);
  pt_entry_t t = reinterpret_cast<pt_entry_t *>
    ((p & ~PAGE_MASK) + mem_phys)[(a >> PTESHIFT) & PTEMASK];

  return (t & INTEL_PTE_VALID) ? (t & ~PAGE_MASK) | (a & PAGE_MASK)
    : 0xffffffff;
}

void *
kmem::page_alloc(vm_offset_t va, zero_fill_t do_zero_fill = no_zero_fill)
{
  assert(va >= mem_user_max);

  void *page;
  
  if (do_zero_fill != zero_map)
    {
      page = page_alloc();
      if (!page) 
	{
	  kdb::ke("page_alloc: can't alloc new page");
	  return 0;
	}
    }

  // insert page into master page table
  pd_entry_t *p = kdir + ((va >> PDESHIFT) & PDEMASK);
  pt_entry_t *e;

  if (! (*p & INTEL_PDE_VALID))
    {
      pt_entry_t *new_pt = static_cast<pt_entry_t *>(page_alloc());
      if (! new_pt)
	{
	  kdb::ke("page_alloc: can't alloc page table");
	  goto error;
	}
      memset(new_pt, 0, PAGE_SIZE);
      *p = virt_to_phys(new_pt)
	| INTEL_PDE_VALID | INTEL_PDE_WRITE | INTEL_PDE_REF
	| cpu_global;
      e = new_pt + ((va >> PTESHIFT) & PTEMASK);
    }
  else if ((*p & INTEL_PDE_SUPERPAGE) 
	   || (e = static_cast<pt_entry_t *>(phys_to_virt(*p & ~PAGE_MASK))
	           + ((va >> PTESHIFT) & PTEMASK),
	       *e & INTEL_PTE_VALID))
    {
      kdb::ke("page_alloc: va already mapped");
      goto error;		// there's already mapped something there...
    }
	   
  if (do_zero_fill == zero_fill)
    memset(page, 0, PAGE_SIZE);

  *e = (do_zero_fill == zero_map 
	? zero_page
	: virt_to_phys(page) | INTEL_PTE_WRITE | INTEL_PTE_MOD)
    | INTEL_PTE_VALID | INTEL_PTE_REF | cpu_global;

  return reinterpret_cast<void *>(va);

error:
  if (do_zero_fill != zero_map) 
    page_free_phys(page);
  return 0;
}


// initialize a new task's page directory with the current master copy
// of kernel page tables
void kmem::dir_init(pd_entry_t *d)
{
  memcpy(d + ((mem_user_max >> PDESHIFT) & PDEMASK),
	 kdir + ((mem_user_max >> PDESHIFT) & PDEMASK),
	 ((~0 - mem_user_max + 1) >> PDESHIFT) * sizeof(pd_entry_t));
}
