// -*- c++ -*-
#ifndef KMEM_H
#define KMEM_H

#define GDT_TSS (0x08)
#define GDT_CODE_KERNEL (0x10)
#define GDT_DATA_KERNEL (0x18)
#define GDT_CODE_USER (0x20)
#define GDT_DATA_USER (0x28)
#define GDT_MAX (0x30)

#ifndef ASSEMBLER

#include <stddef.h>
#include <flux/lmm.h>
#include <flux/machine/multiboot.h>
#include <flux/machine/types.h>
#include <flux/machine/tss.h>
#include <flux/machine/seg.h>
#include <flux/machine/paging.h>
#include <flux/machine/proc_reg.h>

#include <l4/kernel.h>

/* our own implementation of C++ memory management: disallow dynamic
   allocation (except where class-specific new/delete functions exist) */

void *operator new(size_t);
void operator delete(void *block);

// more specialized memory allocation/deallocation functions follow
// below in the "kmem" namespace

// kernel.ld definitions

extern "C" {
  extern char _start, _edata, _end;
  extern char _physmem_1;
  extern char _tcbs_1;
  extern char _iobitmap_1;
  extern char _unused1_1, _unused2_1, _unused3_1;
}

class kdb;

// the kernel memory is a singleton object.  we access it through a
// static class interface.  (we would have preferred to access it like
// a class, but our method saves the overhead of always passing the
// "this" pointer on the (very small) kernel stack).

// this object is the system's page-level allocator
class kmem
{
public:
  enum zero_fill_t { no_zero_fill = 0, zero_fill, zero_map };

  static void init(vm_offset_t mbi_pa); // initialize the kmem singleton

  // page-level allocator based on lowest-level allocator
  static void *page_alloc();	// allocate a page at some kernel virtual addr
  static void *page_alloc(vm_offset_t va, // allocate a page at specified kva
			  zero_fill_t do_zero_fill = no_zero_fill);
  static void page_free(void *page); // free page at specified kva
  static void page_free_phys(void *page); // free mem_phys page at specfd kva
  
  // simple allocator based on lowest-level allocator
  static void *alloc(vm_size_t size);

  // kmem virtual-space related functions (similar to space_t)
  static vm_offset_t virt_to_phys(const void *addr); // kernel-virt to physical
  static void *phys_to_virt(vm_offset_t addr); // physical to kernel-virtual
  
  static void tlb_flush(vm_offset_t addr); // flush tlb at virtual addr
  static void tlb_flush();

  // constants from the above kernel.ld declarations
  const vm_offset_t mem_phys = reinterpret_cast<vm_offset_t>(&_physmem_1);
  const vm_offset_t mem_tcbs = reinterpret_cast<vm_offset_t>(&_tcbs_1);
  const vm_offset_t mem_user_max = 0xc0000000;
  const char *const io_bitmap = &_iobitmap_1;

  enum { gdt_tss = GDT_TSS, 
    gdt_code_kernel = GDT_CODE_KERNEL, gdt_data_kernel = GDT_DATA_KERNEL,
    gdt_code_user = GDT_CODE_USER, gdt_data_user = GDT_DATA_USER, 
    gdt_max = GDT_MAX };

  static l4_kernel_info_t *info(); // returns the kernel info page
  static const pd_entry_t *dir(); // returns the kernel's page directory
  static pd_entry_t pde_global(); // returns mask for global pg dir entries
  static const multiboot_info *mbi();
  static const char *cmdline();

  static vm_offset_t *kernel_esp();

  static void dir_init(pd_entry_t *d); // initializes a new page directory

private:
  kmem();			// default constructors are undefined
  kmem(const kmem&);

  friend class kdb;
  friend class profile;

  static vm_offset_t mem_max, himem;
  static vm_offset_t zero_page;
  static pd_entry_t *kdir;	// kernel page directory

  const pd_entry_t flag_global = 0x200;	// l4-specific pg dir entry flag
  static pd_entry_t cpu_global;

  static x86_gate *idt;
  static x86_tss *tss;
  static x86_desc *gdt;

  static l4_kernel_info_t *kinfo;

  static lmm_t lmm;
  static lmm_region_t lmm_region_all;

  static multiboot_info kmbi;
  static char kcmdline[256];
};

//
// inline implementations follow
//
inline void *kmem::phys_to_virt(vm_offset_t addr)
{
  return reinterpret_cast<void *>(addr + mem_phys);
}

inline void kmem::tlb_flush()
{
  inval_tlb();
}

inline void kmem::tlb_flush(vm_offset_t addr)
{
  asm volatile
    ("invlpg %0" : : "m" (addr) : "memory");
}


inline void *kmem::page_alloc()
{
  unsigned flags = get_eflags();
  cli();
  void *ret = lmm_alloc_page(&lmm, 0);
  set_eflags(flags);
  return ret;
}

inline void *kmem::alloc(vm_size_t size)
{
  unsigned flags = get_eflags();
  cli();
  void *ret = lmm_alloc(&lmm, size, 0);
  set_eflags(flags);
  return ret;
}

inline void kmem::page_free_phys(void *page)
{
  unsigned flags = get_eflags();
  cli();
  lmm_free_page(&lmm, page);
  set_eflags(flags);
}

inline void kmem::page_free(void *page)
{
  vm_offset_t phys = virt_to_phys(page);
  if (phys == 0xffffffff) return;

  if (phys != zero_page)
    // convert back to virt (do not use "page") to get canonic mapping
    page_free_phys(phys_to_virt(phys)); 

  vm_offset_t va = reinterpret_cast<vm_offset_t>(page);
  if (va < mem_phys)
    {
      // clean out pgdir entry
      (static_cast<pt_entry_t *>
       (phys_to_virt(kdir[(va >> PDESHIFT) & PDEMASK] & ~PAGE_MASK)))
	[(va >> PTESHIFT) & PTEMASK] = 0;

      tlb_flush(va);
    }
}

inline const pd_entry_t *kmem::dir()
{
  return kdir;
}

inline l4_kernel_info_t *kmem::info()
{
  return kinfo;
}

inline const multiboot_info *kmem::mbi()
{
  return &kmbi;
}

inline const char *kmem::cmdline()
{
  return kcmdline;
}

inline pd_entry_t kmem::pde_global()
{
  return cpu_global;
}

inline vm_offset_t *kmem::kernel_esp()
{
  return reinterpret_cast<vm_offset_t*>(& tss->esp0);
}

#endif /* ! ASSEMBLER */

#endif
