#include <stdio.h>
#include <string.h>

#include <flux/machine/proc_reg.h>
#include <flux/machine/pc/pic.h>
#include <flux/machine/pc/rtc.h>

#include "config.h"
#include "profile.h"
#include "space.h"
#include "thread.h"
#include "kdb.h"
#include "irq.h"

#include "globals.h"

class kernel_thread_t : public thread_t
{
public:
  void bootstrap();
  unsigned long *init_stack() { return kernel_sp; }

  // overload allocator -- we cannot allcate by page fault during
  // bootstrapping
  void *operator new(size_t, threadid_t id); 
};

int main()
{
  // caution: no stack variables in this function because we're going
  // to change the stack pointer!

  // make some basic initializations, then create and run the kernel
  // thread
  printf("%s\n", static_cast<char*>(kmem::info()) 
	         + (kmem::info()->offset_version_strings << 4));

  // disallow all interrupts before we selectively enable them 
  pic_disable_all();

  // initialize the kernel debugger first
  if (! strstr(kmem::cmdline(), " -nokdb"))
    {
      kdb::init();
      if (! strstr(kmem::cmdline(), "nowait"))
	kdb::ke("init");
    }

  // all initializations done, go start the system
  sti();

  // create kernel thread
  static kernel_thread_t *kernel = new (&config::kernel_id) kernel_thread_t;
  nil_thread = kernel_thread = kernel; // fill public variable

  // switch to stack of kernel thread and bootstrap the kernel
  asm volatile
    (" movl %%esp,%%ecx \n"	// save stack pointer in safe register
     " movl %0,%%esp \n"	// switch stack
     " pushl %1 \n"		// push "this" pointer
     " call  %2 \n"		// call kernel->bootstrap();
     " movl %%ecx,%%esp"	// restore old stack pointer
     : : "q" (kernel->init_stack()), "q" (kernel), "m" (kernel->bootstrap)
     : "ecx");


  printf("main() finished\n");

  return 0;
}

void *kernel_thread_t::operator new(size_t s, threadid_t id)
{
  // call superclass' allocator
  void *addr = thread_t::operator new(s, id);

  // explicitly allocate and enter in page table -- we cannot allocate
  // by page fault during bootstrapping
  if (! kmem::page_alloc(static_cast<vm_offset_t>(addr) & ~PAGE_MASK,
			 kmem::zero_fill))
    panic("can't allocate kernel tcb");

  return addr;
}

extern "C" int thread_handle_trap(trap_state *state); // in thread.cc

// the kernel bootstrap routine
void kernel_thread_t::bootstrap()
{
  //
  // set up my own thread control block
  //
  my_state = Thread_running;
  my_id = config::kernel_id;
  my_prio = config::kernel_prio;
  my_mcp = config::kernel_mcp;
  my_timeslice = config::default_time_slice;

  present_next = present_prev = this;
  ready_next = ready_prev = this;

  // XXX ugly, but OK for now -- perhaps spaces and kmem should be
  // inherited from a common super class
  my_space = static_cast<space_t*>(const_cast<pd_entry_t *>(kmem::dir()));

  //
  // set up class variables
  //
  for (int i = 0; i < 256; i++) prio_next[i] = prio_first[i] = 0;
  prio_next[config::kernel_prio] = prio_first[config::kernel_prio] = this;

  // 
  // install our slow trap handler
  //
  nested_trap_handler = base_trap_handler;
  base_trap_handler = thread_handle_trap;

  //
  // initialize interrupts
  //
  irq_t::lookup(2)->alloc(this); // reserve cascade irq
  irq_t::lookup(8)->alloc(this); // reserve timer irq

  pic_enable_irq(2);		// allow cascaded irqs

  // set up serial console
  if (! strstr(kmem::cmdline(), " -I-") 
      && !strstr(kmem::cmdline(), " -irqcom"))
    {
      int com_port = config::serial_com_port;
      int com_irq = com_port & 1 ? 4 : 3;

      irq_t::lookup(com_irq)->alloc(this); // the remote-gdb interrupt
      pic_enable_irq(com_irq);
    }

  // initialize the profiling timer
  bool user_irq0 = strstr(kmem::cmdline(), "irq0");

  if (config::profiling)
    {
      if (user_irq0)
	{
	  kdb::ke("options `-profile' and `-irq0' don't mix "
		  "-- disabling `-irq0'");
	}
      irq_t::lookup(0)->alloc(this);

      profile::init();
    }
  else if (! user_irq0)
    irq_t::lookup(0)->alloc(this); // reserve irq0 even though we don't use it

  // 
  // set up timer interrupt (~ 1ms)
  //
  while (rtcin(RTC_STATUSA) & RTCSA_TUP) ; // wait till RTC ready
  rtcout(RTC_STATUSA, RTCSA_DIVIDER | RTCSA_1024); // 1024 Hz
  // set up 1024 Hz interrupt
  rtcout(RTC_STATUSB, rtcin(RTC_STATUSB) | RTCSB_PINTR | RTCSB_SQWE); 
  rtcin(RTC_INTR);		// reset

  pic_enable_irq(8);		// allow this interrupt
  
  //
  // allow the boot task to create more tasks
  //
  for (unsigned i = config::boot_taskno + 1; 
       i < space_index_t::max_space_number;
       i++)
    {
      check(space_index_t(i).set_chief(space_index(), 
				       space_index_t(config::boot_taskno)));
    }

  //
  // create sigma0
  //
  sigma0 = new space_t(config::sigma0_id.id.task);
  sigma0_thread = new (&config::sigma0_id) thread_t;

  sigma0_thread->reset(sigma0, &config::sigma0_id, config::sigma0_prio, 
		       config::sigma0_mcp);
  
  // push address of kernel info page to sigma0's stack
  vm_offset_t *esp = static_cast<vm_offset_t *>(kmem::info()->sigma0_esp);
  *--esp = kmem::virt_to_phys(kmem::info());

  sigma0_thread->initialize(kmem::info()->sigma0_eip,
			    static_cast<vm_offset_t>(esp),
			    0, 0);

  //
  // create the boot task
  //
  space_t *boot = new space_t(config::boot_id.id.task);
  thread_t *boot_thread = new (&config::boot_id) thread_t;

  boot_thread->reset(boot, &config::boot_id, config::boot_prio, 
		     config::boot_mcp);
  boot_thread->initialize(kmem::info()->root_eip,
			  kmem::info()->root_esp,
			  sigma0_thread, 0);

  //
  // the idle loop
  //
  for (;;) 
    {
      // printf("I");

      sti();			// enable irqs, otherwise idling is fatal

      if (config::hlt_works_ok)
	asm("hlt");		// stop the CPU, waiting for an int

      while (ready_next != this) // are there any other threads ready?
	schedule();
    }
}

