#include <stdio.h>
#include <assert.h>

#include "list.h"
#include "config.h"

#include "mapping.h"

//
// mapping_t
//

// allocation functions: we use a free list which allocates new kmem
// pages on demand

static list_stack_t<mapping_t> free_list;

void *
mapping_t::operator new(size_t)
{
  mapping_t *m = free_list.dequeue();

  if (! m) 
    {
      // free list is empty -- alloc new page

      m = static_cast<mapping_t*>(kmem::page_alloc());

      if (!m)
	panic("out of pages for mapdb");
      
      for (int i = PAGE_SIZE / sizeof(mapping_t) - 1; i > 0; i--)
	free_list.insert(m + i);
    }

  return static_cast<void*>(m);
}

void mapping_t::operator delete(void *block)
{
  free_list.insert(static_cast<mapping_t*>(block));
}



inline void
mapping_t::set_superpage(bool is_superpage)
{
  _flags.superpage = is_superpage;
}

inline void
mapping_t::set_child(mapping_t *m)
{
  _flags.child = 0;
  _child_ptr |= reinterpret_cast<unsigned>(m);
}

inline void
mapping_t::set_next(mapping_t *m)
{
  _next = m;
}

inline void
mapping_t::set_prev(mapping_t *m)
{
  _prev = m;
}

inline void
mapping_t::set_last_child(bool is_last)
{
  _flags.last_child = is_last;
}

inline void
mapping_t::set_space(unsigned n)
{
  _data.space_number = n;
}

inline void
mapping_t::set_vaddr(vm_offset_t va)
{
  _data.frame_number = va >> PAGE_SHIFT;
}

inline bool
mapping_t::is_last_child()
{
  return _flags.last_child;
}

inline bool
mapping_t::is_superpage()
{
  return _flags.superpage;
}

inline mapping_t *
mapping_t::next()
{
  return _next;
}

inline mapping_t *
mapping_t::prev()
{
  return _prev;
}


// 
// mapdb_t
//

extern "C" {
  extern char _mappings_1;
}

// XXX mapdb_t is a singleton.  the initialization assumes but doesn't
// enforce this
mapdb_t::mapdb_t()
{
  _sigma_mappings = static_cast<mapping_t**>(&_mappings_1);

  // we need as many pointers to sigma0 mappings as we have page frames

  for (unsigned pages = (kmem::info()->main_memory.high / PAGE_SIZE
			 * sizeof(mapping_t*)
			 / PAGE_SIZE) + 1;
       pages--;
       )
    {
      check( kmem::page_alloc(reinterpret_cast<vm_offset_t>(_sigma_mappings)
			      + pages * PAGE_SIZE) );
    }

  // define the sigma0 mappings

  for (vm_offset_t address = 0;
       address < kmem::info()->main_memory.high;
       address += PAGE_SIZE)
    {
      mapping_t *m = new mapping_t;
      _sigma_mappings[address >> 12] = m;

      m->set_vaddr(address);
      m->set_space(config::sigma0_taskno);
      m->set_next(0);
      m->set_prev(0);
      m->set_child(0);
      m->set_last_child(false);
      m->set_superpage(false);
    }
}

mapping_t *
mapdb_t::insert(mapping_t *parent,
		space_t *space, 
		vm_offset_t va, 
		vm_size_t size,
		mapping_type_t /* ignored */)
{
  mapping_t *m = new mapping_t;

  if (!m)
    return 0;

  m->set_vaddr(va);
  m->set_space(space->space());
  m->set_child(0);
  m->set_superpage(size == SUPERPAGE_SIZE);

  m->set_prev(parent);

  unsigned flags = get_eflags();
  cli();			// XXX oops!  ugly lock!

  mapping_t *c = parent->first_child();

  if (c)
    {
      c->set_prev(m);
      m->set_next(c);
      m->set_last_child(false);
    }
  else
    {
      m->set_next(parent);
      m->set_last_child(true);
    }

  parent->set_child(m);

  set_eflags(flags);

  return m;
}

void
mapdb_t::grant(mapping_t *m, space_t *new_space, vm_offset_t va)
{
  unsigned flags = get_eflags();
  cli();			// XXX oops!  ugly lock!

  m->set_vaddr(va);
  m->set_space(new_space->space());

  set_eflags(flags);
}

mapping_t *
mapdb_t::lookup(space_t *space,      
		vm_offset_t va, 
		mapping_type_t /* ignore */ )
{
  unsigned space_number = space->space();

  vm_offset_t phys = space->virt_to_phys(va);

  if (phys == 0xffffffff)
    return 0;

  assert(phys < kmem::info()->main_memory.high);

  for (mapping_t *m = _sigma_mappings[phys >> PAGE_SHIFT];
       m;
       m = m->next_iter())
    {
      if (m->space() == space_number 
	  && m->vaddr() == va)
	return m;
    }
       
  return 0;
}

void 
mapdb_t::free(mapping_t* /* m */)
{
  // we don't do locking yet.
}

// flush a single mapping subtree
bool 
mapdb_t::flush(mapping_t *m, bool me_too)
{

  mapping_t *n, *n1, *p;

  // XXX this is quite a stupid implementation of depth-first

  while (m->first_child())
    {
      n = m;

      do 
	{
	  // find deepest child
	  for (p = n, n = n->first_child();
	       n->first_child();
	       p = n, n = n->first_child()) ;
	  
	  // n is deepest child, p is parent of n
	  
	  // free all brothers as long as they don't have children themselves
	  while (! n->is_last_child()
		 && ! n->next()->first_child())
	    {
	      n1 = n->next();
	      n1->set_prev(p);
	      delete n;
	      n = n1;
	    }

	  // n either is the last child of p, or n's next brother has children
	  
	  if (! n->is_last_child()) // n is not last child, next has children
	    {
	      n1 = n->next();	// resume descent at next brother
	      
	      assert(n1->first_child()); // must have children

	      p->set_child(n1);
	      n1->set_prev(p);	// next becomes first child of p
	    }
	  else 
	    {
	      n1 = 0;
	      p->set_child(0);
	    }
	  
	  delete n;
	  
	  n = n1;
	}
      while (n && n->first_child()); // resume if we have a brother w/children
    }      

  if (me_too)			// should m be dequeued?
    {
      p = m->prev();

      if (p->next() != m)	// p is the parent
	{
	  if (m->is_last_child())
	    p->set_child(0);
	  else
	    {
	      m->next()->set_prev(p);
	      p->set_child(m->next());
	    }
	}
      else			// p is predecessor
	{
	  p->set_next(m->next());
	  if (! m->is_last_child())
	    m->next()->set_prev(p);
	  else
	    p->set_last_child(true);
	}

      delete m;
    }

  return true;
}
