// -*- c++ -*-
#ifndef MAPPING_H
#define MAPPING_H

#include <flux/x86/types.h>

#include "list.h"
#include "kmem.h"

#include "space.h"

enum mapping_type_t { Map_mem = 0, Map_io };

class mapping_t
{
public:
  space_index_t space() const;	// the space this mapping belongs to
  vm_offset_t vaddr();		// the virtual address in the space for this
				// mapping
  vm_size_t   size();		// size of the mapping
  mapping_type_t type();	// type of mapping
  mapping_t *parent();		// get parent mapping
  mapping_t *first_child();	// get first child mapping
  mapping_t *next_iter();	// get child, next sibling, or parent's next
				// sibling
  mapping_t *next_child(mapping_t *parent); // get next anchestor of parent

private:
  friend class mapdb_t;

  mapping_t() {};		// default constructors are private:
  mapping_t(const mapping_t&);	// mapping must be created in the mapdb

  void *operator new(size_t);	// special allocators
  void operator delete(void *block);
  
  void set_superpage(bool is_superpage);
  void set_last_child(bool is_last);
  void set_child(mapping_t *m);
  void set_next(mapping_t *m);
  void set_prev(mapping_t *m);
  void set_space(unsigned n);
  void set_vaddr(vm_offset_t va);

  bool is_last_child();
  bool is_superpage();
  mapping_t *next();		// get next sibling or parent
  mapping_t *prev();		// get prev sibling or parent

  // implementation details follow
  
  union {
    sllist_elem_t<mapping_t> list_property;

    unsigned _vaddr;
    struct {
      unsigned space_number : 12;
      unsigned frame_number : 20;
    } _data;
  };      
  union {
    unsigned _child_ptr;
    struct {
      bool superpage : 1;
      bool last_child : 1;
      unsigned child : 30;
    } _flags;
  };    
  mapping_t *_next, *_prev;
};

// ******************************************
// implementaion of mapping_t inline methodes
// ******************************************

inline space_index_t mapping_t::space() const
{
  return space_index_t(_data.space_number);
};

inline vm_offset_t mapping_t::vaddr()
{
  return _vaddr & ~PAGE_MASK;
};

inline vm_size_t mapping_t::size()
{
  if (_flags.superpage) return SUPERPAGE_SIZE;
  else return PAGE_SIZE;
};

inline mapping_type_t mapping_t::type()
{
  return Map_mem;
};


inline mapping_t *mapping_t::parent()
{
  mapping_t *m = this;

  while (m->_next && ! m->_flags.last_child)
    m = m->_next;

  return m->_next;
}

inline mapping_t *mapping_t::first_child()
{
  return reinterpret_cast<mapping_t*>(_child_ptr & ~3);
}

inline mapping_t *mapping_t::next_iter()
{
  mapping_t *m;

  if ((m = first_child()))
    return m;

  m = this;
  while (m->_next && m->_flags.last_child)
    m = m->_next;		// m becomes parent of m

  return m->_next;
}

inline mapping_t *mapping_t::next_child(mapping_t *parent)
{
  mapping_t *m;

  if ((m = reinterpret_cast<mapping_t*>(_child_ptr & ~3)))
    return m;

  m = this;
  while (m != parent && m->_next && m->_flags.last_child)
    m = m->_next;		// m becomes parent of m

  if (m == parent)
    return 0;

  return m->_next; 
};

/*
 * mapping db
 *****************/

class mapdb_t
{
public:
  mapdb_t();
#if 0
  ~mapdb_t();
#endif
  mapping_t *insert(mapping_t *parent,
		    space_t *space, 
		    vm_offset_t va, 
		    vm_size_t size,
		    mapping_type_t type);
#if 0
  mapping_t *sigma_insert(unsigned32_t page_id,
			  mapping_type_t type);
#endif
  void grant(mapping_t *m, space_t *new_space, vm_offset_t va);
  mapping_t *lookup(space_t *space,      
		    vm_offset_t va, 
		    mapping_type_t type);
  void free(mapping_t* m);      // free the lock to the tree of this mapping
  bool split(mapping_t *m);	// split a large (4MB) mapping into many
				// small ones
  bool flush(mapping_t *m, bool me_too); // flush a single mapping subtree
  bool flush(space_t *space);	// flush all mappings of a given task

private:
  mapping_t **_sigma_mappings;
};



#endif
