#include <stdlib.h>
#include <flux/machine/pc/pit.h>
#include <flux/machine/pc/pic.h>

#include "config.h"
#include "kmem.h"
#include "irq.h"
#include "lib.h"

#include "profile.h"

static const PROFILE_IRQ = 0;

extern "C" void profile_interrupt_entry(); // in entry.S

void
profile::init()
{
  monstartup(&_start, &_end);
  atexit(_mcleanup);

  fill_gate(kmem::idt + 0x20 + PROFILE_IRQ,
	    (unsigned)profile_interrupt_entry,
	    kmem::gdt_code_kernel, ACC_INTR_GATE | ACC_PL_K, 0);
       
  init_pit(config::profiling_rate);
  pic_enable_irq(PROFILE_IRQ);
}

// irq routine invoked by profile_interrupt_entry in entry.S

/*
 * Scale is a fixed-point number with the binary point 16 bits
 * into the value, and is <= 1.0.  pc is at most 32 bits, so the
 * intermediate result is at most 48 bits.
 */
#define PC_TO_INDEX(pc, off, scale)				\
        ((vm_offset_t)(((unsigned long long)((pc) - (off)) *	\
		(unsigned long long)((scale))) >> 16) & ~1)

unsigned long profile::ticks = 0;
bool profile::exit = false;

inline void
profile::handle_profile_interrupt(vm_offset_t pc)
{
  // runs with disabled irqs

  pic_disable_irq(PROFILE_IRQ);
  irq_ack(PROFILE_IRQ);
  pic_enable_irq(PROFILE_IRQ);

  ticks++;

  vm_size_t i;

  if (! pr_scale)
    return;

  if (pc < pr_off 
      || (i = PC_TO_INDEX(pc, pr_off, pr_scale)) >= pr_size)
    return;			// out of range - ignore

  *reinterpret_cast<unsigned short*>(pr_base + i) += 1;

  if (exit)
    ::exit(0);
}

extern "C" void
profile_interrupt(vm_offset_t pc)
{
  profile::handle_profile_interrupt(pc);
}
