// -*- c++ -*-

#ifndef SIGNAL_H
#define SIGNAL_H

#include <assert.h>
#include "list.h"
#include "globals.h"

struct trap_state;
class sender_t;
class thread_t;
class thread_user_regs_t;
class space_t;
class irq_t;

// signal_t is intended as an abstract base class for actual signal classes
class signal_t
{
public:
  enum
  {
    Sig_aborted = -1,
    Sig_success,
    Sig_fail
  };

  // status requests
  bool done();			// marks signal as handled
  unsigned long ret();		// return value of sig func;
				// valid if done() == 1
  thread_t *sender();

  // manipulations
  void set_ret(unsigned long ret); // also sets done(); called by scheduler
				// when signal is done

  // signal function -- to be overloaded
  virtual unsigned long func(thread_t *target) = 0;

  // constructor
  signal_t();

private:
  signal_t(signal_t&);		// default constructors are undefined

  // implementation details follow...

  // data layout 
  union {
    struct {
      unsigned done : 1;	// thread ptr's lowest bit is always 0,
      unsigned other : 31;	// that's why we can use it for the "done" bit
    } my_state;
    unsigned long my_sender;
  };

  union {
    unsigned long my_ret;	// return value of sig function...
    sllist_elem_t<signal_t> list_property; // ...or its list property (next pointer)
  };

  friend class list_stack_t;	// so that we can be enqueued
};

//
// inline stuff for signal_t
//

inline signal_t::signal_t()
{
  my_sender = static_cast<unsigned long>(current());
}

inline bool signal_t::done()
{
  return my_state.done;
}

inline unsigned long signal_t::ret()
{
  assert(done());
  return my_ret;
}

inline thread_t *signal_t::sender()
{
  return static_cast<thread_t*>(my_sender & ~1);
}

inline void signal_t::set_ret(unsigned long ret)
{
  my_ret = ret;
  my_state.done = 1;
}

// the various signals
// we want their constructors to be inline

class space_t;
class thread_t;

class signal_reset_t : public signal_t 
{
  space_t* my_space;
  const l4_threadid_t* my_id;
  int my_init_prio;
  unsigned short my_mcp;

  virtual unsigned long func(thread_t *target);

public:
  signal_reset_t(space_t* space,
		 const l4_threadid_t* id, 
		 int init_prio, unsigned short mcp)
    : my_space(space), my_id(id), my_init_prio(init_prio),
      my_mcp(mcp)
  {}
};

class signal_init_t : public signal_t
{
  vm_offset_t my_eip, my_esp;
  thread_t* my_pager;
  thread_t* my_preempter;
  vm_offset_t *my_o_eip, *my_o_esp, *my_o_eflags;
  thread_t* *my_o_pager;
  thread_t* *my_o_preempter;

  virtual unsigned long func(thread_t *target);

public:
  signal_init_t(vm_offset_t eip, vm_offset_t esp,
		thread_t* pager, thread_t* preempter,
		vm_offset_t *o_eip, vm_offset_t *o_esp,
		thread_t* *o_pager, thread_t* *o_preempter,
		vm_offset_t *o_eflags)
    : my_eip(eip), my_esp(esp), my_pager(pager), my_preempter(preempter),
      my_o_eip(o_eip), my_o_esp(o_esp), my_o_eflags(o_eflags),
      my_o_pager(o_pager), my_o_preempter(o_preempter)
  {}
};

class signal_send_regs_t : public signal_t
{
  const thread_user_regs_t *my_sender_regs;

  virtual unsigned long func(thread_t *target);

public:
  signal_send_regs_t(const thread_user_regs_t *sr)
    : my_sender_regs(sr)
  {}
};

class signal_send_irq_t : public signal_t
{
  irq_t *my_irq;

  virtual unsigned long func(thread_t *target);

public:
  signal_send_irq_t(irq_t *i)
    : my_irq(i)
  {}
};

class signal_receiver_ready_t: public signal_t
{
  virtual unsigned long func(thread_t *target);

public:
  signal_receiver_ready_t()
  {}
};
  
class signal_schedule_t : public signal_t
{
  l4_sched_param_t my_param;
  thread_t* my_ext_preempter;
  l4_sched_param_t *my_o_param;
  l4_low_high_t *my_time;
  thread_t* *my_o_ext_preempter;
  sender_t* *my_partner;

  virtual unsigned long func(thread_t *target);

public:
  signal_schedule_t(l4_sched_param_t param, thread_t* pre, 
		    l4_sched_param_t *oparam, l4_low_high_t *t,
		    thread_t **opre, sender_t **partner)
    : my_param(param), my_ext_preempter(pre), my_o_param(oparam),
      my_time(t), my_o_ext_preempter(opre), my_partner(partner)
  {}
};

class signal_exception_t : public signal_t
{
  trap_state *_ts;
  vm_offset_t _handler;

  virtual unsigned long func(thread_t *target);

public:
  signal_exception_t(trap_state *ts, vm_offset_t handler)
    : _ts(ts), _handler(handler)
  {}
};

class signal_kill_t : public signal_t
{
  virtual unsigned long func(thread_t *target);

public:
  signal_kill_t()
  {}
};

class signal_kill_task_t : public signal_t
{
  unsigned _taskno;

  virtual unsigned long func(thread_t *target);

public:
  signal_kill_task_t(unsigned taskno)
    : _taskno(taskno)
  {}
};



#endif
