#include <assert.h>
#include <string.h>

#include "kmem.h"
#include "thread.h"
#include "globals.h"

#include "space.h"

// 
// class space_index_t
//

// class space_index_t::space_registry_t
// XXX should be page-aligned
space_index_t::space_registry_t space_index_t::spaces[max_space_number]; 

inline bool space_index_t::add(space_t *new_space, unsigned new_number)
{
  space_registry_t o(spaces[new_number]);
  assert(o.state.dead);

  space_registry_t n;
  n.space = new_space;

  return compare_and_swap(&spaces[new_number], o, n);
}

inline bool space_index_t::del(space_index_t number)
{
  assert(number < max_space_number);
  assert(! spaces[number].state.dead);

  // when deleting a task, write its owner (chief) into the task register
  spaces[number].state.dead = 1;
  spaces[number].state.chief = current()->space_index();

  return true;
}


// 
// class space_t
//

static pt_entry_t *split_pgtable(pd_entry_t *p); // forward decl

// allocator and deallocator

void *space_t::operator new(size_t)
{
  void *ret;

  check((ret = kmem::page_alloc()));

  return ret;
}

void space_t::operator delete(void *block)
{
  kmem::page_free(block);
}

// constructor

space_t::space_t(unsigned new_number)
{
  memset(dir, 0, PAGE_SIZE);	// initialize to zero

  kmem::dir_init(dir);		// copy current shared kernel page directory

  // scribble task/chief numbers into an unused part of the page directory
  dir[number_index] = new_number << 8; // must not be INTEL_PDE_VALID
  dir[chief_index] = space_index_t(new_number).chief() << 8; // dito

  space_index_t::add(this, new_number);	// register in task table
}

// destructor

space_t::~space_t()
{
  // free all page tables we have allocated for this address space
  for (unsigned i = 0; i < NPDES; i++)
    {
      if ((dir[i] & INTEL_PDE_VALID)
	  && !(dir[i] & (INTEL_PDE_SUPERPAGE | kmem::pde_global())))
	{
	  kmem::page_free(kmem::phys_to_virt(dir[i] & ~PAGE_MASK));
	}
    }

  // deregister from task table
  space_index_t::del(space_index_t(dir[number_index] >> 8));
}

// routines

space_t::status_t
space_t::v_insert(vm_offset_t phys, vm_offset_t virt, vm_offset_t size,
		  bool read_only = false)
{
  // insert page into page table

  // XXX should modify page table using compare-and-swap
  
  pd_entry_t *p = dir + ((virt >> PDESHIFT) & PDEMASK);
  pt_entry_t *e;
  
  if (! (*p & INTEL_PDE_VALID)) // we don't have a page table for this area
    {
      // check if we can insert a 4MB mapping
      if (size == SUPERPAGE_SIZE
	  && (cpu.feature_flags & CPUF_4MB_PAGES)
	  && superpage_aligned(virt) && superpage_aligned(phys))
	{
	  *p = phys | INTEL_PDE_SUPERPAGE 
	    | INTEL_PDE_VALID | INTEL_PDE_REF
	    | INTEL_PDE_MOD | INTEL_PDE_USER
	    | (read_only ? 0 : INTEL_PDE_WRITE);

	  return Insert_ok;
	}
      
      // can't map a superpage -- alloc new page table
      pt_entry_t *new_pt = static_cast<pt_entry_t *>(kmem::page_alloc());
      if (! new_pt)
	return Insert_err_nomem;
      
      memset(new_pt, 0, PAGE_SIZE);
      
      *p = kmem::virt_to_phys(new_pt)
	| INTEL_PDE_VALID | INTEL_PDE_WRITE 
	| INTEL_PDE_REF | INTEL_PDE_USER;
      e = new_pt + ((virt >> PTESHIFT) & PTEMASK);
    }
  else if (*p & INTEL_PDE_SUPERPAGE)
    {
      // already have mapped a superpage
      
      if (trunc_superpage(*p) + (virt & SUPERPAGE_MASK) != phys)
	return Insert_err_exists;

      // same mapping

      if ((*p & INTEL_PDE_WRITE) || read_only) // no upgrade?
	return Insert_warn_exists;

      // upgrade from read-only to read-write
      
      if (size == SUPERPAGE_SIZE)
	{
	  // upgrade the whole superpage to read-write at once
	  *p |= INTEL_PDE_WRITE;
	  
	  return Insert_warn_rw_upgrade;
	}

      // we need to split the superpage mapping because
      // it's only partly upgraded to read-write
      
      pt_entry_t *new_pt = split_pgtable(p);
      if (! new_pt)
	return Insert_err_nomem;
      
      *p = kmem::virt_to_phys(new_pt)
	| INTEL_PDE_VALID | INTEL_PDE_WRITE 
	| INTEL_PDE_REF | INTEL_PDE_USER;
      
      e = new_pt + ((virt >> PTESHIFT) & PTEMASK);
    }
  else
    {
      e = static_cast<pt_entry_t *>(kmem::phys_to_virt(*p & ~PAGE_MASK))
	+ ((virt >> PTESHIFT) & PTEMASK);
    }

if (size != PAGE_SIZE) printf("space.cc: size = 0x%x\n", size);
  assert(size == PAGE_SIZE);
  
  if (*e & INTEL_PTE_VALID)	// anything mapped already?
    {
      if (trunc_page(*e) != phys)
	return Insert_err_exists;
      
      if ((*e & INTEL_PTE_WRITE) || read_only) // no upgrade?
	return Insert_warn_exists;

      // upgrade from read-only to read-write
      *e |= INTEL_PTE_WRITE;
    }
  else			// we don't have mapped anything
    {
      *e = phys | INTEL_PTE_MOD | INTEL_PTE_VALID | INTEL_PTE_REF 
	| INTEL_PTE_USER 
	| (read_only ? 0 : INTEL_PTE_WRITE);
    }

  return Insert_ok;
}

// Look up a page table entry.  We find something, return true and the
// location and size of the mapped physical area.  If we don't find
// anything, return false and the size of the unmapped area.
bool 
space_t::v_lookup(vm_offset_t virt, vm_offset_t *phys = 0, 
		  vm_offset_t *size = 0, bool *read_only = 0)
{
  const pd_entry_t *p = dir + ((virt >> PDESHIFT) & PDEMASK);

  if (! (*p & INTEL_PDE_VALID))
    {
      if (size) *size = SUPERPAGE_SIZE;
      return false;
    }

  if (*p & INTEL_PDE_SUPERPAGE)
    {
      if (phys) *phys = *p & ~SUPERPAGE_MASK;
      if (size) *size = SUPERPAGE_SIZE;
      if (read_only) *read_only = ! (*p & INTEL_PDE_WRITE);

      return true;
    }

  const pt_entry_t *e = static_cast<pt_entry_t *>
    (kmem::phys_to_virt(*p & ~PAGE_MASK))
    + ((virt >> PTESHIFT) & PTEMASK);

  if (size) *size = PAGE_SIZE;

  if (! (*e & INTEL_PTE_VALID))
    return false;

  if (phys) *phys = *e & ~PAGE_MASK;
  if (read_only) *read_only = ! (*e & INTEL_PTE_WRITE);

  return true;
}

bool 
space_t::v_delete(vm_offset_t virt, vm_offset_t size, 
		  bool read_only = false)
{
  // delete page from page table
  
  for (vm_offset_t va = virt;
       va < virt + size;
       va += PAGE_SIZE)
    {
      pd_entry_t *p = dir + ((va >> PDESHIFT) & PDEMASK);
      pt_entry_t *e;
      
      if (! (*p & INTEL_PDE_VALID))
	{
	  // no page dir entry -- warp to next page dir entry
	  va = round_superpage(va + 1) - PAGE_SIZE;

	  continue;
	}

      if (*p & (INTEL_PDE_SUPERPAGE | kmem::pde_global()))
	{
	  // oops, a superpage or a shared ptable; see if we can unmap
	  // it at once, otherwise split it

	  if (superpage_aligned(va) // superpage boundary
	      && va + SUPERPAGE_SIZE <= virt + size)
	    {
	      // unmap superpage
	      if (read_only)
		*p &= ~INTEL_PDE_WRITE;
	      else
		*p = 0;
	      va += SUPERPAGE_SIZE - PAGE_SIZE;

	      continue;
	    }

	  // need to unshare/split

	  pt_entry_t *new_pt = split_pgtable(p);
	  if (! new_pt)
	    return false;
	  
	  *p = kmem::virt_to_phys(new_pt)
	    | INTEL_PDE_VALID | INTEL_PDE_WRITE 
	    | INTEL_PDE_REF | INTEL_PTE_USER;

	  e = new_pt + ((va >> PTESHIFT) & PTEMASK);
	}
      else
	e = static_cast<pt_entry_t *>(kmem::phys_to_virt(*p & ~PAGE_MASK))
	  + ((va >> PTESHIFT) & PTEMASK);

      if (read_only)
	*e &= ~INTEL_PTE_WRITE;
      else
	*e = 0;			// delete page table entry
    }

  return true;
}

//
// helpers
//

static pt_entry_t *
split_pgtable(pd_entry_t *p)
{
  // alloc new page table
  pt_entry_t *new_pt = static_cast<pt_entry_t *>(kmem::page_alloc());
  if (! new_pt)
    return 0;
	  
  if (*p & INTEL_PDE_SUPERPAGE)
    {
      for (unsigned i = 0; i < 1024; i++)
	{
	  new_pt[i] = ((*p & ~SUPERPAGE_MASK) + (i << PAGE_SHIFT))
	    | INTEL_PTE_MOD | INTEL_PTE_VALID | INTEL_PTE_REF | INTEL_PTE_USER
	    | ((*p & INTEL_PDE_WRITE) ? INTEL_PTE_WRITE : 0);
	}
    }
  else
    {
      memcpy(new_pt, p, PAGE_SIZE);
    }

  return new_pt;
}
