// -*- c++ -*-
#ifndef SPACE_H
#define SPACE_H

#include <stddef.h>
#include <flux/x86/types.h>
#include <flux/x86/paging.h>

#include "config.h"
#include "globals.h"
#include "atomic.h"
#include "kmem.h"

class space_t;
class thread_t;

class space_index_t
{
public:
  explicit space_index_t (unsigned number); // type-conversion cons.:
				// from task number

  space_t *lookup();
  thread_t *lookup_thread();	// find first thread of task
  operator unsigned ();		// yields the number itself

  space_index_t chief();	// return chief number

  bool set_chief(space_index_t old_chief, space_index_t new_chief);

  enum { max_space_number = 1 << 11 };

private:
  friend class space_t;

  static bool add(space_t *new_space, unsigned new_number);
  static bool del(space_index_t number);

  // implementation details follow...
  unsigned space_id;

  struct space_registry_t {
    union {
      space_t *space;
      struct {
	unsigned dead : 1;
	unsigned chief : 31;
      } state;
    };

    space_registry_t();		// constructor: initialize to "dead = 1"
  };

  static space_registry_t spaces[max_space_number];
};

class space_t
{
public:

  enum status_t {
    Insert_ok = 0,
    Insert_warn_exists,		// mapping already existed
    Insert_warn_rw_upgrade,	// mapping already existed and was r/w-upgraded
    Insert_err_nomem,		// couldn't alloc new page table
    Insert_err_exists		// a mapping already exists at the target addr
  };

  explicit space_t(unsigned new_number); // con: create new empty address space
  ~space_t();			// destructor

  // state requests/manipulation
  space_index_t space() const;  // returns task number
  space_index_t chief() const;  // returns chief number

  void switchin_context();

  vm_offset_t virt_to_phys(vm_offset_t virtual_address); // pgtable lookup

  status_t v_insert(vm_offset_t phys, vm_offset_t virt, vm_offset_t size,
		   bool read_only = false);
  bool v_delete(vm_offset_t virt, vm_offset_t size, 
		bool read_only = false);
  bool v_lookup(vm_offset_t virt, vm_offset_t *phys = 0, 
		vm_offset_t *size = 0, bool *read_only = 0);

  void kmem_update(vm_offset_t addr); // update from shared kernel pgtable

  // manip. i/o privilege map
  bool io_insert(vm_offset_t phys, vm_offset_t size);
  bool io_delete(vm_offset_t phys, vm_offset_t size);
  
  void *operator new(size_t);	// when creating task, makes sure pgdir is
				// page-aligned
  void operator delete(void *block);

private:
  space_t();			// default constructors are undefined
  space_t(const space_t&);

  // implementation details follow...

  // data layout (be careful here!):  just a page directory
  pd_entry_t dir[NPDES];

  // store the number of the address space in an unused part of the page dir
  const number_index 
    = (reinterpret_cast<unsigned long>(&_unused1_1) >> PDESHIFT) & PDEMASK;
  const chief_index 
    = (reinterpret_cast<unsigned long>(&_unused2_1) >> PDESHIFT) & PDEMASK;
  const brother_index 
    = (reinterpret_cast<unsigned long>(&_unused3_1) >> PDESHIFT) & PDEMASK;
};

// 
// inline implementations follow
//

// space_index_t

// constructor: initialize to "dead = 1"
inline space_index_t::space_registry_t::space_registry_t()
{ 
  state.dead = 1; 
} 

inline space_index_t::space_index_t(unsigned number)
  : space_id(number)
{ }

inline space_index_t::operator unsigned ()
{ return space_id; }

inline space_t *space_index_t::lookup()
{ 
  if (spaces[space_id].state.dead)
    return 0;

  return spaces[space_id].space;
}

inline thread_t * space_index_t::lookup_thread()
{ 
  return reinterpret_cast<thread_t*>(kmem::mem_tcbs
				     | ((space_id << 17) 
					* config::thread_block_id_factor)); 
}

inline bool 
space_index_t::set_chief(space_index_t old_chief, space_index_t new_chief)
{
  space_registry_t o(spaces[space_id]);

  if (! o.state.dead || o.state.chief != old_chief)
    return false;

  space_registry_t n;
  n.state.chief = new_chief;

  return compare_and_swap(&spaces[space_id], o, n);
}

// space_t

inline space_index_t space_t::space() const
{
  return space_index_t(dir[number_index] >> 8);
}

inline space_index_t space_t::chief() const
{
  return space_index_t(dir[chief_index] >> 8);
}

inline void space_t::switchin_context()
{
  register unsigned long my_pdir 
    = reinterpret_cast<unsigned long>(this) - kmem::mem_phys;

  if (get_pdbr() != my_pdir)
    set_pdbr(my_pdir);
}

inline void space_t::kmem_update(vm_offset_t addr)
{
  unsigned i = (addr >> PDESHIFT) & PDEMASK;
  dir[i] = kmem::dir()[i];
}

inline vm_offset_t
space_t::virt_to_phys(vm_offset_t a)
{
  if (this == sigma0)		// sigma0 doesn't have mapped everything
    return a;

  pd_entry_t p = dir[(a >> PDESHIFT) & PDEMASK];
  if (! (p & INTEL_PDE_VALID)) return 0xffffffff;
  if (p & INTEL_PDE_SUPERPAGE) return (p & ~SUPERPAGE_MASK) 
                                 | (a & SUPERPAGE_MASK);
  pt_entry_t t = reinterpret_cast<pt_entry_t *>
    ((p & ~PAGE_MASK) + kmem::mem_phys)[(a >> PTESHIFT) & PTEMASK];

  return (t & INTEL_PTE_VALID) ? (t & ~PAGE_MASK) | (a & PAGE_MASK)
    : 0xffffffff;
}

// more of space_index_t (stuff which is implemented in terms of space_t)

inline space_index_t space_index_t::chief()
{
  if (spaces[space_id].state.dead) // space does not exist?
    return space_index_t(spaces[space_id].state.chief); // we have the
				                        // chief number
  return lookup()->chief();	// otherwise ask the space object
}


#endif
