#include <l4/ipc.h>

#include <stdio.h>

#include "config.h"
#include "kdb.h"
#include "space.h"
#include "ipc.h"
#include "thread.h"
#include "thread_list.h"

unsigned (thread_t::*syscall_table[])(thread_regs_t *regs) = 
{ 
  thread_t::sys_ipc, 
  thread_t::sys_id_nearest, 
  thread_t::sys_fpage_unmap, 
  thread_t::sys_thread_switch, 
  thread_t::sys_thread_schedule, 
  thread_t::sys_thread_ex_regs, 
  thread_t::sys_task_new
};

unsigned 
thread_t::sys_id_nearest(thread_regs_t *regs)
{
  l4_threadid_t *dst_id_long = regs->thread_esi_edi();
  threadid_t dst_id(dst_id_long);

  if (dst_id.is_nil())
    {
      *dst_id_long = id();
      return 0;
    }

  panic("sys_id_nearest");

#if 0
  thread_t *partner = me->find_partner(dst_id);
  if (partner)
    {
      *dst_id_long = partner->id();
      return partner->chief() == me->chief() ?
        0 : (L4_IPC_REDIRECT_MASK 
             | ((partner->chief() == me->space()) ? 
                L4_IPC_SRC_MASK : 0));
    }
  
  /* non-existent thread which would belong to our clan? */
  if (me->chief() == dst_id_long->id.chief)
    return 0;

  /* target hasn't yet been assiged to a clan -- so it belongs to the 
     outermost clan */
  if (me->nest() > 0)
    {
      // send to our chief
      *dst_id_long = me->chief().lookup_thread()->id();
      return L4_IPC_REDIRECT_MASK;
    }

  /* oh, we're in the outermost clan already, and we can't let the
     kernel redirect a message to this destination */
  return L4_IPC_ENOT_EXISTENT;
#endif
}

unsigned 
thread_t::sys_thread_ex_regs(thread_regs_t *regs)
{
  l4_threadid_t dst_id_long = id();
  dst_id_long.id.lthread = regs->eax;
  threadid_t dst_id(&dst_id_long);

  thread_t *dst = (dst_id.lookup() == this)
    ? this
    : new (dst_id) thread_t;

  bool new_thread = false;

  if (! dst->exists())
    {
      new_thread = true;
      dst->reset(space(), &dst_id_long, my_prio, my_mcp); // ignore errs
    }

  thread_t *o_pager, *o_preempter;
  vm_offset_t o_flags;

  // initialize the thread with new values.

  // undocumented but jochen-compatible: if we're creating a new
  // thread (and aren't modifying an existing one), don't change the
  // register parameters that have been passed to us to ``old'' values
  // (i.e., all zeroes)
  if (dst->initialize(regs->edx, regs->ecx, 
		      threadid_t(regs->thread_esi_edi()).lookup(),
		      threadid_t(regs->thread_ebx_ebp()).lookup(),
		      new_thread ? 0 : & regs->edx, & regs->ecx, 
		      new_thread ? 0 : & o_pager, 
		      new_thread ? 0 : & o_preempter, 
		      new_thread ? 0 : & o_flags))
    {
      if (new_thread) return 0;

      *(regs->thread_esi_edi()) = o_pager->id();
      *(regs->thread_ebx_ebp()) = o_preempter->id();

      return o_flags;
    }

  return 0xffffffff;		// error
}

unsigned
thread_t::sys_fpage_unmap(thread_regs_t *regs)
{
  return ! fpage_unmap(space(), (l4_fpage_t){fpage: regs->eax},
		       regs->ecx & 0x80000000, 
		       !(regs->ecx & 2));
}

unsigned
thread_t::sys_thread_switch(thread_regs_t *regs)
{
  thread_t *t = threadid_t(regs->thread_esi_edi()).lookup();

  if (t && (t->state() & Thread_running))
    switch_to(t);		// this also subsumes the case t == nilthread
				// because the idle thread will just schedule
  else
    schedule();

  return 0;
}

unsigned
thread_t::sys_thread_schedule(thread_regs_t *regs)
{
  thread_t *dst = threadid_t(regs->thread_esi_edi()).lookup();

  if (!dst || dst->state() == Thread_invalid)
    return 0xffffffff;		// error

  l4_sched_param_t s;
  thread_t *opre;
  sender_t *partner;

  if (! dst->set_sched_param((l4_sched_param_t){sched_param: regs->eax},
			     threadid_t(regs->thread_ebx_ebp()).lookup(),
			     &s, regs->hilo_ecx_edx(),
			     &opre, &partner))
    return 0xffffffff;

  *regs->thread_esi_edi() = partner->id();
  *regs->thread_ebx_ebp() = opre->id();

  return s.sched_param;
}

unsigned
thread_t::sys_task_new(thread_regs_t *regs)
{
  l4_threadid_t *id = regs->thread_esi_edi();
  unsigned taskno = id->id.task;
  space_index_t si = space_index_t(taskno);

  space_t *s;

  for (;;)
    {
      if (si.chief() != space_index())
	{
	  if (config::conservative)
	    kdb::ke("task new: not chief");
	  break;
	}

      if ((s = si.lookup()))
	{
	  //
	  // task exists -- delete it
	  //

	  if (! space_index().lookup_thread()->kill_task(si))
	    {
	      printf("KERNEL: deletion of task 0x%x failed\n",
		     unsigned(si));
	      break;
	    }

	  delete s;
	}
      
      if (regs->ebx == 0)
	{
	  //
	  // do not create a new task -- transfer ownership
	  //
	  l4_threadid_t e;
	  e.lh.low = regs->eax;
	  
	  if (! si.set_chief(space_index(), space_index_t(e.id.task)))
	    break;		// someone else was faster
	  
	  id->id.lthread = 0;
	  id->id.chief = si.chief();
	  id->id.nest = 0;

	  space_index_t c = si.chief();
	  while (c.chief() != config::kernel_taskno)
	    {
	      c = c.chief();
	      id->id.nest++;
	    }

	  return 0;
	}
      
      //
      // create the task
      //
      s = new space_t(taskno);
      if (! s) 
	break;
      
      if (si.lookup() != s)
	{
	  // someone else has been faster that in creating this task!
	  
	  delete s;
	  continue;		// try again
	}
      
      id->id.lthread = 0;
      id->id.chief = space_index();
      id->id.nest = (nest() == 0 && space_index() == config::boot_taskno) 
	? 0 : nest() + 1;
  
      //
      // create the first thread of the task
      //
      thread_t *t = new (regs->thread_esi_edi()) thread_t;
      
      check(t->reset(s, id, my_prio,
		     (my_mcp < regs->eax) ? my_mcp : regs->eax));
      check(t->initialize(regs->edx, regs->ecx, 
			  threadid_t(regs->thread_ebx_ebp()).lookup(),
			  0));
      
      return 0;
    }

  *regs->thread_esi_edi() = L4_NIL_ID;
  return 0xffffffff;
}
