// -*- c++ -*-
#ifndef THREAD_H
#define THREAD_H

#include <flux/machine/base_trap.h>

#include <l4/types.h>
#include <l4/ipc.h>

#include "config.h"
#include "space.h"
#include "signal.h"
#include "atomic.h"
#include "list.h"

class thread_t;
class timeout_t;
class irq_t;

class threadid_t		// a shorter version of l4_threadid_t
{
public:
  threadid_t(const l4_threadid_t* public_id);	// type-conversion constructor
  threadid_t(thread_t* thread);	// type-conversion constructor
  thread_t *lookup() const;	// find thread control block (tcb)
  
  bool is_nil() const;
  bool is_valid() const;
  
private:
  // implementation details follow...
  // data layout
  thread_t *t;
};

enum
{
  Thread_invalid = 0,		// tcb unallocated
  Thread_running = 0x1,		// can be scheduled
  Thread_waiting = 0x2,		// waiting for any message
  Thread_receiving = 0x4,	// waiting for message from specified sender
  Thread_polling = 0x8,		// waiting to send message to specified rcvr
  Thread_ipc_in_progress = 0x10,// an IPC operation has not been finished
  Thread_send_in_progress = 0x20,// an IPC send operation has not been finished
  Thread_signalling = 0x40,	// waiting for a different thread to handle sig
  Thread_cancel = 0x80,		// state has been changed -- cancel activity
  Thread_dead = 0x100		// tcb allocated, but inactive (not in any q)
};

// the stack frame used as a parameter block for system services
struct thread_user_regs_t
{
  unsigned ecx;
  unsigned edx;
  unsigned esi;
  unsigned edi;
  unsigned ebx;
  unsigned ebp;
  unsigned eax;

  l4_threadid_t *thread_esi_edi();
  l4_threadid_t *thread_ebx_ebp();
  l4_low_high_t *hilo_ecx_edx();
};

// the stack frame pushed by all user->kernel entry points (including
// traps which invoke the debugger, slow traps, timer interrupt, page
// fault)
struct thread_ret_regs_t
{
  unsigned eip;
  unsigned short cs, __csu;
  unsigned eflags;
  unsigned esp;
  unsigned short ss, __ssu;
};

// the stack frame pushed by system call entry points
struct thread_regs_t : public thread_user_regs_t, public thread_ret_regs_t
{
};


class sender_t
{
public:
  //
  // state requests/manipulation
  //
  unsigned state();
  l4_threadid_t id() const;
  space_index_t space_index() const;	// returns task number
  space_index_t chief_index() const;	// returns chief number
  unsigned nest() const;
  bool exists();

  // activation function
  virtual bool ipc_receiver_ready() = 0;

protected:
  // implementation details follow...

  // data layout (be careful here!)
  unsigned my_state;
  l4_threadid_t my_id;
  sender_t *sender_next, *sender_prev;

  // queueing functions
  bool in_sender_list();
  void sender_enqueue(thread_t *receiver);
  void sender_dequeue(thread_t *receiver);

  // state requests/manipulation
  bool state_add(unsigned bits);
  bool state_del(unsigned bits);
  bool state_change(unsigned mask, unsigned bits);
  bool state_change_safely(unsigned mask, unsigned bits);

};

class thread_t : public sender_t
{
public:
  static const vm_size_t size = config::thread_block_size;

  //
  // state requests/manipulation
  //
  space_t *space() const;

  //
  // activation routines:
  //

  // reset a thread control block.  it is an error to reset a tcb
  // which is not Thread_invalid
  bool reset(space_t* t, const l4_threadid_t* id, int prio, 
	     unsigned short mcp);

  // all other routines return an error if they're Thread_invalid

  // end IPC (if any is in progress) and resume at given address.
  bool initialize(vm_offset_t eip, vm_offset_t esp,
		  thread_t* pager, thread_t* preempter,
		  vm_offset_t *o_eip = 0, 
		  vm_offset_t *o_esp = 0,
		  thread_t* *o_pager = 0, 
		  thread_t* *o_preempter = 0,
		  vm_offset_t *o_eflags = 0);

  // send an IPC message
  unsigned ipc_send_regs(const thread_user_regs_t *sr);

  // send an hardware interrupt IPC
  int ipc_send_irq(irq_t *i);

  // signal that the receiver is ready for a handshake
  virtual bool ipc_receiver_ready();

  // modify scheduling parameters
  bool set_sched_param(l4_sched_param_t param, thread_t* pre, 
		       l4_sched_param_t *oparam, l4_low_high_t *t,
		       thread_t **opre, sender_t **partner);

  // raise an exception
  bool raise_exception(trap_state *ts, vm_offset_t handler);

  // kill this thread and task
  bool kill(); 

  // kill a subtask.  this must be executed on thread 0 of the chief
  // of the task to be killed
  bool kill_task(space_index_t subtask);

  //
  // scheduler
  //
  void schedule();		// select a different thread for running

  //
  // system calls and other kernel entry points
  //
  unsigned sys_ipc(thread_regs_t *regs);
  unsigned sys_id_nearest(thread_regs_t *regs);
  unsigned sys_fpage_unmap(thread_regs_t *regs);
  unsigned sys_thread_switch(thread_regs_t *regs);
  unsigned sys_thread_schedule(thread_regs_t *regs);
  unsigned sys_thread_ex_regs(thread_regs_t *regs);
  unsigned sys_task_new(thread_regs_t *regs);

  bool handle_page_fault(vm_offset_t pfa, unsigned error_code);
  bool handle_page_fault_pager(vm_offset_t pfa, unsigned error_code);
  bool handle_slow_trap(trap_state *state);

  //
  // type operations
  //
  thread_t();			// constructor
  virtual ~thread_t();		// destructor
  void *operator new(size_t, threadid_t id); // allocate tcb in tcb space
  void operator delete(void *block);

protected:
  thread_t(const thread_t&);	// default constructor is undefined
  void *operator new(size_t);	// default new operator undefined

  // implementation details follow...

  // data layout (be careful here!)

  unsigned short my_prio, my_timeslice;
  unsigned long *kernel_sp;	// 
  simple_lock_t switching_in;
  
  // signal sending state
  list_stack_t<signal_t> my_signals; // list of signals I've been sent
  signal_t *my_sent_signal;	// signal I'm being sending
  thread_t *my_signal_partner;	// who i'm sending a signal to

  // ipc state
  sender_t *my_partner;		// IPC partner I'm waiting for/involved with
  thread_t *my_send_partner;
  timeout_t *my_timeout;
  thread_user_regs_t *my_receive_regs; // registers used for receive operation

  thread_t *my_pager, *my_preempter, *my_ext_preempter;
  space_t *my_space;
  
  thread_t *present_next, *present_prev;
  thread_t *ready_next, *ready_prev;
  sender_t *sender_first;

  unsigned short my_mcp;

  irq_t *my_irq;
  
  x86_gate *my_idt;
  unsigned short my_idt_limit;

  unsigned _magic;
  static const unsigned magic = 0xf001c001;
  // end of data

  // private functions -- these must only ever be called for *this,
  // not for any other objects!

  // enqueue a signal for this thread
  bool insert_signal(signal_t *s);

  // state requests/manipulation
  thread_ret_regs_t *regs() const;

  thread_t *find_partner(threadid_t dest_id);
  
  // signal handling
  unsigned long send_signal(thread_t *to, signal_t *s);

  // thread-wide functions
  bool switch_to(thread_t *next); // actually switch to different thread

  // deliver any accumulated signals to thread and switch page table
  // if nessecary
  bool do_signals();
  bool switchin_context(); 

  void do_kill();
  unsigned do_send(thread_t *partner, l4_timeout_t t, 
		   thread_user_regs_t *regs);
  unsigned do_receive(sender_t *partner, l4_timeout_t t, 
		      thread_user_regs_t *regs);
  void prepare_receive(sender_t *partner, thread_user_regs_t *regs);

  static thread_t *prio_first[256];
  static thread_t *prio_next[256];
  static unsigned prio_highest;

  bool in_ready_list();
  void ready_enqueue();
  void ready_dequeue();
  bool in_present_list();
  void present_enqueue(thread_t *sibling);
  void present_dequeue();
  sender_t *sender_dequeue_first();

  static int (*nested_trap_handler)(trap_state *state);

  static void user_invoke();

  friend class sender_t;	// so that senders can enqueue in sender list
  friend class timeout_t;	// timeouts may change our state
  friend class irq_t;		// as well as irqs

  friend class signal_t;	// so that signals can call private functions
  friend class signal_reset_t;
  friend class signal_init_t;
  friend class signal_send_regs_t;
  friend class signal_send_irq_t;
  friend class signal_receiver_ready_t;
  friend class signal_schedule_t;
  friend class signal_kill_t;
};

// 
// inline implementations follow
//

// threadid_t

inline threadid_t::threadid_t(const l4_threadid_t* public_id)
  : t(static_cast<thread_t *>(public_id->lh.low == 0xffffffff 
			      ? 0 // invalid
			      : kmem::mem_tcbs
			        | ((public_id->lh.low 
				    * config::thread_block_id_factor)
				   & ~config::thread_block_mask)))
{ }

inline threadid_t::threadid_t(thread_t* thread)
  : t(thread)
{ }

inline thread_t * threadid_t::lookup() const
{ return t; }

inline bool threadid_t::is_nil() const
{ return static_cast<unsigned long>(t) == kmem::mem_tcbs; }

inline bool threadid_t::is_valid() const
{ return t; }



// thread_user_regs_t

inline l4_threadid_t * thread_user_regs_t::thread_esi_edi()
{ return static_cast<l4_threadid_t *>(&this->esi); }

inline l4_threadid_t * thread_user_regs_t::thread_ebx_ebp()
{ return static_cast<l4_threadid_t *>(&this->ebx); }

inline l4_low_high_t * thread_user_regs_t::hilo_ecx_edx()
{ return static_cast<l4_low_high_t *>(&this->ecx); }


// sender_t

inline l4_threadid_t sender_t::id() const
{ return this ? my_id : L4_INVALID_ID; } // also works for this == 0

inline unsigned sender_t::state()
{ return my_state; }

inline space_index_t sender_t::space_index() const
{ return space_index_t(my_id.id.task); }

inline space_index_t sender_t::chief_index() const
{ return space_index_t(my_id.id.chief); }

inline unsigned sender_t::nest() const
{ return my_id.id.nest; }

inline bool sender_t::state_add(unsigned bits)
{ 
  bool ret;  
  unsigned old;
  do 
    {
      old = my_state;
      ret = (my_state & bits) == 0;
    }
  while (! compare_and_swap(&my_state, old, old | bits));

  return ret;
}

inline bool sender_t::state_del(unsigned bits)
{ 
  bool ret;  
  unsigned old;
  do 
    {
      old = my_state;
      ret = (my_state & bits) == bits;
    }
  while (! compare_and_swap(&my_state, old, old & ~bits));

  return ret;
}

inline bool sender_t::state_change(unsigned mask, unsigned bits)
{
  bool ret;  
  unsigned old;
  do 
    {
      old = my_state;
      ret = ((old & bits & mask) == 0) && ((old & ~mask) == ~mask);
    }
  while (! compare_and_swap(&my_state, old, (old & mask) | bits));

  return ret;
}

inline bool sender_t::state_change_safely(unsigned mask, unsigned bits)
{
  bool ret;  
  unsigned old;
  do 
    {
      old = my_state;
      ret = ((old & bits & mask) == 0) && ((old & ~mask) == ~mask);
      if (! ret)
	return false;
    }
  while (! compare_and_swap(&my_state, old, (old & mask) | bits));

  return true;
}

inline bool sender_t::exists()
{ return my_state != Thread_invalid; }


// thread_t

inline void *thread_t::operator new(size_t, threadid_t id)
{
  // allocate tcb in tcb space
  return static_cast<void*>(id.lookup());
}

// routines which are implemented as signals should be inline
inline bool thread_t::reset(space_t* space,
			    const l4_threadid_t* id, 
			    int init_prio, unsigned short mcp)
{
  signal_reset_t s(space, id, init_prio, mcp);
  return current()->send_signal(this, &s) == signal_t::Sig_success;
}

inline bool thread_t::initialize(vm_offset_t eip, vm_offset_t esp,
				 thread_t* pager, thread_t* preempter,
				 vm_offset_t *o_eip = 0, 
				 vm_offset_t *o_esp = 0,
				 thread_t* *o_pager = 0, 
				 thread_t* *o_preempter = 0,
				 vm_offset_t *o_eflags = 0)
{
  signal_init_t s(eip, esp, pager, preempter, 
		  o_eip, o_esp, o_pager, o_preempter, o_eflags);
  return current()->send_signal(this, &s) == signal_t::Sig_success;
}

inline unsigned 
thread_t::ipc_send_regs(const thread_user_regs_t *sr)
{
  signal_send_regs_t s(sr);

  unsigned long ret = current()->send_signal(this, &s);

  // if send_signal() has to abort for some reason, convert to an IPC
  // error code here
  return ret == static_cast<unsigned long>(signal_t::Sig_aborted)
    ? L4_IPC_SECANCELED
    : ret;
}

inline int
thread_t::ipc_send_irq(irq_t *i)
{
  signal_send_irq_t s(i);

  return current()->send_signal(this, &s);
}

inline bool 
thread_t::set_sched_param(l4_sched_param_t param, thread_t* pre, 
			  l4_sched_param_t *oparam, l4_low_high_t *t,
			  thread_t **opre, sender_t **partner)
{
  signal_schedule_t s(param, pre, oparam, t, opre, partner);

  return current()->send_signal(this, &s) == signal_t::Sig_success;
}  

inline bool 
thread_t::raise_exception(trap_state *ts, vm_offset_t handler)
{
  signal_exception_t s(ts, handler);

  return current()->send_signal(this, &s) == signal_t::Sig_success;
}

inline bool 
thread_t::kill()
{
  signal_kill_t s;

  return current()->send_signal(this, &s) == signal_t::Sig_success;
}

inline bool 
thread_t::kill_task(space_index_t subtask)
{
  signal_kill_task_t s(subtask);

  return current()->send_signal(this, &s) == signal_t::Sig_success;
}



// thread_t private functions

inline space_t *thread_t::space() const
{ return my_space; }

inline thread_ret_regs_t *thread_t::regs() const
{ 
  return static_cast<thread_ret_regs_t *>(static_cast<unsigned long>(this) 
					  + size - sizeof(thread_ret_regs_t));
}

inline bool thread_t::insert_signal(signal_t *s)
{
  my_signals.insert(s);
  /* this cannot be used as we send the reset signal while the thread is 
     still invalid:
     return (my_state != Thread_invalid); */
  return true;
}


#endif
