// -*- c++ -*-

#ifndef TIME_H
#define TIME_H

#include "config.h"
#include "kmem.h"
#include "config.h"
#include "thread.h"
#include "thread_list.h"

class timeout_t
{
public:
  bool is_set();
  bool has_hit();

  void set(unsigned long long abs_microsec);  
  void reset();

  thread_t *owner();

  timeout_t();			// constructor

private:
  timeout_t(const timeout_t&);	// default copy constructor is undefined

  bool check_hit();

  // implementation details follow
  unsigned long long _wakeup;	// absolute time we want to be woken up
  timeout_t *_next, *_prev;	// next/prev in timer queue

  union {
    struct {
      bool set : 1;
      bool hit : 1;
    } _flags;
    unsigned short _flagword;
  };

  friend class timer;
};

class timer
{
public:
  static void update_system_clock();


private:
  timer();			// default constructors are undefined
  timer(const timer&);

  friend class timeout_t;	// timeouts can add themselves to the timeout
				// queue

  static void do_timeouts();

  // implementation details follow

  static timeout_t *first_timeout;
};

//
// inlines
//

// timeout_t

inline 
timeout_t::timeout_t()
{
  _flagword = 0;
}

inline bool 
timeout_t::is_set()
{
  return _flags.set;
}

inline bool 
timeout_t::has_hit()
{
  return _flags.hit;
}

inline thread_t *
timeout_t::owner()
{
  return static_cast<thread_t *>(static_cast<vm_offset_t>(this) 
				 & ~config::thread_block_mask);
}

inline void
timeout_t::set(unsigned long long abs_microsec)
{
  // XXX uses cli/sti
  
  _wakeup = abs_microsec;
  _flags.set = 1;

  unsigned flags = get_eflags();
  cli();

  if (!timer::first_timeout)
    {
      timer::first_timeout = this;
      _prev = _next = 0;
    }
  else
    {
      timeout_t *ti = timer::first_timeout;

      for (;;)
	{
	  if (ti->_wakeup >= _wakeup)
	    {
	      // insert before ti
	      _next = ti;
	      _prev = ti->_prev;
	      ti->_prev = this;
	      if (_prev) _prev->_next = this;

	      goto done;
	    }

	  if (ti->_next)
	    {
	      ti = ti->_next;
	      continue;
	    }
	}

      // insert as last item after ti
      ti->_next = this;
      _prev = ti;
      _next = 0;
    }

done:
  set_eflags(flags);
}

inline void 
timeout_t::reset()
{
  // XXX uses cli/sti

  if (! is_set())
    return;			// avoid cli/sti overhead if not set

  unsigned flags = get_eflags();
  cli();

  if (is_set())
    {
      if (_prev) _prev->_next = _next;
      if (_next) _next->_prev = _prev;
    }

  set_eflags(flags);

  _flags.set = 0;
}

inline bool 
timeout_t::check_hit()
{
  // XXX assume we run with irqs off

  if (kmem::info()->clock < _wakeup)
    return false;

  timer::first_timeout = _next;		// dequeue
  if (_next) _next->_prev = 0;
  _flags.hit = 1;
  _flags.set = 0;
  owner()->state_change(~(Thread_polling|Thread_waiting|Thread_receiving), 
			Thread_running); // set thread running

  owner()->ready_enqueue();

  return true;
}

// timer

inline void 
timer::do_timeouts()
{
  while (first_timeout)
    {
      if (! first_timeout->check_hit())
	break;
    }
}

inline void
timer::update_system_clock()
{
  kmem::info()->clock += config::microsec_per_tick;

  do_timeouts();
}

//
// helpers
//

inline unsigned long long
timeout_to_microsec(unsigned man, unsigned exp)
{
  return static_cast<unsigned long long>(man) << ((15 - exp) << 1);
}

#endif
