/*
 *  $Id: l4_com.c,v 1.1.1.1 1999/04/06 19:20:06 yoonho Exp $
 */
#include <l4/l4_com.h>

#include <l4/types.h>
#include <l4/syscalls.h>
#include <l4/ipc.h>
#include <l4/kdebug.h>



void set_8250_mode(void);
void unmask_irq3(void);
char readChar(void);
void writeChar(char);
void putDebugChar(char);
char getDebugChar(void);
void COM_irq_thread(void);
void Init_COM_thread(void);
void writeString(char *ptr);

extern void log(const char *);

typedef struct
{
    l4_fpage_t fp_opt;
    l4_msgdope_t dope1;
    l4_msgdope_t dope2;
}message;


/*
 * Initializes the SIO for DIVISOR bps, 8N1, only receive interrupts
 */

void
set_8250_mode(void)
{
    asm(
        /* set speed in DLLO/DLHI */
        "             movw %0,%%dx               \n\t" /* adr of LCR */
        "             inb %%dx,%%al              \n\t"
        "             orb $0x80,%%al             \n\t" /* set DLAB in LCR */ 
        "             outb %%al,%%dx             \n\t"  

        "             movw %2,%%dx               \n\t" /* adr of DLLO */
        "             movb %3,%%al               \n\t" /* lo divisor value */
        "             outb %%al,%%dx             \n\t"

        /* DLHI can be omitted for speed > 460 bps */
        
        "             movw %0,%%dx               \n\t" /* adr of LCR */
        "             movb $0x03,%%al            \n\t" /* 8N1, clear DLAB */
        "             outb %%al,%%dx             \n\t"

        /* reset 8250 to cancel possible pending interrupts */
        "             movw %4,%%dx               \n\t" /* read IIR */
	"             inb %%dx,%%al              \n\t" 
        "             movw %5,%%dx               \n\t" /* read LSR */
	"             inb %%dx,%%al              \n\t" 
        "             movw %4,%%dx               \n\t" /* read IIR */
	"             inb %%dx,%%al              \n\t"
        "             movw %6,%%dx               \n\t" /* read RBR */
	"             inb %%dx,%%al              \n\t"
        "             movw %4,%%dx               \n\t" /* read IIR */
	"             inb %%dx,%%al              \n\t"
        "             movw %7,%%dx               \n\t" /* read MSR */
	"             inb %%dx,%%al              \n\t"
        "             movw %4,%%dx               \n\t" /* read IIR */
	"             inb %%dx,%%al              \n\t"

        /* enable RCV interrupts, clear rest */
        "             movw %1,%%dx               \n\t" /* adr of IER */
        "             movb $0x01,%%al            \n\t"
        "             outb %%al,%%dx             \n\t"

        /* set DTR and RTS in Modem Control Register */
	"             movw %8,%%dx               \n\t" /* adr of MCR */
        "             movb $0b01011,%%al         \n\t" /* set DTR, RTS, MCRE */
        "             outb %%al,%%dx             \n\t"
        :
        /* no output */
        :
        "i" (COM_LCR),               /* 0 */
        "i" (COM_IER),               /* 1 */
        "i" (COM_DLLO),              /* 2 */
        "i" (DIVISOR),               /* 3 */
        "i" (COM_IIR),               /* 4 */
        "i" (COM_LSR),               /* 5 */
        "i" (COM_RBR),               /* 6 */
        "i" (COM_MSR),               /* 7 */
        "i" (COM_MCR)                /* 8 */
        :
        "eax", "edx"
    );
}

/*
 * unmask IRQ3 by programming the PIC
 */
void unmask_irq3(void)
{
  asm(
      "             inb $0x21,%%al             \n\t"
      "             andb $0xf7,%%al            \n\t" /* mask for IRQ 3 */
      "             outb %%al,$0x21            \n\t"
      :
      :
      :
      "eax"
      );  
}

/*
 * Sends the character to SIO via polling
 */
void 
writeChar(char value)
{
    asm
    (
        "             movw %1,%%dx               \n\t"   /* adr of LSR */
        "0:           inb %%dx,%%al              \n\t"    
        "             testb $0b01000000,%%al     \n\t"   /* polling */
        "             jz 0b                      \n\t"

        "             movb %0,%%al               \n\t"
        "             movw %2,%%dx               \n\t"   /* adr of THR */
        "             outb %%al,%%dx             \n\t"
        : 
        /* no output */
        :
        "r" (value),       /* 0 */
        "i" (COM_LSR),     /* 1 */
        "i" (COM_THR)      /* 2 */
        :
        "eax", "edx"
    );
}

/*
 * Reads one character from SIO (after interrrupt occured)
 */
char
readChar(void)
{
    int result;

    asm
    (
     /* no test of line status necessary, received interrupt means char can be
        read from RBR */
     
        "             movw %2,%%dx               \n\t"  /* adr of RBR */
        "             inb %%dx,%%al              \n\t"
        "             movzbl %%al,%%eax          \n\t"  /* zero extend al */
        :
        "=a" (result)      /* 0 */
        :
        "i" (COM_LSR),     /* 1 */
        "i" (COM_RBR)      /* 2 */     
        :
        "eax", "edx"
    );
    return (char) result;
}

#define STACK_SIZE 256
#define LTHREAD_NO_GDB_COM 9
#define COM_BUFFER_SIZE 512

char COM_buffer[COM_BUFFER_SIZE];         /* circ. buffer for received chars */
volatile int rd_ptr = 0, wr_ptr = 0;        /* indices into buffer */

void putDebugChar(char ch)
{
  writeChar(ch);
}

char getDebugChar(void)
{
  char ch;

  /* read from circular buffer instead of directly from serial port */
  while(rd_ptr == wr_ptr); /* no char in buffer, wait for interrupt */
  ch = COM_buffer[rd_ptr++];
  if (rd_ptr == COM_BUFFER_SIZE)
    {
      rd_ptr = 0;                /* wrap around */
    }
  return ch;
}

void Init_COM_thread(void)
{
  static char threadstack[STACK_SIZE];
  static int initialized = 0;
  char *stackpt = & threadstack[STACK_SIZE -1];

  dword_t dummy, w0, w1;
  l4_threadid_t me, my_preempter, my_pager, COM_th_id;
  l4_msgdope_t result;
  message rcvmsg = { {0}, {0}, {0} };
  int ret;

  if (initialized)
    goto irq_init;

  initialized = 1;

  /* get thread parameters */
  me = l4_myself();
  my_preempter = L4_INVALID_ID;
  my_pager = L4_INVALID_ID;
  l4_thread_ex_regs(me, (dword_t) -1, (dword_t) -1,
                    &my_preempter,
                    &my_pager,
                    &dummy,
                    &dummy,
                    &dummy);
  COM_th_id = me;
  COM_th_id.id.lthread = LTHREAD_NO_GDB_COM;

  /* create thread */
  l4_thread_ex_regs(COM_th_id,
		    (dword_t) COM_irq_thread, /* eip */
		    (dword_t) stackpt, /* esp */
		    &my_preempter, /* preempter */
		    &my_pager,     /* pager */
		    &dummy,     /* old_flags */
		    &dummy,     /* old_eip */
		    &dummy);    /* old_esp */

  /* make sure, COM_irq_thread is running before continuing */
  ret = l4_i386_ipc_wait(&COM_th_id, &rcvmsg, &w0, &w1,  L4_IPC_NEVER, 
		   &result);

irq_init:

  set_8250_mode();		/* set characteristics of serial interface */
  unmask_irq3();

  log("Ending Init_COM_Thread\n");  
}

void COM_irq_thread(void)
{
  message msg = { {0}, {0}, {0} };
  dword_t dummy;
  l4_msgdope_t dummydope;
  int code;
  l4_threadid_t me, irq_th;  
  char ch;

  me =l4_myself();

  irq_th.lh.low = COM_IRQ + 1;
  irq_th.lh.high = 0;

  /* try to attach to IRQ thread */
  code = l4_i386_ipc_receive(irq_th,
                             0, /* receive descriptor */
                             &dummy,
                             &dummy,
                             L4_IPC_TIMEOUT(0,0,0,1,0,0), /* rcv = 0,
                                                             snd = inf */
                             &dummydope);

  if (code != L4_IPC_RETIMEOUT)
    {
      log("irq_thread: can't register to irq. Stopping.\n"); 
      while(1);
    }
  else
    {
      log("Registered to COM irq.\n");
    }
  /* synchronize to booter task */
  me.id.lthread = 0; 
  l4_i386_ipc_send(me, &msg, (dword_t) 0, (dword_t) 0, L4_IPC_NEVER, 
		   &dummydope);
  /*
   * enter infinite loop of reading and storing characters
   */
  for (;;) 
    {
      code = l4_i386_ipc_receive(irq_th,
                                 0, /* receive descriptor */
                                 &dummy,
                                 &dummy,
                                 L4_IPC_NEVER,
                                 &dummydope);
      if (code)
        {
          log("irq_thread: irq receive failed\n");
        }

      ch = readChar();  /* read received char from port */
      asm
	(  /* acknowledge the interrupt, unspecific EOI */
	 "             movb $0x20,%%al               \n\t"
	 "             outb %%al,$0x20               \n\t"
	 : /* no output */
	 : /* no input */
	 : "al"
	 );
      COM_buffer[wr_ptr++] = ch;  
      if (wr_ptr == COM_BUFFER_SIZE)
	{ 
	  wr_ptr = 0;  /* wrap around */
	}
      if (wr_ptr == rd_ptr)
	{ /* pointers overlap, COM_buffer[rd_ptr] is lost */
	  log("Buffer overflow. Losing characters.\n");
	}
    }
}

/*
 * Writes the string pointed to by ptr to COM2
 */

void writeString(char *ptr)
{
  while(*ptr != '\0')  
    {
      writeChar(*ptr);
      ptr++;
    }
}








