/*
 * $Id: wait_queue.c,v 1.1.1.1 1999/04/06 19:20:15 yoonho Exp $
 */

/*****************************************************************************
 * wait_queue.c                                                              *
 * simple wait queue implementation                                          *
 *****************************************************************************/


/* L4 includes */
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>

#include <l4/wait_queue.h>

#define __sti() __asm__ __volatile__ ("sti": : :"memory")
#define __cli() __asm__ __volatile__ ("cli": : :"memory")

#define __save_flags(x) \
__asm__ __volatile__("pushfl ; popl %0":"=g" (x): /* no input */ :"memory")

#define __restore_flags(x) \
__asm__ __volatile__("pushl %0 ; popfl": /* no output */ :"g" (x):"memory")

#define WQ_WAKEUP_MAGIC   0xfee1dead
#define WQ_WAKEUP_TIMEOUT L4_IPC_TIMEOUT(98,9,0,0,0,0)  /* 100ms */

/*****************************************************************************
 * suspend thread                                                            *
 *****************************************************************************/
extern inline void __wq_suspend_thread(l4_threadid_t wakeup_thread)
{
  int error;
  dword_t dummy;
  l4_msgdope_t result;

  while(1)
    {
#if 0
      ko('s');
#endif

      error = l4_i386_ipc_receive(wakeup_thread,L4_IPC_SHORT_MSG,
				  &dummy,&dummy,L4_IPC_NEVER,&result);

#if 0
      ko('w');
#endif

      if (error)
	{
	  outstring("suspend_thread: IPC error ");
	  outhex16(error);
	  outstring("\n\r");
	  continue;
	}

      if (dummy == WQ_WAKEUP_MAGIC)
	return;

      outstring("suspend_thread: ");
      outstring("bad wakeup message!\n\r");
    }
}

/*****************************************************************************
 * wake up next thread in wait queue                                         *
 *****************************************************************************/
extern inline void __wq_wakeup_thread(l4_threadid_t tid)
{
  int error;
  l4_msgdope_t result;

  /* send wakup message */
  error = l4_i386_ipc_send(tid,L4_IPC_SHORT_MSG,
			   WQ_WAKEUP_MAGIC,WQ_WAKEUP_MAGIC,
			   WQ_WAKEUP_TIMEOUT,&result);
  
  if (error)
    {
      outstring("wakeup_thread: ");
      outstring("wakeup failed (");
      outhex16(error);
      outstring(")\n\r");
    }
}

/*****************************************************************************
 * acquire lock                                                              *
 *****************************************************************************/
inline void l4_wq_lock(l4_wait_queue_t *wq)
{
  unsigned long flags;
  l4_wait_queue_entry_t wqe;
  l4_threadid_t wakeup_thread;

  __save_flags(flags);
  __cli();
  
  if (l4_is_invalid_id(wq->owner))
    {
      /* lock free */
      wq->owner = l4_myself();
      __restore_flags(flags);
      return;
    }

  wqe.thread = l4_myself();
  wqe.next = NULL;

  if (!wq->head)
    {
      /* wait queue empty */
      wq->head = wq->tail = &wqe;
      wakeup_thread = wq->owner;
    }
  else
    {
      /* insert thread into wait queue */
      wakeup_thread = wq->tail->thread;
      wq->tail->next = &wqe;
      wq->tail = &wqe;
    }

  __wq_suspend_thread(wakeup_thread);

  /* acquire lock */
  wq->owner = l4_myself();

  /* remove thread from wait queue */
  if (wq->head != &wqe)
    {
      outstring("lock: wait queue destroyed!\n\r");
      enter_kdebug("wq_lock");
      __restore_flags(flags);
      return;
    }

  wq->head = wq->head->next;
  if (!wq->head) 
    wq->tail = NULL;

  /* done */
  __restore_flags(flags);
}

/*****************************************************************************
 * release lock                                                              *
 *****************************************************************************/
inline void l4_wq_unlock(l4_wait_queue_t *wq)
{
  unsigned long flags;
  l4_threadid_t tid;

  __save_flags(flags);
  __cli();

  if (!thread_equal(wq->owner,l4_myself()))
    {
      outstring("unlock: ");
      outstring("what's that: owner != me\n");
      enter_kdebug("wq_unlock");
      __restore_flags(flags);
      return;
    }

  if (!wq->head)
    {
      /* empty wait queue, release lock */
      wq->owner = L4_INVALID_ID;
      __restore_flags(flags);
      return;
    }

  /* wake up first thread in wait queue */
  tid = wq->head->thread;
  __wq_wakeup_thread(tid);

  /* done */
  __restore_flags(flags);
}
