/*
 * $Id: mem_lock.c,v 1.2 1999/05/13 15:22:10 yoonho Exp $
 */

/*****************************************************************************
 * liboskit_support/src/mem_lock.c                                           *
 * OSKit support functions, mem_lock implementation                          *
 *****************************************************************************/

/* L4 includes */
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>

#ifndef NULL
#define NULL ((void *) 0)
#endif

#define mb()  __asm__ __volatile__ (""   : : :"memory")
#define sti() __asm__ __volatile__ ("sti": : :"memory")
#define cli() __asm__ __volatile__ ("cli": : :"memory")

#define save_flags(x) \
__asm__ __volatile__("pushfl ; popl %0":"=g" (x): /* no input */ :"memory")

#define restore_flags(x) \
__asm__ __volatile__("pushl %0 ; popfl": /* no output */ :"g" (x):"memory")

struct mem_lock_wait_queue {
  unsigned lthread;
  struct mem_lock_wait_queue *next; 
};

static int mem_lock_lock = 0;
static unsigned char mem_lock_initialized = 0;
struct mem_lock_wait_queue mem_lock_queue, *lq = NULL;

#define WAIT_QUEUE_HEAD(x) ((struct mem_lock_wait_queue *)((x)-1))

static inline void 
mem_lock_init_waitqueue(struct mem_lock_wait_queue **q)
{
  *q = WAIT_QUEUE_HEAD(q);
}

extern inline void 
__mem_lock_add_wait_queue(struct mem_lock_wait_queue ** p, 
			  struct mem_lock_wait_queue * wait)
{
  struct mem_lock_wait_queue *head = *p;
  struct mem_lock_wait_queue *next = WAIT_QUEUE_HEAD(p);
  
  if (head)
    next = head;
  *p = wait;
  wait->next = next;
}

extern inline void 
__mem_lock_remove_wait_queue(struct mem_lock_wait_queue ** p, 
			     struct mem_lock_wait_queue * wait)
{
  struct mem_lock_wait_queue * next = wait->next;
  struct mem_lock_wait_queue * head = next;
  
  for (;;) {
    struct mem_lock_wait_queue * nextlist = head->next;
    if (nextlist == wait)
      break;
    head = nextlist;
  }
  head->next = next;
}

#define WAKEUP_MAGIC 0xbadefeed

extern inline void
mem_lock_suspend(void)
{
  int ipc_error;
  l4_msgdope_t ipc_res;
  dword_t r_kind;
  l4_threadid_t th;
  
  while(1)
    {
#ifdef __L4_VERSION_X__
      ipc_error = l4_i386_ipc_wait(&th,
				   L4_IPC_SHORT_MSG, 
				   &r_kind, 
				   &r_kind,
				   &r_kind,
				   L4_IPC_NEVER, 
				   &ipc_res);
#else /* __L4_VERSION_X__ */
      ipc_error = l4_i386_ipc_wait(&th,
				   L4_IPC_SHORT_MSG, 
				   &r_kind, 
				   &r_kind,
				   L4_IPC_NEVER, 
				   &ipc_res);
#endif /* __L4_VERSION_X__ */
      if(ipc_error)
	/* debug */ ;
      
      /* ignore spurious wakeups */
      if(r_kind == WAKEUP_MAGIC)
	break;
      else
	{
	  /* debug */
	}
    }
}

#define RESTART_TIMEOUT L4_IPC_TIMEOUT(250,9,0,0,0,0) /* 1s */

extern inline void 
mem_lock_restart(unsigned lthread)
{
  int ipc_error;
  l4_msgdope_t ipc_res;
  dword_t r_kind;
  l4_threadid_t dest = l4_myself();
  
  dest.id.lthread = lthread;

  r_kind = WAKEUP_MAGIC;

#ifdef __L4_VERSION_X__
  ipc_error = l4_i386_ipc_send(dest,
			       L4_IPC_SHORT_MSG, 
			       r_kind, 
			       r_kind,
			       r_kind,
			       L4_IPC_NEVER, 
			       &ipc_res);
#else /* __L4_VERSION_X__ */
  ipc_error = l4_i386_ipc_send(dest,
			       L4_IPC_SHORT_MSG, 
			       r_kind, 
			       r_kind,
			       L4_IPC_NEVER, 
			       &ipc_res);
#endif /* __L4_VERSION_X__ */

  if (ipc_error)
    {
      /* debug */
    }
}

void __mem_lock(void)
{

  unsigned long flags;
  struct mem_lock_wait_queue wait;


  if(!mem_lock_initialized) mem_lock_init_waitqueue(&lq);

  for(;;) 
    {
      save_flags(flags);
      cli();
      if(mem_lock_lock == 0) 
	{
	  ++mem_lock_lock;
	  restore_flags(flags);
	  return;
	}
      wait.lthread = l4_myself().id.lthread;
      wait.next = NULL;
      
      __mem_lock_add_wait_queue(&lq, &wait);
      restore_flags(flags);
      
      mem_lock_suspend();
      
      save_flags(flags); cli();
      __mem_lock_remove_wait_queue(&lq, &wait);
      restore_flags(flags);
    }
}

void __mem_unlock(void)
{
  struct mem_lock_wait_queue *next;
  struct mem_lock_wait_queue *head;
  unsigned long flags;

  save_flags(flags); cli();
  mem_lock_lock = 0;
  
  if (!(&lq) || !(next = lq))
    {
      restore_flags(flags);
      return;
    }

  head = WAIT_QUEUE_HEAD(&lq);
  while (next != head) {
    unsigned p = next->lthread;
    next = next->next;

    if (p != 0)
      mem_lock_restart(p);

    if (!next) 
      goto bad;
  }

  restore_flags(flags);
  return;
bad:
#if 0
  BP("wait_queue is bad (eip = %p)\n",
	 __builtin_return_address(0));
  BP("        q = %p\n",q);
  BP("       *q = %p\n",*q);
#endif
  restore_flags(flags);
}

