/*
 * $Id: oskit_support.c,v 1.2 1999/05/13 15:22:11 yoonho Exp $
 */

/*****************************************************************************
 * liboskit_support/src/oskit_support.c                                      *
 * OSKit support functions                                                   *
 *****************************************************************************/

/* L4 includes */
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>

#include <l4/librmgr.h>

/* OSKit includes */
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include <flux/lmm.h>

#include <flux/machine/pc/direct_cons.h>
#include <flux/machine/pc/reset.h>

#define DEBUG_OSKIT_MALLOC 0

/*****************************************************************************
 * output character (printf)                                                 *
 *****************************************************************************/
int putchar(int __c)
{
  outchar(__c);

  if (__c == '\n')
    outchar('\r');

  return __c;
}

/*****************************************************************************
 * read character                                                            *
 *****************************************************************************/
int getchar(void)
{
  return direct_cons_getchar();
}

/*****************************************************************************
 * exit                                                                      *
 *****************************************************************************/
void _exit(int fd)
{
  char c;

  printf("\nReturn reboots, \"k\" enters L4 kernel debugger...\n");

  c = getchar();

  if (c == 'k' || c == 'K') 
    enter_kdebug("before reboot");

  pc_reset();
}

/*****************************************************************************
 * OSKit malloc                                                              *
 *****************************************************************************/

lmm_region_t malloc_region;
dword_t malloc_heap_end;
dword_t malloc_heap_add_size;
dword_t malloc_heap_cur_pages;
dword_t malloc_heap_max_pages;

extern l4_threadid_t rmgr_pager_id;

/*****************************************************************************
 * allocate new pages for malloc                                             *
 *****************************************************************************/
int __morecore_allocate(int page_count)
{
  int i,error;
  dword_t addr = malloc_heap_end;
  dword_t dw0,dw1,dummy;
  l4_msgdope_t result;

  if (!rmgr_init())
    {
      /* no ressource manager, can't allocate memory */
      printf("morecore: no ressource manager!\n");
      return 0;
    }

  if ((malloc_heap_cur_pages + page_count) > malloc_heap_max_pages)
    {
      printf("morecore: max heap size reached!\n");
      page_count = malloc_heap_max_pages - malloc_heap_cur_pages;
    }

  /* request pages from rmgr */
  for (i = 0; i < page_count; i++)
    {
#if DEBUG_OSKIT_MALLOC
      printf("morecore: addr = 0x%08x\n",addr);
#endif

#ifdef __L4_VERSION_X__
      error = l4_i386_ipc_call(rmgr_pager_id,
			       L4_IPC_SHORT_MSG, 0xfffffffc, 0, dummy,
			       L4_IPC_MAPMSG(addr, L4_LOG2_PAGESIZE), 
			       &dw0, &dw1, &dummy,
			       L4_IPC_NEVER, &result);
#else /* __L4_VERSION_X__ */
      error = l4_i386_ipc_call(rmgr_pager_id,
			       L4_IPC_SHORT_MSG, 0xfffffffc, 0,
			       L4_IPC_MAPMSG(addr, L4_LOG2_PAGESIZE), 
			       &dw0, &dw1,L4_IPC_NEVER, &result);
#endif /* __L4_VERSION_X__ */

      if (error)
	{
	  printf("morecore: IPC error 0x%02x!\n",error);
	}

      if (!l4_ipc_fpage_received(result))
	{
	  printf("morecore: page request failed!\n");
	  break;
	}
      
#if DEBUG_OSKIT_MALLOC
      printf("morecore: dw0 = 0x%08x, dw1 = 0x%08x, result = 0x%08x\n",
	     dw0,dw1,result.msgdope);
      enter_kdebug("-");
#endif

      addr += L4_PAGESIZE;
    }

#if DEBUG_OSKIT_MALLOC
  printf("morecore: allocated %d pages.\n",i);
  enter_kdebug("-");
#endif

  /* add memory to malloc region */
  lmm_add_free(&malloc_lmm,(void *)malloc_heap_end,i * L4_PAGESIZE);

  malloc_heap_end = addr;
  malloc_heap_cur_pages += i;

  return (i * L4_PAGESIZE);
}

int morecore(size_t size)
{
  int page_count; 

  page_count = (size > malloc_heap_add_size) ? 
    (size + L4_PAGESIZE - 1) >> L4_LOG2_PAGESIZE : 
    (malloc_heap_add_size) >> L4_LOG2_PAGESIZE;

#if DEBUG_OSKIT_MALLOC
  printf("morecore: size = %u (%d pages)\n",size,page_count);
#endif

  return __morecore_allocate(page_count);
}

/****************************************************************************
 * init OSKit malloc                                                        *
 ****************************************************************************/
void init_OSKit_malloc(dword_t heap_addr, dword_t init_size,
		       dword_t max_size, dword_t add_size)
{
  /* memory manager */
  lmm_init(&malloc_lmm);
  lmm_add_region(&malloc_lmm,&malloc_region,
		 (void*)heap_addr,heap_addr + max_size,0,0);

  /* request initial memory area */
  malloc_heap_end = heap_addr;
  malloc_heap_add_size = (add_size + L4_PAGESIZE - 1) & L4_PAGEMASK;
  malloc_heap_cur_pages = 0;
  malloc_heap_max_pages = (max_size + L4_PAGESIZE - 1) >> L4_LOG2_PAGESIZE;

  __morecore_allocate((init_size + L4_PAGESIZE - 1) >> L4_LOG2_PAGESIZE);
    
#if DEBUG_OSKIT_MALLOC
  printf("malloc pool at [0x%08x,0x%08x], add = %u, max = %u pages\n",
	 heap_addr,malloc_heap_end,malloc_heap_add_size,
	 malloc_heap_max_pages);
#endif
}

/*****************************************************************************
 * lock/unlock for OSKit malloc                                              *
 *****************************************************************************/
void __mem_unlock(void);
void __mem_lock(void);

void mem_unlock(void)
{
  __mem_unlock();
}

void mem_lock(void)
{
  __mem_lock();
}

