/* startup stuff */

/* it should be possible to throw away the text/data/bss of the object
   file resulting from this source -- so, we don't define here
   anything we could still use at a later time.  instead, globals are
   defined in globals.c */

/* OSKit stuff */

#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <flux/machine/multiboot.h>

/* L4 stuff */

#include <l4/compiler.h>
#include <l4/kernel.h>

/* local stuff */
#include "exec.h"
#include "globals.h"
#include "rmgr.h"
#include "init.h"

void startup(struct multiboot_info *mbi, unsigned int flag);

static exec_read_exec_func_t l4_exec_read_exec;
static vm_offset_t l4_low, l4_high;

static int jochen = 1;

static void
check_l4_version(char *start, char *end)
{
  char *p;

  jochen = no_pentium = 0;

  for (p = start; p < end; p++)
    {
      if (memcmp(p, "L4/486 \346-Kernel", 15) == 0)
	{
	  int m4_ok;

	  printf("RMGR: detected L4/486\n");
	  jochen = no_pentium = 1;

	  /* remove old 4M event */
	  m4_ok = 0;
	  for (p=start; p<end; p++)
	    {
	      if (memcmp(p, "\314\353\0024M", 5) == 0)
		{
		  p[0] = 0x90; /* replace "int $3" by "nop" */
		  m4_ok=1;
		  break;
		}
	    }
	  if (!m4_ok)
	    printf("RMGR: 4M sequence not found in L4/486 -- that's OK.\n");
	  return;
	}

      if (memcmp(p, "L4/Pentium \346-Kernel", 19) == 0)
	{
	  /* it's Jochen's Pentium version */
	  printf("RMGR: detected L4/Pentium\n");	  
	  jochen = 1;
	  return;
	}

      if (memcmp(p, "DD-L4/x86 microkernel", 21) == 0)
	{
	  /* it's the Dresden version */
	  printf("RMGR: detected new-style DD-L4\n");
	  return;
	}
    }

  printf("RMGR: could not detect version of L4 -- assuming Jochen.\n");
  jochen = 1;
  return;
}


void* _scan_kernel(const void* sequence, const int length, const int line);
void* _scan_kernel(const void* sequence, const int length, const int line)
{
  /* checker */
  unsigned char* find_p;
  void* found_p = 0;
  int found_count = 0;
  for(find_p = (unsigned char *) l4_low;
      find_p < (unsigned char *) l4_high;
      find_p++)
    /* check for some code */
    if (memcmp(find_p, sequence, length) == 0)
      { found_p = (void*) find_p; found_count++; };
  switch (found_count)
    {
    case 0:
      printf("RMGR: sequence in line %d not found\n", line);
      break;
    case 1:
      break;
    default:
      printf("RMGR: sequence in line %d found more than once\n", line);
      found_p = 0;
      break;
    };
  return found_p;
};
#define scan_kernel(sequence) \
_scan_kernel(##sequence, sizeof(sequence)-1, __LINE__)


/* started from crt0.S */
void
startup(struct multiboot_info *mbi, unsigned int flag)
{
  unsigned i;
  l4_kernel_info_t *l4i;
  static struct multiboot_info l4_mbi;
  exec_info_t exec_info;
  vm_offset_t sigma0_low = 0, sigma0_high = 0, sigma0_start = 0, 
    sigma0_stack = 0; /* zero-initialized to avoid spurious warnings */
  int have_sigma0 = 0;

  /* we're not an L4 task yet -- we're just a GRUB-booted kernel */

  assert(flag == MULTIBOOT_VALID); /* we need to be multiboot-booted */

  assert(mbi->flags & MULTIBOOT_MODS); /* we need at least two boot modules: */
  assert(mbi->mods_count >= 2);	/* 1 = L4 kernel, 2 = first user task */

  /* copy Multiboot data structures we still need to a safe place
     before playing with memory we don't own and starting L4 */

  /* rmgr.c defines variables holding copies */
  assert(mbi->mods_count <= MODS_MAX);
  memcpy(mb_mod, (void *) mbi->mods_addr, 
	 mbi->mods_count * sizeof (struct multiboot_module));
  mb_info = *mbi;
  mb_info.mods_addr = (vm_offset_t) mb_mod;

  /* shouldn't touch original Multiboot parameters after here */

  /* compute range for modules which we must not overwrite during
     booting the startup tasks */
  mod_exec_init(mb_mod, 0, mb_info.mods_count);

  /* lets look if we have a user-specified sigma0 */
  if ((mb_info.flags & MULTIBOOT_CMDLINE)
      && strstr((char *) mb_info.cmdline, "-sigma0"))
    {
      /* we define a small stack for sigma0 here -- it is used by L4
         for parameter passing to sigma0.  however, sigma0 must switch
         to its own private stack as soon as it has initialized itself
         because this memory region is later recycled in init.c */
      static char sigma0_init_stack[64]; /* XXX hardcoded */

      l4_low = 0xffffffff;
      l4_high = 0;

      assert(mbi->mods_count >= 3); /* need one more module -- sigma0 */

      printf("RMGR: loading %s\n", 
	     mb_mod[1].string ? (char *) mb_mod[1].string : "[SIGMA0]");

      exec_load(mod_exec_read, l4_exec_read_exec, 
		(void *)(mb_mod[1].mod_start), &exec_info);

      sigma0_low = l4_low;
      sigma0_high = l4_high;
      sigma0_start = exec_info.entry;
      sigma0_stack = 
	(vm_offset_t)(sigma0_init_stack + sizeof(sigma0_init_stack));

      have_sigma0 = 1;
      first_task_module++;
    }

  l4_low = 0xffffffff;
  l4_high = 0;

  /* the first module is the L4 microkernel; load it */
  exec_load(mod_exec_read, l4_exec_read_exec, 
	    (void *)(mb_mod[0].mod_start), &exec_info);
  
  /* XXX we don't look at the Multiboot header of the L4 kernel image */

  /* check for L4 version */
  check_l4_version((unsigned char *) l4_low, 
		   (unsigned char *) l4_low + (l4_high - l4_low));

  if (jochen)
    {
      /* assertion: the entry point is at the beginning of the kernel info
	 page */
      l4i = (l4_kernel_info_t *) exec_info.entry;
      assert(0x00001000 <= l4i->sigma0_eip && l4i->sigma0_eip < 0x000a0000);
      /* XXX can't think of more sanity checks that l4i really points to
	 the L4 kernel info page :-( */

      /* XXX undocumented */
      l4i->reserved1.low = l4i->reserved1.high = 0;
      for (i = 2; i < 4; i++)
	l4i->dedicated[i].low = l4i->dedicated[i].high = 0;

      if (mb_mod[0].string)	/* do we have a command line for L4? */
	{
	  /* more undocumented stuff: process kernel options */
	  if (strstr((char *) mb_mod[0].string, "hercules"))
	    {
	      unsigned char *kd_init = 
		(char *)((vm_offset_t)l4i - 0x1000 + l4i->init_default_kdebug);

	      assert(kd_init[0] == 0xb0 && kd_init[1] == 'a');
	      kd_init[1] = 'h';
	    }

	  if (strstr((char *) mb_mod[0].string, "nowait"))
	    {
	      unsigned char *kd_debug;
	      int nowait_ok = 0;

	      for(kd_debug = (unsigned char *) l4_low;
		  kd_debug < (unsigned char *) l4_low + (l4_high - l4_low);
		  kd_debug++)
		{
		  /* check for machine code 
		     "int $3; jmp 0f; .ascii "debug"; 0:" */
		  if (memcmp(kd_debug, "\314\353\005debug", 8) == 0)
		    {
		      kd_debug[0] = 0x90; /* replace "int $3" by "nop" */
		      nowait_ok = 1;
		      break;
		    }
		}
	      assert(nowait_ok);
	    }

	  /* all those serial port related options */
	  {
	    unsigned short portaddr = 0x02F8;	/* This is COM2, the default */
	    unsigned short divisor  = 1;	/* 115200 bps */
	    unsigned char* ratetable = 0;

	    if (strstr((char *) mb_mod[0].string, " -comport"))
	      {
		unsigned port;
		unsigned char* p_adr;
		
		const unsigned char port_adr_hi[] = {0x00, 0x3F, 0x2F, 0x3E, 0x2E};
		
		port = strtoul((strstr((char *) mb_mod[0].string, " -comport")
				+ strlen(" -comport")), NULL, 10);
		
		if (port)
		  {
		    p_adr = scan_kernel("\270\003\207\057\000");
		    if (p_adr)
		      {
			printf("RMGR: Setting COM%d as L4KD serial port\n", port);
			p_adr[3] = port_adr_hi[port];
			portaddr = (port_adr_hi[port] << 4) | 0x8;
		      }
		  }
		else
		  printf("RMGR: INVALID serial port\n");
	      }
	    
	    if (strstr((char *) mb_mod[0].string, " -comspeed"))
	      {
		unsigned rate;
	      
		rate = strtoul((strstr((char *) mb_mod[0].string, " -comspeed")
				+ strlen(" -comspeed")), NULL, 10);
	      
		if (rate)
		  {
		    /* check for rate table
		       dw 192, dw 96, dw 48, dw 24, ...
		       the 7th entry is used by default */
		    ratetable = scan_kernel("\300\000\140\000\060\000\030\000");
		    if (ratetable)
		      {
			printf("RMGR: Setting L4KD serial port rate to %d bps\n", rate);
			((short*) ratetable)[7] = 115200/rate;
			divisor = 115200/rate;
		      }
		  }
		else
		  printf("RMGR: INVALID serial rate\n");
	      }
	    
	    if (strstr((char *) mb_mod[0].string, " -VT"))
	      {
		
		if (!ratetable)
		  ratetable = scan_kernel("\300\000\140\000\060\000\030\000");
		
		if (ratetable)
		  {
		    printf("RMGR: Enabling serial terminal at %3x, %d bps\n", portaddr, 115200/divisor);
		    /* Uaah, the word before the rate table holds the port address for remote_outchar */
		    ((short*) ratetable)[-1] = portaddr;
		    
		    /* initialize the serial port */
		    asm("mov $0x83,%%al;add $3,%%dx;outb %%al,%%dx;"
			"sub $3,%%dx;mov %%bx,%%ax;outw %%ax,%%dx;add $3,%%dx;"
			"mov $0x03,%%al;outb %%al,%%dx;dec %%dx;inb %%dx,%%al;"
			"add $3,%%dx;inb %%dx,%%al;sub $3,%%dx;inb %%dx,%%al;"
			"sub $2,%%dx;inb %%dx,%%al;add $2,%%dx;inb %%dx,%%al;"
			"add $4,%%dx;inb %%dx,%%al;sub $4,%%dx;inb %%dx,%%al;"
			"dec %%dx;xor %%al,%%al;outb %%al,%%dx;mov $11,%%al;"
			"add $3,%%dx;outb %%al,%%dx\n"
			:
			/* no output */
			:
			"d" (portaddr),
			"b" (divisor)
			:
			"eax"
			);

		    if (strstr((char *) mb_mod[0].string, " -VT+"))
		      {
			unsigned char* p;
			if ((p = scan_kernel("\203\370\377\165\034\350")))
			  {
			    p[-6] = 1;
			    printf("RMGR: -VT+ mode enabled\n");
			  };
		      }
		  }
		
	      }
	    
	    if (strstr((char *) mb_mod[0].string, " -I+"))
	      {
		
		/* idea:
		   in init_timer the idt_gate is set up for the timer irq handler.
		   There we install the kdebug_timer_intr_handler.
		   in kdebug_timer... timer_int is called (saved location of timer_int) */
		
		unsigned long timer_int, kdebug_timer_intr_handler;
		void* real_location;
		void* tmp_location;
		unsigned long relocation;
		
		unsigned short* new;
		void* p;
		void* dest;
		
		/* find remote_info_port */
		tmp_location = scan_kernel("\300\000\140\000\060\000\030\000");
		if (tmp_location)
		  tmp_location -= 2;
		
		/* find reference to remote_info_port */
		real_location = scan_kernel("\146\201\342\377\017");
		if (real_location)
		  real_location = (void*) ((unsigned long) (((unsigned short*) real_location)[4]));
		
		relocation = tmp_location - real_location;
		
		/* find the place where the timer_int is set up */
		p = scan_kernel("\000\000"	/* this is a bit illegal !!!*/
				"\303"		/* c3		*/
				"\263\050"	/* b3 28	*/
				"\267\000"	/* b7 00	*/
				"\270"		/* b8		*/
				);
		if (p)
		  {
		    timer_int = ((unsigned long*)p)[2];
		    
		    /* find kdebug_timer_intr_handler */
		    dest = scan_kernel("\152\200\140\036\152\010"
				       "\037\314\074\014\074\033");
		    /* no more comments */
		    if (dest)
		      {
			kdebug_timer_intr_handler = (((unsigned long) dest) - relocation);
			((unsigned long*)p)[2] = kdebug_timer_intr_handler;
			dest = scan_kernel("\015\260\055\314"
					   );
			if (dest)
			  {
			    new = (unsigned short*) ((((unsigned short*)dest)[-5]) + relocation);
			    *new = timer_int;
			    printf("RMGR: I+ enabled\n");
			  };
		      };
		  };
	      };
	  }

	  /*
	    This option disables the patch that eliminates the miscalculation
	    of string offsets in messages containing indirect strings

	    The patch changes calculation of the first string's offset in the
	    message buffer (in ipc_strings).
	    This offset was calculated

		addr(msgbuffer) + offset(dword_2) + 4*number_of_dwords

	    this is changed to
		addr(msgbuffer) + offset(dword_0) + 4*number_of_dwords
					 ^^^^^^^
	    */
	  if (strstr((char *) mb_mod[0].string, " -disablestringipcpatch"))
	    printf("RMGR: string IPC patch disabled\n");
	  else
	    {
	      unsigned char* patchme;
	      patchme = scan_kernel("\024\213\126\004\301\352\015\215\164\226\024");
	      if (patchme)
		{
		  patchme[0] = 0x0c;
		  patchme[10] = 0x0c;
		  printf("RMGR: string IPC patch applied\n");
		}
	      else
		printf("RMGR: string IPC patch not applied - signature not found\n");
	    };
	  
	  /* heavy undocumented stuff, we simply patch the kernel to get
	     access to irq0 and to be able to use the high resolution timer */
	  if (strstr((char *) mb_mod[0].string, "irq0"))
	    {
	      unsigned char *kd_debug;
	      int irq0_ok = 0;

	      for(kd_debug = (unsigned char *) l4_low;
		  kd_debug < (unsigned char *) l4_low + (l4_high - l4_low);
		  kd_debug++)
		{
		  /* check for byte sequence 
		     0xb8 0x01 0x01 0x00 0x00 0xe8 ?? ?? ?? ?? 0x61 0xc3 */
		  if (memcmp(kd_debug, "\270\001\001\000\000\350", 6) == 0)
		    {
		      if ((kd_debug[10] == 0141) && (kd_debug[11] == 0303)) 
			{
			  kd_debug[1] = '\0'; /* enable irq0 */
			  irq0_ok = 1;
			  break;
			}
		    }
		}
	      assert(irq0_ok);
	    }

	  /* boot L4 in real-mode (for old versions) */
	  if (strstr((char *) mb_mod[0].string, "boothack"))
	    {
	      /* find "cmpl $0x2badb002,%eax" */
	      vm_offset_t dest 
		= (vm_offset_t) scan_kernel("\075\002\260\255\053");
	      
	      if (dest)
		exec_info.entry = dest;
	    }
	}
    }
  else				/* not Jochen version of L4 */
    {
      static char cmdline[256];
      int l;

      /* use an info page prototype allocated from our address space;
         this will be copied later into the kernel's address space */
      static char proto[L4_PAGESIZE];

      memset(proto, 0, L4_PAGESIZE);
      l4i = (l4_kernel_info_t *) proto;

      /* append address of kernel info prototype to kernel's command
         line */
      if (mb_mod[0].string)
	{
	  l = strlen((char *) mb_mod[0].string);
	  /* make sure it fits into cmdline[] */
	  assert(l < sizeof(cmdline) - 40);
	  strcpy(cmdline, (char *) mb_mod[0].string);
	}
      else
	l = 0;

      sprintf(cmdline + l, " proto=0x%x", (vm_offset_t) proto);
      mb_mod[0].string = (vm_offset_t) cmdline;
    }

  /* setup the L4 kernel info page before booting the L4 microkernel:
     patch ourselves into the booter task addresses */
  l4i->root_esp = (dword_t) &_stack;
  l4i->root_eip = (dword_t) init;
  l4i->root_memory.low = l4i->dedicated[0].low = (dword_t) &__crt_dummy__;
  l4i->root_memory.high = l4i->dedicated[0].high = (dword_t) &_end;

  /* set up sigma0 info */
  if (have_sigma0)
    {
      l4i->sigma0_eip = sigma0_start;
      l4i->sigma0_memory.low = sigma0_low;
      l4i->sigma0_memory.high = sigma0_high;

      /* XXX UGLY HACK: Jochen's kernel can't pass args on a sigma0
         stack not within the L4 kernel itself -- therefore we use the
         kernel info page itself as the stack for sigma0.  the field
         we use is the task descriptor for the unused "ktest2" task */
      l4i->sigma0_esp = jochen ? 0x1060 : sigma0_stack;
    }

  /* reserve memory for the modules */
  mod_exec_init(mb_mod, 1, mb_info.mods_count);
  l4i->dedicated[1].low = mod_range_start;
  l4i->dedicated[1].high = mod_range_end;

  /* now boot the L4 microkernel in a multiboot-compliant way */
  
  l4_mbi = mb_info;
  assert(l4_mbi.flags & MULTIBOOT_MEMORY);
  l4_mbi.flags = MULTIBOOT_MEMORY;
  if (mb_mod[0].string)
    {
      l4_mbi.cmdline = mb_mod[0].string;
      l4_mbi.flags |= MULTIBOOT_CMDLINE;
    }
  
  printf("RMGR: starting %s\n", 
	 mb_mod[0].string ? (char *) mb_mod[0].string : "[L4]");

  asm volatile
    ("pushl $" SYMBOL_NAME_STR(exit) "
      pushl %2
      ret"
     :				/* no output */
     : "a" (MULTIBOOT_VALID), "b" (&l4_mbi), "r" (exec_info.entry));

  /*NORETURN*/

}


static int l4_exec_read_exec(void *handle,
			     vm_offset_t file_ofs, vm_size_t file_size,
			     vm_offset_t mem_addr, vm_size_t mem_size,
			     exec_sectype_t section_type)
{
  if (! (section_type & EXEC_SECTYPE_ALLOC))
    return 0;
  
  if (mem_addr < l4_low) l4_low = mem_addr;
  if (mem_addr + mem_size > l4_high) l4_high = mem_addr + mem_size;

  return mod_exec_read_exec(handle, file_ofs, file_size, 
			    mem_addr, mem_size, section_type);
}

