/* startup stuff */

/* it should be possible to throw away the text/data/bss of the object
   file resulting from this source -- so, we don't define here
   anything we could still use at a later time.  instead, globals are
   defined in globals.c */

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <flux/machine/multiboot.h>

#include <l4/syscalls.h>
#include <l4/ipc.h>

#include "config.h"
#include "memmap.h"
#include "globals.h"
#include "exec.h"
#include "rmgr.h"
#include "cfg.h"

#include "init.h"

struct multiboot_module mb_mod[MODS_MAX];

static void init_globals(void);
static void init_memmap(void);
static void init_irq(void);
static void init_task(void);
static void init_small(void);
static void configure(void);
static void start_tasks(void);

static void setup_task(l4_threadid_t t);
static int overlaps_myself(vm_offset_t begin, vm_offset_t end);
static int alloc_exec_read_exec(void *handle,
				vm_offset_t file_ofs, vm_size_t file_size,
				vm_offset_t mem_addr, vm_size_t mem_size,
				exec_sectype_t section_type);
static int alloc_exec_read(void *handle, vm_offset_t file_ofs,
			   void *buf, vm_size_t size,
			   vm_size_t *out_actual);
static int set_small(l4_threadid_t t, int num);
static int set_prio(l4_threadid_t t, int num);


int boot_errors = 0, boot_wait = 0;
bootquota_t bootquota[TASK_MAX];

/* started as the L4 booter task through startup() in startup.c */
void init(void)
{
  vm_offset_t address;

  printf("RMGR: Hi there!\n");

  init_globals();

  init_memmap();

  init_irq();

  init_task();

  init_small();

  configure();

  setup_task(myself);
  setup_task(my_pager);

  /* start the tasks loaded as modules */
  start_tasks();

  /* add the memory used by this module and not any longer required
     (the ".init" section) to the free memory pool */
  for (address = (vm_offset_t) &__crt_dummy__ & L4_PAGEMASK;
       address < ((vm_offset_t) &_stext & L4_PAGEMASK);
       address += L4_PAGESIZE)
    {
      check(memmap_free_page(address, O_RESERVED));
    }

  if (boot_errors || boot_wait)
    {
      char c;

      if (boot_errors)
	printf("RMGR: WARNING: there were %d boot errors -- continue?\n"
	       "               Return continues, Esc panics, \"k\" enters L4 kernel debugger...\n",
	       boot_errors);
      else
	printf("RMGR: bootwait option was set -- please press a key\n"
	       "      Return continues, Esc panics, \"k\" enters L4 kernel debugger...\n");
      c = getchar();

      if (c == 27)
	panic("boot error");
      else if (c == 'k' || c == 'K')
	enter_kdebug("boot error");
    }


  /* now start the resource manager */
  rmgr();
}

/* support functions for init() */

static void configure(void)
{
  char *config, *config_end;

  cfg_init();

  /* find the config data */
  if (! (mb_info.flags & MULTIBOOT_CMDLINE))
    return;

  config = strchr((char *) mb_info.cmdline, ' ');
  if (! config)
    return;

  config++;			/* skip ' ' character */
  if (!strstr(config, "-configfile"))
    {
      /* not using a configfile -- use the cmdline */
      config_end = config + strlen(config);

      /* skip options */
      while (config < config_end)
	{
	  if (*config == ' ')	/* skip spaces */
	    {
	      config++;
	      continue;
	    }

	  if (*config != '-')	/* end of options found */
	    break;

	  config = strchr(config, ' ');	/* find end of this option */
	  if (!config)
	    return;
	}
    }
  else
    {
      /* the configfile is in the first module... */

      /* make sure the config file is not the last module */
      check(mb_info.mods_count >= first_task_module + 2);

      config = (char *) mb_mod[first_task_module].mod_start;
      config_end = (char *) mb_mod[first_task_module].mod_end;

      /* is it really a config file? */
      if (strncmp(config, "#!rmgr", 6) != 0)
	{
	  printf("RMGR: ERROR: 2nd module is not a config file "
		 "(doesn't start with \"#!rmgr\"\n");
	  boot_errors++;
	  return;
	}

      first_task_module++;
    }

  cfg_setup_input(config, config_end);

  if (cfg_parse() != 0)
    boot_errors++;

  /* a few post-parsing actions follow... */
  if (small_space_size)
    {
      vm_offset_t s = small_space_size;

      small_space_size /= 0x400000; /* divide by 4MB */
      if (s > small_space_size * 0x400000 /* did we delete any size bits? */
	  || (small_space_size & (small_space_size - 1))) /* > 1 bit set? */
	{
	  /* use next largest legal size */
	  int count;
	  for (count = 0; small_space_size; count++)
	    {
	      small_space_size >>= 1;
	    }

	  small_space_size = 1L << count;
	}
      
      if (small_space_size > 64)
	{
	  printf("RMGR: ERROR: small_space_size 0x%x too large\n", s);
	  small_space_size = 0;
	  boot_errors++;
	}

      else
	{
	  unsigned i;

	  for (i = 1; i < 128/small_space_size; i++)
	    check(small_free(i, O_RESERVED));

	  printf("RMGR: configured small_space_size = %d MB\n",
		 small_space_size * 4);
	}
    }
}

static void setup_task(l4_threadid_t t)
{
  unsigned me = t.id.task;
  bootquota_t *b = & bootquota[me];

  if (b->small_space != 0xff)
    set_small(t, b->small_space);

  set_prio(t, b->prio);
}

static void init_globals(void)
{
  dword_t dummy;

  /* set some globals */

  /* set myself (my thread id) */
  myself = l4_myself();

  /* set my_pager */
  my_preempter = my_pager = L4_INVALID_ID;
  l4_thread_ex_regs(myself, (dword_t) -1, (dword_t) -1, 
		    &my_preempter, &my_pager, &dummy, &dummy, &dummy);

  /* set mem size */
  mem_upper = mb_info.mem_upper;
  mem_lower = mb_info.mem_lower & ~3; /* round down to next 4k boundary */

  /* more global initialization stuff */
  small_space_size = 0;

  debug = verbose = debug_log_mask = debug_log_types = 0;

  if (no_pentium == 1)
    printf("RMGR: running on L4/486\n");
  else if (no_pentium == 0)
    printf("RMGR: running on L4/Pentium\n");
  else {
    printf("RMGR: unknown L4 version, assuming L4/486\n");
    no_pentium = 1;
  }

  /* check for -nopentium option */
  if (mb_info.flags & MULTIBOOT_CMDLINE)
    {
      if (strstr((char *) mb_info.cmdline, "-nopentium"))
	{
	  no_pentium=1;
	  printf("no l4/pentium\n");
	}
    }
  quota_init();
}

static void init_irq(void)
{
  int err, i;
  unsigned *sp;
  dword_t code, dummy;
  l4_msgdope_t result;
  l4_threadid_t t, preempter, pager;

  /* initialize to "reserved" */
  irq_init();

  printf("RMGR: attached irqs = [ ");

  /* start a thread for each irq */
  t = myself;
  
  for (i = 0; i < IRQ_MAX; i++)
    {
      t.id.lthread = LTHREAD_NO_IRQ(i);
      sp = (unsigned *)(__irq_stacks + (i + 1) * __IRQ_STACKSIZE);

      *--sp = i;		/* pass irq number as argument to thr func */
      *--sp = 0;		/* faked return address */
      preempter = my_preempter; pager = my_pager;
      l4_thread_ex_regs(t, (dword_t) __irq_thread, (dword_t) sp,
                        &preempter, &pager, &dummy, &dummy, &dummy);

      /* handshake and status check */
      err = l4_i386_ipc_receive(t, L4_IPC_SHORT_MSG, &code, &dummy,
                                L4_IPC_NEVER, &result);

      assert(err == 0);
      
      if (i == 2)
        {
          /* enable IRQ2 */
          __asm__("inb $0x21,%al\n"\
                  "jmp 1f\n"\
                  "1:\tjmp 1f\n"\
                  "1:\tandb $0xfb,%al\n"\
                  "outb %al,$0x21\n");
        }
      
      if (code == 0)
        {
          if (i != 2) 
	    irq_free(i, O_RESERVED); /* free irq, except if irq2 */
          printf("%x ", i);
        }
          else
	    printf("<!%x> ", i);
    }
  printf("]\n");
}

static void init_task(void)
{
  unsigned i;

  task_init();

  for (i = myself.id.task + 1; i < TASK_MAX; i++)
    __task[i] = O_FREE;
}

static void init_small(void)
{
  small_init();
}


static void init_memmap(void)
{
  l4_msgdope_t result;
  l4_snd_fpage_t sfpage;
  vm_offset_t address, a;
  vm_size_t free_size = 0, reserved_size = 0;
  int error;
  int ignore_high_pages = 0;
  
  /* initialize to "reserved" */
  memmap_init();

  mem_high = 0;

  if (!no_pentium) 
    {
      /* get as much super (4MB) pages as possible, starting with
     superpage 1 (0x400000) */
      for (address = L4_SUPERPAGESIZE; address < MEM_MAX; 
	   address += L4_SUPERPAGESIZE)
	{
	  error = l4_i386_ipc_call(my_pager, L4_IPC_SHORT_MSG,
				   address | 1, L4_LOG2_SUPERPAGESIZE << 2,
				   L4_IPC_MAPMSG(0, L4_WHOLE_ADDRESS_SPACE),
				   &sfpage.snd_base, &sfpage.fpage.fpage,
				   L4_IPC_NEVER, &result);
	  /* XXX should i check for errors? */

	  if (! (l4_ipc_fpage_received(result) 
		 && l4_ipc_is_fpage_writable(sfpage.fpage)
		 && sfpage.fpage.fp.size == L4_LOG2_SUPERPAGESIZE))
	    {
	      vm_size_t size;

	      if (!l4_ipc_fpage_received(result))
		break;

	      size = 1L << sfpage.fpage.fp.size;
	      assert(size == (size & L4_PAGEMASK));

	      for (; size > 0; size -= L4_PAGESIZE, address += L4_PAGESIZE)
		{
		  free_size += L4_PAGESIZE;
		  memmap_free_page(address, O_RESERVED);
		}

	      break;
	    }

	  /* kewl, we've received an super awsum page */
	  free_size += L4_SUPERPAGESIZE;

	  for (a = address; a < address + L4_SUPERPAGESIZE; a += L4_PAGESIZE)
	    {
	      memmap_free_page(a, O_RESERVED);
	    }

	  assert(memmap_owner_superpage(address) == O_FREE);
	}
    }
  /* now add all memory we can get from L4 to the memory map as free
     memory */
  for (;;)
    {
      error = l4_i386_ipc_call(my_pager, L4_IPC_SHORT_MSG,
			       0xfffffffc, 0,
			       L4_IPC_MAPMSG(0, L4_WHOLE_ADDRESS_SPACE),
			       &sfpage.snd_base, &sfpage.fpage.fpage,
			       L4_IPC_NEVER, &result);
      /* XXX should i check for errors? */

      if (! sfpage.fpage.fpage)
	break;			/* out of memory */

      if (sfpage.snd_base >= MEM_MAX)
	{
	  if (!ignore_high_pages)
	    {
	      ignore_high_pages = 1;
	      printf("RMGR: WARNING: all memory above %ld MB is wasted!\n",
		     MEM_MAX/(1024*1024));
	    }
	  continue;
	}

      if (mem_high < sfpage.snd_base) 
	mem_high = sfpage.snd_base;

      if (memmap_owner_page(sfpage.snd_base) == O_FREE)
	{
	  printf("RMGR: WARNING: we've been given page 0x%x twice!\n",
		 sfpage.snd_base);
	  continue;
	}

      free_size += L4_PAGESIZE;
      memmap_free_page(sfpage.snd_base, O_RESERVED);
    }

  assert(mem_high > 0x200000);	/* 2MB is the minimum for any useful work */

  /* page in memory we own and we reserved before booting; we need to
     do this explicitly as the L4 kernel won't hand it out voluntarily
     (see L4 rerefence manual, sec. 2.7).
     don't care for overlaps with our own area here because we will
     reserve our memory below */
  for (address = mod_range_start & L4_PAGEMASK;
       address <= mod_range_end;
       address += L4_PAGESIZE)
    {
      /* force write page fault */
      /* * (volatile int *) address |= 0;  */
      asm volatile ("orl $0, (%0)" 
		    : /* nothing out */
		    : "r" (address)
		    );
      if (memmap_owner_page(address) != O_FREE)
	{
	  memmap_free_page(address, O_RESERVED);
	  free_size += L4_PAGESIZE;
	}
    }

  /* page in adapter space explicitly.  the adapter space will be
     marked "reserved" and special-cased in the pager() in memmap.c */
  for (address = mem_lower * 1024L;
       address < 0x100000L;
       address +=  L4_PAGESIZE)
    {
      error = l4_i386_ipc_call(my_pager, L4_IPC_SHORT_MSG,
			       address, 0,
			       L4_IPC_MAPMSG(0, L4_WHOLE_ADDRESS_SPACE),
			       &sfpage.snd_base, &sfpage.fpage.fpage,
			       L4_IPC_NEVER, &result);
      /* XXX should i check for errors? */
    }

  /* reserve memory for ourselves; page in our own memory en-passant */
  for (address = ((vm_offset_t) (&__crt_dummy__)) & L4_PAGEMASK;
       address <= (vm_offset_t) &_end;
       address += L4_PAGESIZE)
    {
      memmap_alloc_page(address, O_RESERVED);
      reserved_size += L4_PAGESIZE;

      /* force write page fault */
      /* * (volatile int *) address |= 0;  */
      asm volatile ("orl $0, (%0)" 
		    : /* nothing out */
		    : "r" (address)
		    );
    }

  /* reserve address 0x1000 for the kernel info page */
  /* we assume the L4 kernel is loaded to 0x1000, creating a virtual
     memory area which doesn't contain mapped RAM so we can use it as
     a scratch area.  this is at least consistent with the L4
     reference manual, appendix A.  however, we flush the page anyway
     before attempting to map something in there, just in case. */
  l4_info = (l4_kernel_info_t *) 0x1000;
  l4_fpage_unmap(l4_fpage((dword_t) l4_info, L4_LOG2_PAGESIZE, 0, 0),
		 L4_FP_FLUSH_PAGE|L4_FP_ALL_SPACES);
  error = l4_i386_ipc_call(my_pager, L4_IPC_SHORT_MSG,
			   1, 1,
			   L4_IPC_MAPMSG((dword_t) l4_info, L4_LOG2_PAGESIZE),
			   &sfpage.snd_base, &sfpage.fpage.fpage,
			   L4_IPC_NEVER, &result);
  if (error)
    panic("can't map kernel info page: IPC error 0x%x\n", error);
  memmap_alloc_page((dword_t) l4_info, O_RESERVED);
  reserved_size += L4_PAGESIZE;

  assert(l4_info->magic == L4_KERNEL_INFO_MAGIC);

  /* the non-RAM high-memory superpages are all free for allocation */
  for (address = 0x8000; address < 0x10000;
       address += L4_SUPERPAGESIZE/0x10000) /* scaled by 0x10000**-1 to
					       prevent overflow */
    {
      memmap_free_superpage(address * 0x10000, O_RESERVED);
    }


  printf("RMGR: total RAM size = %6d KB (reported by bootloader)\n"
	 "              received %6ld KB RAM from sigma0\n"
	 "                       %6ld KB reserved for RMGR\n", 
	 mb_info.mem_upper + mb_info.mem_lower,
	 free_size/1024L, reserved_size/1024L);

  return;
}

struct exec_task
{
  void *mod_start;
  unsigned task_no;
};

#define alloc_from_page(pageptr, size)				\
({								\
  vm_offset_t ret;						\
  vm_offset_t new_pageptr = ((pageptr) + (size) + 3) & ~3;	\
  if (((pageptr) & L4_PAGEMASK) != (new_pageptr & L4_PAGEMASK))	\
    ret = 0;							\
  else								\
    {								\
      ret = (pageptr);						\
      (pageptr) = new_pageptr;					\
    }								\
  ret;								\
})

static void start_tasks(void)
{
  unsigned i;
  l4_threadid_t t;
  exec_info_t exec_info;
  int exec_ret;
  struct exec_task e;
  char *name;
  vm_offset_t address;
  struct multiboot_info *mbi;
  dword_t *sp, entry;

  t = myself;
/*   t.id.nest++; */
  t.id.chief = t.id.task;
  t.id.task++;
  
  for (i = first_task_module; i < mb_info.mods_count; i++)
    {
      check(task_alloc(t.id.task, myself.id.task, myself.id.version_low));

      mod_exec_init(mb_mod, i, mb_info.mods_count);

      e.mod_start = (void *)(mb_mod[i].mod_start);
      e.task_no = t.id.task;

      name = mb_mod[i].string ? (char *)  mb_mod[i].string : "[unknown]";

      /* register name and id for the RMGR_TASK_GET_ID function */
      mb_mod_ids[i] = t;
      strncpy(mb_mod_names[i], name, MOD_NAME_MAX);
      (mb_mod_names[i])[MOD_NAME_MAX - 1] = 0; /* terminate string */

      printf("RMGR: loading task %s [ ", name);
      exec_ret = exec_load(alloc_exec_read, alloc_exec_read_exec, 
			   &e, &exec_info);
      printf("]\n");
      
      if (exec_ret)
	{
	  printf("RMGR: ERROR: exec_load() failed with code %d\n", exec_ret);
	  boot_errors++;
	  continue;
	}

      /* pass multiboot info to new task in extra page */
      for (address = (mod_range_end + L4_PAGESIZE - 1) & L4_PAGEMASK;
	   address < mem_high; address += L4_PAGESIZE)
	{
	  if (memmap_owner_page(address) != O_FREE)
	    continue;
	  if (memmap_alloc_page(address, t.id.task))
	    break;
	}

      mbi = (struct multiboot_info *) 
	alloc_from_page(address, sizeof(struct multiboot_info));

      *mbi = mb_info;		/* copy our multiboot info structure */
      mbi->flags &= MULTIBOOT_MEMORY|MULTIBOOT_BOOT_DEVICE;
      if (mb_mod[i].string)
	{
	  vm_size_t len = strlen(name) + 1;
	  char *string;

	  if (! (string = (char *) alloc_from_page(address, len)))
	    {
	      printf("RMGR: ERROR: command line too long -- truncating\n");
	      boot_errors++;

	      len = L4_PAGESIZE - 0x120;
	      check(string = (char *) alloc_from_page(address, len));
	    }
	  
	  mbi->flags |= MULTIBOOT_CMDLINE;
	  mbi->cmdline = (vm_offset_t) string;
	  strncpy(string, name, len);
	  *(string + len - 1) = 0; /* zero-terminate string */
	}

      /* before we start allocating extra mem, first allocate stack space */
      check((sp = (dword_t *) alloc_from_page(address, 0x100)));

      sp = (dword_t *) (0x100 + (vm_offset_t) sp);

      /* if any modules have been passed on to this task, copy module info */
      if (bootquota[t.id.task].mods)
	{
	  struct multiboot_module *m;
	  unsigned mod_i = i + 1;
	  unsigned j = bootquota[t.id.task].mods;

	  i += j;
	  if (i >= mb_info.mods_count)
	    {
	      printf("RMGR: ERROR: task configured to use %d modules, "
		     "but there are only %d modules\n"
		     "             left for loading -- not starting %s\n",
		     j, mb_info.mods_count - (i - j) - 1, name);
	      boot_errors++;
	      break;
	    }

	  if (! (m = (struct multiboot_module *) 
		     alloc_from_page(address, 
				     j * sizeof(struct multiboot_module))))
	    {
	      printf("RMGR: ERROR: out of boot-info memory space --\n"
		     "             won't start %s",
		     name);
	      boot_errors++;
	      continue;
	    }
	  
	  mbi->mods_addr = (vm_offset_t) m;
	  mbi->mods_count = j;
	    
	  for (; j--; m++, mod_i++)
	    {
	      vm_offset_t mod_mem;
	      *m = mb_mod[mod_i];

	      if (m->string)
		{
		  vm_size_t len = strlen((char *) (m->string)) + 1;
		  char *string = (char *) alloc_from_page(address, len);

		  if (! string)
		    {
		      printf("RMGR: ERROR: not enough room for module string"
			     " --\n"
			     "      not passing string \"%s\"\n", 
			     (char *) (m->string));
		      boot_errors++;

		      m->string = 0;
		      continue;
		    }

		  strcpy(string, (char *) (m->string));
		  m->string = (vm_offset_t) string;
		}

	      for (mod_mem = m->mod_start & L4_PAGEMASK;
		   mod_mem < m->mod_end;
		   mod_mem += L4_PAGESIZE)
		check(memmap_alloc_page(mod_mem, t.id.task));

	      printf("RMGR: passing module %s [ 0x%x-0x%x ]\n", 
		     m->string ? (char *) (m->string) : "[unknown]",
		     m->mod_start, m->mod_end);
	    }

	  mbi->flags |= MULTIBOOT_MODS;
	}
     
      /* copy task_trampoline PIC code to new task's stack */
      sp -= 1 + ((vm_offset_t) task_trampoline_end
		 - (vm_offset_t) task_trampoline) / sizeof(*sp);
      memcpy(sp, task_trampoline, 
	     (vm_offset_t) task_trampoline_end 
	     - (vm_offset_t) task_trampoline);
      entry = (dword_t) sp;

      /* setup stack for task_trampoline() */
      *--sp = (dword_t) mbi;
      *--sp = exec_info.entry;
      *--sp = 0;		/* fake return address */

      printf("RMGR: starting task %s\n"
	     "      at entry 0x%x via trampoline page code 0x%x\n", 
	     name, exec_info.entry, entry);
      
      t = l4_task_new(t, bootquota[t.id.task].mcp, 
		      (dword_t) sp, entry, myself);
      
      setup_task(t);

      t.id.task++;
    }
}

/* support for loading tasks from boot modules and allocating memory
   for them */

static int alloc_exec_read_exec(void *handle,
				vm_offset_t file_ofs, vm_size_t file_size,
				vm_offset_t mem_addr, vm_size_t mem_size,
				exec_sectype_t section_type)
{
  vm_offset_t address;
  struct exec_task *e = (struct exec_task *) handle;

  if (! (section_type & EXEC_SECTYPE_ALLOC))
    return 0;
  
  assert(e->task_no <= O_MAX);

  for (address = mem_addr & L4_PAGEMASK;
       address <= mem_addr + mem_size;
       address += L4_PAGESIZE)
    {
      check(memmap_alloc_page(address, e->task_no));
    }

  printf("0x%x-0x%x ", mem_addr, mem_addr + mem_size);
  
  return mod_exec_read_exec(e->mod_start, file_ofs, file_size, 
			    mem_addr, mem_size, section_type);
}

static int alloc_exec_read(void *handle, vm_offset_t file_ofs,
			   void *buf, vm_size_t size,
			   vm_size_t *out_actual)
{
  struct exec_task *e = (struct exec_task *) handle;

  return mod_exec_read(e->mod_start, file_ofs, buf, size, out_actual);
}

/* module loading support -- also used by startup.c to load the L4
   kernel -- therefore exported from here */

vm_offset_t mod_range_start, mod_range_end;

void mod_exec_init(struct multiboot_module *mod, 
		   unsigned mod_first, unsigned mod_count)
{
  unsigned i;

  assert(mod_first < mod_count);

  /* compute range for modules which we must not overwrite during
     booting the startup tasks */
  mod_range_start = mod[mod_first].mod_start;
  mod_range_end = mod[mod_first].mod_end;
  for (i = mod_first + 1; i < mod_count; i++)
    {
      if (mod[i].mod_start < mod_range_start)
	mod_range_start = mod[i].mod_start;
      if (mod[i].mod_end > mod_range_end)
	mod_range_end = mod[i].mod_end;
    }
}

static int
overlaps_myself(vm_offset_t begin, vm_offset_t end)
{
  return (begin <= (vm_offset_t) &__crt_dummy__ 
	  && end >= (vm_offset_t) &__crt_dummy__)
    || (begin <= (vm_offset_t) &_end
	&& end >= (vm_offset_t) &_end)
    || (begin <= mod_range_start
	  && end >= mod_range_start)
    || (begin <= mod_range_end
	  && end >= mod_range_end);
}

int mod_exec_read(void *handle, vm_offset_t file_ofs,
		  void *buf, vm_size_t size,
		  vm_size_t *out_actual)
{
  check(!overlaps_myself((vm_offset_t) buf,
			 (vm_offset_t) buf + size));

  return simple_exec_read(handle, file_ofs, buf, size, out_actual);
}

int mod_exec_read_exec(void *handle,
		       vm_offset_t file_ofs, vm_size_t file_size,
		       vm_offset_t mem_addr, vm_size_t mem_size,
			  exec_sectype_t section_type)
{
  if (! (section_type & EXEC_SECTYPE_ALLOC))
    return 0;

  check(! overlaps_myself(mem_addr, mem_addr + mem_size));

  return simple_exec_read_exec(handle, file_ofs, file_size, mem_addr, mem_size,
			       section_type);
}

static int set_small(l4_threadid_t t, int num)
{
  if (num < SMALL_MAX && small_alloc(num, t.id.task))
    {
      l4_sched_param_t sched;
      l4_threadid_t s;

      s = L4_INVALID_ID;
      l4_thread_schedule(t, L4_INVALID_SCHED_PARAM, &s, &s, &sched);
      sched.sp.small = small_space_size 
	| (num * small_space_size * 2);
      s = L4_INVALID_ID;
      l4_thread_schedule(t, sched, &s, &s, &sched);
    }
  else
    {
      printf("RMGR: ERROR: can't set small space %d for task 0x%x\n",
	     num, t.id.task);
      boot_errors++;

      return 0;
    }

  return 1;
}

static int set_prio(l4_threadid_t t, int num)
{
  l4_sched_param_t sched;
  l4_threadid_t s;
  
  s = L4_INVALID_ID;
  l4_thread_schedule(t, L4_INVALID_SCHED_PARAM, &s, &s, &sched);
  sched.sp.prio = num;
  s = L4_INVALID_ID;
  l4_thread_schedule(t, sched, &s, &s, &sched);
  
  return 1;
}

