#include <services/lib/cow.h>
#include <services/lib/utils.h>
#include <services/client/rm.h>
#include <services/client/dm_anon.h>
#include <services/client/genericdm.h>
#include <services/lib/rm.h>
#include <services/lib/nameservice.h>

typedef struct origpage_desc_s {
  struct copy_desc_s *copies;
  void *orig_page;
  struct origpage_desc_s *prev, *next;
} origpage_desc_t;

typedef struct copy_desc_s {
  copy_id_t copy_id;
  lock_t pagedata_lock;
  void *copy_page;
  struct copy_desc_s *next;
} copy_desc_t;

origpage_desc_t *opdesc_head=NULL, *opdesc_tail=NULL;
#define __inline__

typedef int page_t[1024];
#ifndef COW_PAGE_POOL_SIZE
#define COW_PAGE_POOL_SIZE 1024
#endif
page_t *page_pool;
origpage_desc_t origpagedesc_pool[COW_PAGE_POOL_SIZE];
copy_desc_t copypagedesc_pool[COW_PAGE_POOL_SIZE];
int pagepool_freelist[COW_PAGE_POOL_SIZE];
int page_firstfree;
origpage_desc_t *origpagedesc_firstfree = origpagedesc_pool;
copy_desc_t *copypagedesc_firstfree = copypagedesc_pool;

int initialised_level = 0;
lock_t cow_lock = 0;

static __inline__ origpage_desc_t *
alloc_origpagedesc(void)
{
  origpage_desc_t *op;
  if (origpagedesc_firstfree == NULL) {
    enter_kdebug ("cow.c: resources exhausted");
    halt();
  }
  op = origpagedesc_firstfree;
  origpagedesc_firstfree = *(origpage_desc_t **)op;
  return op;
}

static __inline__ void
dealloc_origpagedesc(origpage_desc_t *condemned_origpagedesc)
{
  origpage_desc_t *op;
  *(origpage_desc_t **)condemned_origpagedesc = origpagedesc_firstfree;
  origpagedesc_firstfree = condemned_origpagedesc;
}

static __inline__ copy_desc_t *
alloc_copydesc(void)
{
  copy_desc_t *cd;
  if (copypagedesc_firstfree == NULL) {
    enter_kdebug ("cow.c: resources exhausted");
    halt();
  }
  cd = copypagedesc_firstfree;
  copypagedesc_firstfree = *(copy_desc_t **)cd;
  return cd;
}

static __inline__ void
dealloc_copydesc(copy_desc_t *condemned_copydesc)
{
  copy_desc_t *cd;
  *(copy_desc_t **)condemned_copydesc = copypagedesc_firstfree;
  copypagedesc_firstfree = condemned_copydesc;
}

static __inline__ void *
alloc_page(void)
{
  page_t *pg;
  if (page_firstfree == -1) {
    enter_kdebug ("cow.c: resources exhausted");
    halt();
  }
  pg = &page_pool[page_firstfree];
  page_firstfree = pagepool_freelist[page_firstfree];
  return pg;
}

static __inline__ void
dealloc_page(void *condemned_page)
{
  unsigned condemned_index;
  condemned_index = ((unsigned)condemned_page - (unsigned)page_pool)/4096; 
  pagepool_freelist[condemned_index] = page_firstfree;
  page_firstfree = condemned_index;
}

static __inline__ void
*find_existing_copy (void *orig_page, copy_id_t copy_id, 
                     origpage_desc_t **opdesc_ptr)
{
  origpage_desc_t *op;
  copy_desc_t *cd;
  op = opdesc_head;
  while (op != NULL) {
    if (op->orig_page == orig_page)
      break;
    op = op->next;
  }
  *opdesc_ptr = op;
  if (op == NULL)
    return NULL;
  cd = op->copies;
  while (cd != NULL) {
    if (cd->copy_id == copy_id)
      break;
    cd = cd->next;
  }
  if (cd == NULL)
    return NULL;
  else {
    leave_lock (&cow_lock);
    enter_lock (&cd->pagedata_lock); /* make sure page has finished 
                                        being copied */
    leave_lock (&cd->pagedata_lock);
    return cd->copy_page;
  }
}

static __inline__ void
*make_new_copy (void *orig_page, copy_id_t copy_id, origpage_desc_t *op,
                unsigned num_bytes_to_copy)
{
  copy_desc_t *cd;

  if (op == NULL) {
    /* this origpage has never been copied before */
    if (opdesc_head == NULL) {
      op = opdesc_tail = opdesc_head = alloc_origpagedesc();
      op->prev = op->next = NULL;
    } else {
      op = alloc_origpagedesc();
      op->next = opdesc_head;
      op->prev = NULL;
      opdesc_head->prev = op;
      opdesc_head = op;
    }
    op->copies = NULL;
    op->orig_page = orig_page;
  }

  cd = alloc_copydesc();
  cd->next = op->copies;
  op->copies = cd;

  cd->pagedata_lock = 1;
  cd->copy_id = copy_id;
  cd->copy_page = alloc_page();
  leave_lock(&cow_lock);
  if (num_bytes_to_copy > 4096) 
    num_bytes_to_copy = 4096;
  memcpy (cd->copy_page, orig_page, num_bytes_to_copy);
  if (num_bytes_to_copy < 4096)
    memset (cd->copy_page + num_bytes_to_copy, 0, 4096 - num_bytes_to_copy);
  leave_lock(&cd->pagedata_lock);
  return cd->copy_page;
}

static __inline__ void
*zero_new_copy (void *vaddr, copy_id_t copy_id, origpage_desc_t *op)
{
  copy_desc_t *cd;

  if (op == NULL) {
    /* this origpage has never been copied before */
    if (opdesc_head == NULL) {
      op = opdesc_tail = opdesc_head = alloc_origpagedesc();
      op->prev = op->next = NULL;
    } else {
      op = alloc_origpagedesc();
      op->next = opdesc_head;
      op->prev = NULL;
      opdesc_head->prev = op;
      opdesc_head = op;
    }
    op->copies = NULL;
    op->orig_page = vaddr;
  }

  cd = alloc_copydesc();
  cd->next = op->copies;
  op->copies = cd;

  cd->pagedata_lock = 1;
  cd->copy_id = copy_id;
  cd->copy_page = alloc_page();
  leave_lock(&cow_lock);
  memset (cd->copy_page, 0, 4096);
  leave_lock(&cd->pagedata_lock);
  return cd->copy_page;
}

static l4_threadid_t myid;

static void 
init_cow (void)
{
  int i;
  sm_service_t anon_dm;
  dataspace_t anon_ds;
  dsid_t yoon_anon_ds;
  sm_exc_t exc;
  dbgprintf ("cow: initialising\n");
  initialised_level = 1;
  myid = l4_myself();
  page_pool = (page_t *) 0xa8000000;
  nameservice_init();
  anon_dm = nameservice_get_service("dm_anon");
  if (anon_dm.dw == L4_INVALID_ID.dw) {
    hprintf ("cow: cannot find anon_dm\n");
    enter_kdebug ("cow: missing anon_dm");
    halt();
  } else {
    dbgprintf ("cow: got anon_dm = %x.%x\n", anon_dm.id.task,
               anon_dm.id.lthread);
  }
  dm_anon_ds_open (anon_dm, &yoon_anon_ds, &exc);
  anon_ds.dsm = anon_dm.dw;
  anon_ds.dsid = yoon_anon_ds;
  hprintf ("opened anon_ds, got {dsm:%x.%x,dsid:%x} exc=%x\n",
           ((l4_threadid_t)anon_ds.dsm).id.task,
           ((l4_threadid_t)anon_ds.dsm).id.lthread, anon_ds.dsid, exc._type);
  attach_region (&anon_ds, (unsigned) page_pool, 0x8000000, 0, PF_R | PF_W);
  /* now set up a freelist */
  for (i = 0; i < COW_PAGE_POOL_SIZE-1; i++) {
    pagepool_freelist[i] = i+1;
    *(origpage_desc_t **) &origpagedesc_pool[i]
      = &origpagedesc_pool[i+1];
    *(copy_desc_t **) &copypagedesc_pool[i]
      = &copypagedesc_pool[i+1];
  }
  pagepool_freelist[COW_PAGE_POOL_SIZE-1] = -1;
  *(void **) &origpagedesc_pool[COW_PAGE_POOL_SIZE-1] = NULL;
  *(void **) &copypagedesc_pool[COW_PAGE_POOL_SIZE-1] = NULL;
  page_firstfree = 0;
}

void *
get_page_copy (void *orig_page, copy_id_t copy_id)
/* Returns a copy of the original page.  Only one copy is ever made per
   copy_id. */
{
  origpage_desc_t *origpage_desc;
  void *copyptr;

  orig_page = (void *) ((unsigned) orig_page &~ 4095);

  enter_lock(&cow_lock);
  if (initialised_level == 0)  /* initialise the library */
    init_cow();

  copyptr = find_existing_copy (orig_page, copy_id, &origpage_desc);
  if (copyptr == NULL) {
    copyptr = make_new_copy (orig_page, copy_id, origpage_desc, 4096);
    dbgprintf ("cow: made new copy at %x of page at %x for %x\n", copyptr, 
                orig_page, copy_id);
  } else
    dbgprintf ("cow: found existing copy at %x of page at %x for %x\n", copyptr,
                orig_page, copy_id);

  return copyptr;
}

void *
get_partial_page_copy (void *orig_page, copy_id_t copy_id,
                       unsigned num_bytes_to_copy)
/* Returns a page containing a copy of the first `num_bytes_to_copy' bytes of
   the original page.  The remainder of the returned page is zero-initialised.
   Only one copy is ever made per copy_id. */
{
  origpage_desc_t *origpage_desc;
  void *copyptr;

  orig_page = (void *) ((unsigned) orig_page &~ 4095);

  enter_lock(&cow_lock);
  if (initialised_level == 0)  /* initialise the library */
    init_cow();

  copyptr = find_existing_copy (orig_page, copy_id, &origpage_desc);
  if (copyptr == NULL) {
    copyptr = make_new_copy (orig_page, copy_id, origpage_desc,
                             num_bytes_to_copy);
    dbgprintf ("cow: made new copy at %x of page at %x for %x\n", copyptr, 
                orig_page, copy_id);
  } else
    dbgprintf ("cow: found existing copy at %x of page at %x for %x\n", copyptr,
                orig_page, copy_id);

  return copyptr;
}

void *
get_zeroed_copy (void *vaddr, copy_id_t copy_id)
/* Returns a copy of the original page.  Only one copy is ever made per
   copy_id. */
{
  origpage_desc_t *origpage_desc;
  void *copyptr;

  vaddr = (void *) ((unsigned) vaddr &~ 4095) + 2048;

  enter_lock(&cow_lock);
  if (initialised_level == 0)  /* initialise the library */
    init_cow();

  copyptr = find_existing_copy (vaddr, copy_id, &origpage_desc);
  if (copyptr == NULL) {
    copyptr = zero_new_copy (vaddr, copy_id, origpage_desc);
    dbgprintf ("cow: zeroed new copy at %x of page at %x for %x\n", copyptr, 
                vaddr, copy_id);
  } else
    dbgprintf ("cow: found existing copy at %x of page at %x for %x\n", copyptr,
                vaddr, copy_id);

  return copyptr;
}

void
dealloc_copies(copy_id_t copy_id)
/* deallocate all the pages associated with one copy_id (eg with one client) */
{
  origpage_desc_t **op;
  copy_desc_t **pcp;
  enter_lock(&cow_lock);
  if (initialised_level == 0)  /* initialise the library */
    init_cow();
  for (op = &opdesc_head; *op != NULL; ) {
    /* go through the list of original page descriptors */
    for (pcp = &(*op)->copies; *pcp != NULL; ) {
      /* go through the copy descriptor for each copy of this orig page */
      if ((*pcp)->copy_id == copy_id) {
        dealloc_page((*pcp)->copy_page);
        dealloc_copydesc(*pcp);
        *pcp = (*pcp)->next;
      } else
        pcp = &(*pcp)->next;
    }
    if ((*op)->copies == NULL) {
      /* if this orig page has no more copies now, then delete the orig
       * page descriptor too */
      dealloc_origpagedesc(*op);
      *op = (*op)->next;
    } else
      op = &(*op)->next;
  }
  leave_lock(&cow_lock);
}

