#include <l4/types.h>
#include <l4/ipc.h>
#include <services/lib/utils.h>
#include <services/lib/ipcpipe.h>
#include <services/lib/nameservice.h>

#define USER_PRIVILEDGE 0x3
#define TRAP_GATE_KIND  0xf      

typedef struct idt_entry_s {
  unsigned short offset_lo;
  unsigned short segment_selector;
  unsigned char reserved;
  unsigned gate_kind:5;
  unsigned priviledge_level:2;
  unsigned segment_present:1;
  unsigned short offset_hi;
} idt_entry_t;

#ifdef gcc_stops_insisting_on_wrong_alignment
typedef struct idt_desc_s {
  unsigned size:16; /* number of bytes - 1 */
  idt_entry_t *ptr;
} idt_desc_t;
#else
typedef unsigned long long idt_desc_t;
#endif

static idt_desc_t idt_desc;
static idt_entry_t idt[20];
static initialised = 0;
static l4_threadid_t gdb_id = L4_NIL_ID;
static listener_stack[2048], speaker_stack[2048];
static ipcpipe_t ipcport;
void (*exceptionHook)(int);

asm (".globl nullhandler");
asm ("nullhandler: iret");
extern void nullhandler;

static void init_idt(void)
{
  int i;
#ifdef gcc_stops_insisting_on_wrong_alignment
  idt_desc.size = 20*8-1;
  idt_desc.ptr = idt;
#else
  idt_desc = (20*8-1) | ((unsigned long long)(unsigned)idt << 16);
#endif
  for (i = 0; i < 20; i++) {
    idt[i].offset_lo = 0/*(unsigned short)&nullhandler*/;
    idt[i].offset_hi = 0/*(unsigned)&nullhandler >> 16*/;
    idt[i].reserved = 0;
    idt[i].gate_kind = TRAP_GATE_KIND;
    idt[i].segment_present = 1;
    idt[i].segment_selector = 0;
    idt[i].priviledge_level = USER_PRIVILEDGE;
  }
  asm volatile ("lidt (%0)" : : "a"(&idt_desc));
  initialised = 1;
}

void exceptionHandler(int i, void (*f)())
{
  if (i < 0 || i > 19) {
    hprintf ("gdbstub: bad exception index: 0x%x\n", i);
    enter_kdebug ("gdbstub: bad exc index");
  }
  if (!initialised) {
    init_idt();
  }
  /* hprintf ("reg exc %d handler %x\n", i, f); */
  idt[i].offset_lo = (unsigned short) (unsigned) f;
  idt[i].offset_hi = (unsigned) f >> 16;
  idt[i].reserved = 0;
  idt[i].gate_kind = TRAP_GATE_KIND;
  idt[i].priviledge_level = USER_PRIVILEDGE;
  idt[i].segment_present = 1;
  idt[i].segment_selector = 0;
  asm volatile ("lidt (%0)" : : "a"(&idt_desc));
}

void initDebug (void)
{
  l4_msgdope_t result;
  unsigned dw1, dw2, dw3;
  l4_threadid_t remote_listener, remote_speaker;
  int ret;
  l4_threadid_t myself;
  char name[3];
  nameservice_init();
  myself = l4_myself();
  hsprintf (name, "%x", myself.id.task);
  nameservice_register_service(name);
  do {
    ret = l4_i386_ipc_wait (&gdb_id, L4_IPC_SHORT_MSG, &dw1, &dw2, &dw3,
                            L4_IPC_NEVER, &result);
  } while (dw2 != 0x9db5e1 || ret != 0);
  hprintf ("gdbstub: got gdb_id=%x.%x\n", gdb_id.id.task, gdb_id.id.lthread);
  remote_listener = remote_speaker = gdb_id;
  remote_listener.id.lthread = 0x3c;
  remote_speaker.id.lthread = 0x3d;
  ipcpipe_open (&ipcport, remote_listener, remote_speaker, 0x3c, 0x3d, 
                myself.id.lthread,
                &listener_stack[2040], &speaker_stack[2040]);
  l4_i386_ipc_send (gdb_id, L4_IPC_SHORT_MSG, 0, 0, 0, L4_IPC_NEVER, &result);
}

void putDebugChar(char c)
{
  ipcpipe_putchar (&ipcport, c);
}

char getDebugChar(void)
{
  return ipcpipe_getchar (&ipcport);
}

void flushDebugOutput(void)
{
  ipcpipe_flushoutput(&ipcport);
}



