/* gdb stub for l4/i386                                     */
/* ported to L4 by Luke Deller, IBM                         */
/* this is internal IBM source code                         */
/* below is the header for the original i386 serial gdbstub */

/****************************************************************************

		THIS SOFTWARE IS NOT COPYRIGHTED

   HP offers the following for use in the public domain.  HP makes no
   warranty with regard to the software or it's performance and the
   user accepts the software "AS IS" with all faults.

   HP DISCLAIMS ANY WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD
   TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO THE WARRANTIES
   OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

****************************************************************************/

/****************************************************************************
 *  Header: remcom.c,v 1.34 91/03/09 12:29:49 glenne Exp $
 *
 *  Module name: remcom.c $
 *  Revision: 1.34 $
 *  Date: 91/03/09 12:29:49 $
 *  Contributor:     Lake Stevens Instrument Division$
 *
 *  Description:     low level support for gdb debugger. $
 *
 *  Considerations:  only works on target hardware $
 *
 *  Written by:      Glenn Engel $
 *  ModuleState:     Experimental $
 *
 *  NOTES:           See Below $
 *
 *  Modified for 386 by Jim Kingdon, Cygnus Support.
 *
 *  To enable debugger support, two things need to happen.  One, a
 *  call to set_debug_traps() is necessary in order to allow any breakpoints
 *  or error conditions to be properly intercepted and reported to gdb.
 *  Two, a breakpoint needs to be generated to begin communication.  This
 *  is most easily accomplished by a call to breakpoint().  Breakpoint()
 *  simulates a breakpoint by executing a trap #1.
 *
 *  The external function exceptionHandler() is
 *  used to attach a specific handler to a specific 386 vector number.
 *  It should use the same privilege level it runs at.  It should
 *  install it as an interrupt gate so that interrupts are masked
 *  while the handler runs.
 *  Also, need to assign exceptionHook and oldExceptionHook.
 *
 *  Because gdb will sometimes write to the stack area to execute function
 *  calls, this program cannot rely on using the supervisor stack so it
 *  uses it's own stack area reserved in the int array remcomStack.
 *
 *************
 *
 *    The following gdb commands are supported:
 *
 * command          function                               Return value
 *
 *    g             return the value of the CPU registers  hex data or ENN
 *    G             set the value of the CPU registers     OK or ENN
 *
 *    mAA..AA,LLLL  Read LLLL bytes at address AA..AA      hex data or ENN
 *    MAA..AA,LLLL: Write LLLL bytes at address AA.AA      OK or ENN
 *
 *    c             Resume at current address              SNN   ( signal NN)
 *    cAA..AA       Continue at address AA..AA             SNN
 *
 *    s             Step one instruction                   SNN
 *    sAA..AA       Step one instruction from AA..AA       SNN
 *
 *    k             kill
 *
 *    ?             What was the last sigval ?             SNN   (signal NN)
 *
 * All commands and responses are sent with a packet which includes a
 * checksum.  A packet consists of
 *
 * $<packet info>#<checksum>.
 *
 * where
 * <packet info> :: <characters representing the command or response>
 * <checksum>    :: < two hex digits computed as modulo 256 sum of <packetinfo>>
 *
 * When a packet is received, it is first acknowledged with either '+' or '-'.
 * '+' indicates a successful transfer.  '-' indicates a failed transfer.
 *
 * Example:
 *
 * Host:                  Reply:
 * $m0,10#2a               +$00010203040506070809101112131415#42
 *
 ****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <l4/types.h>
#include <l4/syscalls.h>
#include <services/client/rm.h>
#include <services/lib/utils.h>

#define GDBSTUB_LTHREAD 3
/* if you change the above line, also change asm code in _remcomHandler */

/************************************************************************
 *
 * external low-level support routines
 */
typedef void (*ExceptionHook)(int);   /* pointer to function with int parm */
typedef void (*Function)();           /* pointer to a function */

extern void putDebugChar();	/* write a single character      */
extern int getDebugChar();	/* read and return a single char */

extern Function exceptionHandler();  /* assign an exception handler */
extern ExceptionHook exceptionHook;  /* hook variable for errors/exceptions */

/************************************************************************/
/* BUFMAX defines the maximum number of characters in inbound/outbound buffers*/
/* at least NUMREGBYTES*2 are needed for register packets */
#define BUFMAX 400

static char initialized;  /* boolean flag. != 0 means we've been initialized */

int     remote_debug;
/*  debug >  0 prints ill-formed commands in valid packets & checksum errors */

void waitabit();

static const char hexchars[]="0123456789abcdef";

static unsigned long long alive_threads = 0; /* bitmap of alive threads */
static const char *(*user_getinfo_fn) (l4_threadid_t threadid, int whatinfo)
  = NULL;

/* Number of registers.  */
#define NUMREGS	16

/* Number of bytes of registers.  */
#define NUMREGBYTES (NUMREGS * 4)

enum regnames {EAX, ECX, EDX, EBX, ESP, EBP, ESI, EDI,
	       PC /* also known as eip */,
	       PS /* also known as eflags */,
	       CS, SS, DS, ES, FS, GS};

/*
 * these should not be static cuz they can be used outside this module
 */

static int otherops_thread;
int threadregisters[16*NUMREGS];
#define registers (&threadregisters[otherops_thread*NUMREGS])
int tempregisters[NUMREGS];
int tempreg_lock;
const char *memfault_errmsg = NULL;

typedef struct postponed_bpt_clear_s {
  unsigned addr;
  unsigned char origval;
} postponed_bpt_clear_t;
#define MAX_POSTPONED_BPT_CLEARS 32
postponed_bpt_clear_t postponed_bpt_clears[MAX_POSTPONED_BPT_CLEARS];
int num_postponed_bpt_clears = 0;
unsigned repair_bpt_early[64];

#define STACKSIZE 10240
int remcomStack[STACKSIZE/sizeof(int)];
static int* stackPtr = &remcomStack[STACKSIZE/sizeof(int) - 2];

/*
 * In many cases, the system will want to continue exception processing
 * when a continue command is given.
 * oldExceptionHook is a function to invoke in this case.
 */

static ExceptionHook oldExceptionHook;

/***************************  ASSEMBLY CODE MACROS *************************/
/* 									   */

extern void
return_to_prog ();

int saved_edi;

/* Restore the program's registers (including the stack pointer, which
   means we get the right stack and don't have to worry about popping our
   return address and any stack frames and so on) and return.  */
asm(".text");
asm(".globl return_to_prog");
asm("return_to_prog:");
asm("        movl threadregisters+16(,%edi), %esp");
asm("        movl threadregisters+4(,%edi), %ecx");
asm("        movl threadregisters+8(,%edi), %edx");
asm("        movl threadregisters+12(,%edi), %ebx");
asm("        movl threadregisters+20(,%edi), %ebp");
asm("        movl threadregisters+24(,%edi), %esi");
asm("        movl threadregisters+36(,%edi), %eax");
asm("        pushl %eax");  /* saved eflags */
asm("        movl threadregisters+40(,%edi), %eax");
asm("        pushl %eax");  /* saved cs */
asm("        movl threadregisters+32(,%edi), %eax");
asm("        pushl %eax");  /* saved eip */
asm("        movl threadregisters(,%edi), %eax");
asm("        movl threadregisters+28(,%edi), %edi");
/* use iret to restore pc and flags together so
   that trace flag works right.  */
asm("        iret");

#define BREAKPOINT() asm("   int $3");

/* Put the error code here just in case the user cares.  */
asm(".data");
int gdb_i386errcode;
/* Likewise, the vector number here (since GDB only gets the signal
   number through the usual means, and that's not very specific).  */
int gdb_i386vector = -1;

/* GDB stores segment registers in 32-bit words (that's just the way
   m-i386v.h is written).  So zero the appropriate areas in registers.  */
#define SAVE_REGISTERS1() \
  /* mutex for accessing tempregisters (spinlock) */ \
  /* don't touch any registers yet!               */ \
  asm ("998:");								     \
  asm ("lock");	/* SMP L4 ready :-) */					     \
  asm ("btsl $0, tempreg_lock");                                             \
  asm ("jnc 999f");                                                          \
  asm ("jmp 998b");							     \
  asm ("999:");								     \
  /* want to put registers somewhere thread-specific, but cannot call */     \
  /* l4_myself() without trashing registers, so put them somewhere    */     \
  /* global temporarily                                               */     \
  asm ("movl %eax, tempregisters");                                   	     \
  asm ("movl %ecx, tempregisters+4");			  		     \
  asm ("movl %edx, tempregisters+8");			  		     \
  asm ("movl %ebx, tempregisters+12");			  		     \
  asm ("movl %ebp, tempregisters+20");			  		     \
  asm ("movl %esi, tempregisters+24");			  		     \
  asm ("movl %edi, tempregisters+28");			  		     \
  /* now call l4_myself() */						     \
  asm ("xorl %esi, %esi");						     \
  asm ("int  $0x31");							     \
  asm ("movl %esi, %edi");						     \
  asm ("shrl $10, %edi");						     \
  asm ("andl $0x3f, %edi"); /* edi = l4_myself().id.lthread */               \
  /* now move tempregisters -> registers[lthread*NUMREGS*4] */                 \
  asm ("shll $6,%edi");							     \
  asm ("movl tempregisters, %eax");					     \
  asm ("movl tempregisters+4, %ecx");			  		     \
  asm ("movl %eax,threadregisters(,%edi)");				     \
  asm ("movl %ecx,threadregisters+4(,%edi)");				     \
  asm ("movl tempregisters+8, %eax");			  		     \
  asm ("movl tempregisters+12, %ecx");			  		     \
  asm ("movl %eax,threadregisters+8(,%edi)");				     \
  asm ("movl %ecx,threadregisters+12(,%edi)");				     \
  asm ("movl tempregisters+20, %eax");			  		     \
  asm ("movl tempregisters+24, %ecx");			  		     \
  asm ("movl %eax,threadregisters+20(,%edi)");				     \
  asm ("movl %ecx,threadregisters+24(,%edi)");				     \
  asm ("movl tempregisters+28, %eax");			  		     \
  asm ("movl $0, tempreg_lock");					     \
  asm ("movl %eax,threadregisters+28(,%edi)");				     \
  asm ("movw $0, %ax");							     \
  asm ("movw %ds, threadregisters+48(,%edi)");		  		     \
  asm ("movw %ax, threadregisters+50(,%edi)");				     \
  asm ("movw %es, threadregisters+52(,%edi)");		  		     \
  asm ("movw %ax, threadregisters+54(,%edi)");				     \
  asm ("movw %fs, threadregisters+56(,%edi)");		  		     \
  asm ("movw %ax, threadregisters+58(,%edi)");				     \
  asm ("movw %gs, threadregisters+60(,%edi)");		  		     \
  asm ("movw %ax, threadregisters+62(,%edi)");
#define SAVE_ERRCODE() \
  asm ("popl %ebx");                                  			     \
  asm ("movl %ebx, gdb_i386errcode");
#define SAVE_REGISTERS2() \
  asm ("popl %ebx"); /* old eip */			  		     \
  asm ("movl %ebx, threadregisters+32(,%edi)");		  		     \
  asm ("popl %ebx");	 /* old cs */			  		     \
  asm ("movl %ebx, threadregisters+40(,%edi)");		  		     \
  asm ("movw %ax, threadregisters+42(,%edi)");                               \
  asm ("popl %ebx");	 /* old eflags */		  		     \
  asm ("movl %ebx, threadregisters+36(,%edi)");		 		     \
  /* Now that we've done the pops, we can save the stack pointer.");  */     \
  asm ("movw %ss, threadregisters+44(,%edi)");				     \
  asm ("movw %ax, threadregisters+46(,%edi)");         	       	       	     \
  asm ("movl %esp, threadregisters+16(,%edi)");

/* See if mem_fault_routine is set, if so just IRET to that address.  */
#define CHECK_FAULT() 
/* this doesn't work so well multithreaded .. now rm sets mem_err instead 
  asm ("cmpl $0, mem_fault_routine");					   \
  asm ("jne mem_fault");
*/

asm (".text");
asm ("mem_fault:");
/* OK to clobber temp registers; we're just going to end up in set_mem_err.  */
/* Pop error code from the stack and save it.  */
asm ("     popl %eax");
asm ("     movl %eax, gdb_i386errcode");

asm ("     popl %eax"); /* eip */
/* We don't want to return there, we want to return to the function
   pointed to by mem_fault_routine instead.  */
asm ("     movl mem_fault_routine, %eax");
asm ("     popl %ecx"); /* cs (low 16 bits; junk in hi 16 bits).  */
asm ("     popl %edx"); /* eflags */

/* Remove this stack frame; when we do the iret, we will be going to
   the start of a function, so we want the stack to look just like it
   would after a "call" instruction.  */
asm ("     leave");

/* Push the stuff that iret wants.  */
asm ("     pushl %edx"); /* eflags */
asm ("     pushl %ecx"); /* cs */
asm ("     pushl %eax"); /* eip */

/* Zero mem_fault_routine.  */
asm ("     movl $0, %eax");
asm ("     movl %eax, mem_fault_routine");

asm ("iret");

#define CALL_HOOK() asm("jmp _remcomHandler");

/* This function is called when a i386 exception occurs.  It saves
 * all the cpu regs in the _registers array, munges the stack a bit,
 * and invokes an exception handler (remcom_handler).
 *
 * stack on entry:                       stack on exit:
 *   old eflags                          vector number
 *   old cs (zero-filled to 32 bits)
 *   old eip
 *
 */
extern void _catchException3();
asm(".text");
asm(".globl _catchException3");
asm("_catchException3:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $3");
CALL_HOOK();

/* Same thing for exception 1.  */
extern void _catchException1();
asm(".text");
asm(".globl _catchException1");
asm("_catchException1:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $1");
CALL_HOOK();

/* Same thing for exception 0.  */
extern void _catchException0();
asm(".text");
asm(".globl _catchException0");
asm("_catchException0:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $0");
CALL_HOOK();

/* Same thing for exception 4.  */
extern void _catchException4();
asm(".text");
asm(".globl _catchException4");
asm("_catchException4:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $4");
CALL_HOOK();

/* Same thing for exception 5.  */
extern void _catchException5();
asm(".text");
asm(".globl _catchException5");
asm("_catchException5:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $5");
CALL_HOOK();

/* Same thing for exception 6.  */
extern void _catchException6();
asm(".text");
asm(".globl _catchException6");
asm("_catchException6:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $6");
CALL_HOOK();

/* Same thing for exception 7.  */
extern void _catchException7();
asm(".text");
asm(".globl _catchException7");
asm("_catchException7:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $7");
CALL_HOOK();

/* Same thing for exception 8.  */
extern void _catchException8();
asm(".text");
asm(".globl _catchException8");
asm("_catchException8:");
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $8");
CALL_HOOK();

/* Same thing for exception 9.  */
extern void _catchException9();
asm(".text");
asm(".globl _catchException9");
asm("_catchException9:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $9");
CALL_HOOK();

/* Same thing for exception 10.  */
extern void _catchException10();
asm(".text");
asm(".globl _catchException10");
asm("_catchException10:");
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $10");
CALL_HOOK();

/* Same thing for exception 12.  */
extern void _catchException12();
asm(".text");
asm(".globl _catchException12");
asm("_catchException12:");
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $12");
CALL_HOOK();

/* Same thing for exception 16.  */
extern void _catchException16();
asm(".text");
asm(".globl _catchException16");
asm("_catchException16:");
SAVE_REGISTERS1();
SAVE_REGISTERS2();
asm ("pushl $16");
CALL_HOOK();

/* For 13, 11, and 14 we have to deal with the CHECK_FAULT stuff.  */

/* Same thing for exception 13.  */
extern void _catchException13 ();
asm (".text");
asm (".globl _catchException13");
asm ("_catchException13:");
CHECK_FAULT();
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $13");
CALL_HOOK();

/* Same thing for exception 11.  */
extern void _catchException11 ();
asm (".text");
asm (".globl _catchException11");
asm ("_catchException11:");
CHECK_FAULT();
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $11");
CALL_HOOK();

/* Same thing for exception 14.  */
extern void _catchException14 ();
asm (".text");
asm (".globl _catchException14");
asm ("_catchException14:");
CHECK_FAULT();
SAVE_REGISTERS1();
SAVE_ERRCODE();
SAVE_REGISTERS2();
asm ("pushl $14");
CALL_HOOK();

/*
 * remcomHandler is a front end for handle_exception.  It moves the
 * stack pointer into an area reserved for debugger use.
 */
asm("_remcomHandler:");
asm("           popl %edx");      /* get the exception number   */
asm("		pushl %edi");
asm("		cmpl $0,repair_bpt_early(,%edi)");
asm("		jz   2f");
asm("		movl repair_bpt_early(,%edi),%ebx");
asm("		movb $0xcc,(%ebx)");
asm("		movl $0,repair_bpt_early(,%edi)");
asm("2:		");
asm("		movl (%esp),%edi");
asm("		xorl %eax,%eax");
asm("		xorl %ecx,%ecx");
asm("		xorl %ebp,%ebp");
asm("		andl $0xFFFF03FF,%esi");
asm("		orl  $0x00000C00,%esi"); /* lthread 3 is gdbthread */
asm("		int  $0x30");
asm("waiting_for_exception_2b_handled:");
asm("		popl %edi");
asm("		jmp  return_to_prog");

extern void waiting_for_exception_2b_handled;

static int hex(ch)
char ch;
{
  if ((ch >= 'a') && (ch <= 'f')) return (ch-'a'+10);
  if ((ch >= '0') && (ch <= '9')) return (ch-'0');
  if ((ch >= 'A') && (ch <= 'F')) return (ch-'A'+10);
  return (-1);
}


/* scan for the sequence $<data>#<checksum>     */
static void getpacket(buffer)
char * buffer;
{
  unsigned char checksum;
  unsigned char xmitcsum;
  int  i;
  int  count;
  char ch;

  do {
    /* wait around for the start character, ignore all other characters */
    while ((ch = (getDebugChar() & 0x7f)) != '$');
    checksum = 0;
    xmitcsum = -1;

    count = 0;

    /* now, read until a # or end of buffer is found */
    while (count < BUFMAX) {
      ch = getDebugChar() & 0x7f;
      if (ch == '#') break;
      checksum = checksum + ch;
      buffer[count] = ch;
      count = count + 1;
      }
    buffer[count] = 0;

    if (ch == '#') {
      xmitcsum = hex(getDebugChar() & 0x7f) << 4;
      xmitcsum += hex(getDebugChar() & 0x7f);
      if ((remote_debug ) && (checksum != xmitcsum)) {
        printf ("gdbstub: bad checksum.  My count = 0x%x, sent=0x%x. buf=%s\n",
		 checksum,xmitcsum,buffer);
      }

      if (checksum != xmitcsum) putDebugChar('-');  /* failed checksum */
      else {
	 putDebugChar('+');  /* successful transfer */
	 /* if a sequence char is present, reply the sequence ID */
	 if (buffer[2] == ':') {
	    putDebugChar( buffer[0] );
	    putDebugChar( buffer[1] );
	    /* remove sequence chars from buffer */
	    count = strlen(buffer);
	    for (i=3; i <= count; i++) buffer[i-3] = buffer[i];
	 }
      }
    }
  } while (checksum != xmitcsum);

}

/* send the packet in buffer.  */


static void putpacket(buffer)
char * buffer;
{
  unsigned char checksum;
  int  count;
  char ch;

  /*  $<packet info>#<checksum>. */
  do {
  putDebugChar('$');
  checksum = 0;
  count    = 0;

  while ((ch=buffer[count])) {
    putDebugChar(ch);
    checksum += ch;
    count += 1;
  }

  putDebugChar('#');
  putDebugChar(hexchars[checksum >> 4]);
  putDebugChar(hexchars[checksum % 16]);

  } while ((getDebugChar() & 0x7f) != '+');

}

static char  remcomInBuffer[BUFMAX];
static char  remcomOutBuffer[BUFMAX];
static short error;


static void debug_error(format, parm)
char * format;
char * parm;
{
  if (remote_debug) {
    printf ("gdbstub: ");
    printf (format,parm);
  }
}

/* Address of a routine to RTE to if we get a memory fault.  */
void (*volatile mem_fault_routine)() = NULL;

/* Indicate to caller of mem2hex or hex2mem that there has been an
   error.  */
volatile int mem_err = 0;

static void
set_mem_err ()
{
  mem_err = 1;
}

/* These are separate functions so that they are so short and sweet
   that the compiler won't save any registers (if there is a fault
   to mem_fault, they won't get restored, so there better not be any
   saved).  */
static int
get_char (addr)
     char *addr;
{
  return *addr;
}

static void
set_char (addr, val)
     char *addr;
     int val;
{
  *addr = val;
}

/* convert the memory pointed to by mem into hex, placing result in buf */
/* return a pointer to the last char put in buf (null) */
/* If MAY_FAULT is non-zero, then we should set mem_err in response to
   a fault; if zero treat a fault like any other fault in the stub.  */
static char* mem2hex(mem, buf, count, may_fault)
char* mem;
char* buf;
int   count;
int may_fault;
{
      int i;
      unsigned char ch;

      if (may_fault)
	  mem_fault_routine = set_mem_err;
      for (i=0;i<count;i++) {
          ch = get_char (mem++);
	  if (may_fault && mem_err)
	    return (buf);
          *buf++ = hexchars[ch >> 4];
          *buf++ = hexchars[ch % 16];
      }
      *buf = 0;
      if (may_fault)
	  mem_fault_routine = NULL;

      buf -= count*2;
      mem -= count;
      for (i = 0; i < num_postponed_bpt_clears; i++) {
        char *addr;
        addr = (char *)postponed_bpt_clears[i].addr;
        if (addr >= mem && addr < mem+count) {
          ch = postponed_bpt_clears[i].origval;
          buf[(addr-mem)*2] = hexchars[ch >> 4];
          buf[(addr-mem)*2+1] = hexchars[ch % 16];
        }
      } 
      return(buf);
}

/* convert the hex array pointed to by buf into binary to be placed in mem */
/* return a pointer to the character AFTER the last byte written */
static char* hex2mem(buf, mem, count, may_fault)
char* buf;
char* mem;
int   count;
int may_fault;
{
      int i;
      unsigned char ch;

      if (may_fault)
	  mem_fault_routine = set_mem_err;
      for (i=0;i<count;i++) {
          ch = hex(*buf++) << 4;
          ch = ch + hex(*buf++);
          set_char (mem++, ch);
	  if (may_fault && mem_err)
	    return (mem);
      }
      if (may_fault)
	  mem_fault_routine = NULL;
      return(mem);
}

/* this function takes the 386 exception vector and attempts to
   translate this number into a unix compatible signal value */
static int computeSignal( exceptionVector )
int exceptionVector;
{
  int sigval;
  switch (exceptionVector) {
    case -2 : sigval = 3; break; /* quit */
    case -1 : sigval = 2; break; /* interrupt; Luke's hack for ^C in gdb */
    case 0 : sigval = 8; break; /* divide by zero */
    case 1 : sigval = 5; break; /* debug exception */
    case 3 : sigval = 5; break; /* breakpoint */
    case 4 : sigval = 16; break; /* into instruction (overflow) */
    case 5 : sigval = 16; break; /* bound instruction */
    case 6 : sigval = 4; break; /* Invalid opcode */
    case 7 : sigval = 8; break; /* coprocessor not available */
    case 8 : sigval = 7; break; /* double fault */
    case 9 : sigval = 11; break; /* coprocessor segment overrun */
    case 10 : sigval = 11; break; /* Invalid TSS */
    case 11 : sigval = 11; break; /* Segment not present */
    case 12 : sigval = 11; break; /* stack exception */
    case 13 : sigval = 11; break; /* general protection */
    case 14 : sigval = 11; break; /* page fault */
    case 16 : sigval = 7; break; /* coprocessor error */
    default:
      sigval = 7;         /* "software generated"*/
  }
  return (sigval);
}

/**********************************************/
/* WHILE WE FIND NICE HEX CHARS, BUILD AN INT */
/* RETURN NUMBER OF CHARS PROCESSED           */
/**********************************************/
static int hexToInt(char **ptr, int *intValue)
{
    int numChars = 0;
    int hexValue;

    *intValue = 0;

    while (**ptr)
    {
        hexValue = hex(**ptr);
        if (hexValue >=0)
        {
            *intValue = (*intValue <<4) | hexValue;
            numChars ++;
        }
        else
            break;

        (*ptr)++;
    }

    return (numChars);
}

static int hexToIntn(char **ptr, int *intValue, int maxlen)
{
    int numChars = 0;
    int hexValue;

    *intValue = 0;

    while (**ptr)
    {
        if (--maxlen < 0) break;
        hexValue = hex(**ptr);
        if (hexValue >=0)
        {
            *intValue = (*intValue <<4) | hexValue;
            numChars ++;
        }
        else
            break;
        (*ptr)++;
    }

    return (numChars);
}

void
intToHex (char *ptr, unsigned intValue, int fieldsize)
/* writes the hex representation of intValue to *ptr, with fieldsize hex digits
 */
{
  while (fieldsize-- > 0)
    *ptr++ = fieldsize>7?'0':hexchars[(intValue>>(fieldsize<<2))&0xf];
}

static int prog_exitstatus;

void tell_gdb_byebye(int exitstatus)
{
  l4_threadid_t gdbthread;
  int mylthread;
  unsigned dum;
  l4_msgdope_t result;
  gdbthread = l4_myself();
  mylthread = gdbthread.id.lthread;
  gdbthread.id.lthread = GDBSTUB_LTHREAD;
  prog_exitstatus = exitstatus;
  l4_i386_ipc_call (gdbthread, L4_IPC_SHORT_MSG, -2, 0, mylthread*16,
                    L4_IPC_SHORT_MSG, &dum, &dum, &dum,
                    L4_IPC_NEVER, &result);
}

static int catch_l4kdebug_call(int *sigval)
  /* emulates l4 kernel debugger calls */
  /* for info on l4kdebug calls, check out services/include/l4/kdebug.h */
{
  unsigned char *peekaddr;
  peekaddr = (unsigned char *)registers[PC];
  mem_fault_routine = set_mem_err;
  if (peekaddr[-1] != 0xcc) {
    mem_fault_routine = NULL;
    return 0;
  }
  if (peekaddr[0] == 0x3c) { /* output call */
    switch(peekaddr[1]) {
      case 0:  /* outchar   */
        mem_fault_routine = NULL;
        remcomOutBuffer[0] = 'O';
        remcomOutBuffer[1] =  hexchars[(registers[EAX] >> 4) % 16];
        remcomOutBuffer[2] =  hexchars[registers[EAX] % 16];
        remcomOutBuffer[3] = 0;
        putpacket(remcomOutBuffer);
        return 1;
      default:
      case 2:  /* outstring */
      case 5:  /* outhex32  */
      case 6:  /* outhex20  */
      case 7:  /* outhex16  */
      case 11: /* outdec    */
        /* emulation unimplemented .. do nothing.  */
        /* does anybody actually use these calls?? */
        mem_fault_routine = NULL;
        return 1;
    }
  } else if (peekaddr[0] == 0xeb) { /* enter_kdebug */
    char *buf = &remcomOutBuffer[1];
    int mesglen;
    mem_fault_routine = NULL;
    remcomOutBuffer[0] = 'O';
    mem2hex("*** enter_kdebug: ", buf, 18, 0);
    buf += 18*2;
    mesglen = (peekaddr[1] < BUFMAX/2-21) ? peekaddr[1] : (BUFMAX/2-21);
    mem_err = 0;
    mem2hex(&peekaddr[2], buf, mesglen, 1);
    buf += mesglen*2;
    *buf++ = '0';
    *buf++ = 'a';
    *buf++ = '\0';
    putpacket(remcomOutBuffer);
    *sigval = 6; /* SIGABRT, abort signal */
    return 0;
  }
  mem_fault_routine = NULL;
  return 0;
}

extern void do_fake_exception(void);
extern void ignore_trap_here(void);
static unsigned mtde_eflags, mtde_eip;
static int reserve_stack[64];

void make_thread_do_exception(l4_threadid_t target)
{
  /* this assumes that the target thread is blocked on a pagefault */
  unsigned mtde_esp;
  l4_threadid_t pager, preempter;
  pager.dw = preempter.dw = (unsigned)-1;
  l4_thread_ex_regs (target, (unsigned)-1, (unsigned)-1, &preempter,
                     &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  pager.dw = preempter.dw = (unsigned)-1;
  memfault_errmsg = NULL;
  if (remote_debug)
    hprintf ("mtde_eip=%x,mtde_esp=%x,mtde_eflags=%x\n", mtde_eip, mtde_esp,
             mtde_eflags);
  if (mtde_eip == (unsigned)do_fake_exception)
    /* D'oh!  We must have had a bad stack pointer.  Fix %esp & try again */
    /* FIXME: %eip reported to gdb will be wrong */
    l4_thread_ex_regs (target, (unsigned)do_fake_exception,
                       (unsigned)reserve_stack,
                       &preempter, &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  else
    l4_thread_ex_regs (target, (unsigned)do_fake_exception, (unsigned)-1,
                       &preempter, &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  l4_thread_switch(target);
  l4_thread_switch(target); /* try to give at least one whole timeslice */
  asm volatile (
    /* FIXME: this code is not thread-safe */
    ".text\n\t"
    "jmp  mtde_skip_to_here\n\t"
    ".globl do_fake_exception\n"
    "do_fake_exception:\n\t"
    "pushl %0\n\t"   /* eflags */
    ".globl ignore_trap_here\n"
    "ignore_trap_here:\n\t"
    "pushl %%cs\n\t" /* cs  */
    "pushl %1\n\t"   /* eip */
    "pushl $0\n\t"   /* dummy error code, not used by stub */
    "jmp   _catchException14\n\t"
    "mtde_skip_to_here:"
    :
    : "m"(mtde_eflags),"m"(mtde_eip)
  );
}

void make_thread_do_exception_wmesg(l4_threadid_t target, const char *errmsg)
{
  /* this assumes that the target thread is blocked on a pagefault */
  unsigned mtde_esp;
  l4_threadid_t pager, preempter;
  pager.dw = preempter.dw = (unsigned)-1;
  l4_thread_ex_regs (target, (unsigned)-1, (unsigned)-1, &preempter,
                     &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  pager.dw = preempter.dw = (unsigned)-1;
  memfault_errmsg = errmsg;
  if (mtde_eip == (unsigned)do_fake_exception)
    /* D'oh!  We must have had a bad stack pointer.  Fix %esp & try again */
    /* FIXME: %eip reported to gdb will be wrong */
    l4_thread_ex_regs (target, (unsigned)do_fake_exception,
                       (unsigned)reserve_stack,
                       &preempter, &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  else
    l4_thread_ex_regs (target, (unsigned)do_fake_exception, (unsigned)-1,
                       &preempter, &pager, &mtde_eflags, &mtde_eip, &mtde_esp);
  l4_thread_switch(target);
  l4_thread_switch(target); /* try to give at least one whole timeslice */
}

/*
 * This function does all command procesing for interfacing to gdb.
 */

static void handle_exception(int, l4_threadid_t);

static void gdbstub_thread(int creator)
{
  l4_threadid_t srcthread, myself;
  unsigned dummy, save;
  int exception_code;
  l4_msgdope_t msgdope;

  myself = l4_myself();
  hprintf("gdbstub: alive, I am %x.%x\n", myself.id.task, myself.id.lthread);
  initDebug();
  l4_i386_ipc_send((l4_threadid_t)(unsigned)creator, L4_IPC_SHORT_MSG,
                   0, 0, 0, L4_IPC_NEVER, &msgdope);
  while (1) {
    l4_i386_ipc_wait (&srcthread, L4_IPC_SHORT_MSG, &exception_code, &dummy,
                      &save, L4_IPC_NEVER, &msgdope);
    while (2) {
      if ((msgdope.msgdope & 0xf0) || !task_equal(srcthread, myself)) {
        hprintf ("gdbstub: received bad ipc\n");
        break;
      }
      if (exception_code == -1 && srcthread.id.lthread == 0x3c) {
        /* break notification from ipcpipe listener */
        if (remote_debug)  hprintf ("got break signal from gdb\n");
        handle_exception (-1, srcthread);
        break;
      } else if (exception_code >= 0 && save != srcthread.id.lthread * 64) {
        hprintf ("gdbstub: received wrong ipc from lthread 0x%x\n",
                 srcthread.id.lthread);
        break;
      }
      if (remote_debug)
        hprintf ("gdbstub: got exc %d from lthread 0x%x\n", exception_code,
                 srcthread.id.lthread);
      handle_exception(exception_code, srcthread);
      if (remote_debug)
        hprintf ("gdbstub: continuing lthread 0x%x\n", srcthread.id.lthread);
      l4_i386_ipc_reply_and_wait (srcthread, L4_IPC_SHORT_MSG, 0, 0, save,
                                  &srcthread, L4_IPC_SHORT_MSG,
                                  &exception_code, &dummy, &save,
                                  L4_IPC_NEVER, &msgdope);
    }
  }
}

static void printPostponedBreakpointClears(void)
{
  /* used for debugging only */
  int i;
  char *pos;
  remcomOutBuffer[0] = 'O';
  pos = &remcomOutBuffer[1];
  for (i = 0; i < num_postponed_bpt_clears; i++)
    hprintf ("pbc[%d].addr == 0x%x\n", i, postponed_bpt_clears[i].addr);
}

static void handle_exception(int exceptionVector, l4_threadid_t threadid)
{
  int    sigval;
  int    addr, length;
  char * ptr;
  int    newPC;
  int    lthread;
  int    a, b, c, i;
  l4_threadid_t thr;
  static int first_breakpoint = 1;


  gdb_i386vector = exceptionVector;
  lthread = threadid.id.lthread;
  otherops_thread = lthread;

  sigval = computeSignal( exceptionVector );

  if (exceptionVector == 3) {
    if (catch_l4kdebug_call(&sigval)) {
          newPC = registers[PC];
          return; /* from exception */
    }
  } else if (exceptionVector == 1) {
    if ((registers[PC] == (int)do_fake_exception) ||
        (registers[PC] == (int)ignore_trap_here)) {
      registers[PS] &=~ 0x100; /* turn off tracing */
      newPC = registers[PC];
      return; /* from exception */
    }
  } else if (exceptionVector == 14) {
    if (memfault_errmsg != NULL) { /* print error mesg from rm */
      int len = strlen(memfault_errmsg);
      remcomOutBuffer[0] = 'O';
      mem2hex (memfault_errmsg, &remcomOutBuffer[1], len, 0);
      putpacket(remcomOutBuffer);
      memfault_errmsg = NULL;
      sigval = 10; /* bus error rather than segfault */
    }
  } else if (exceptionVector == -2) {
    l4_threadid_t rm;
    l4_msgdope_t result;
    sm_exc_t exc;
    remcomOutBuffer[0] = 'W';
    remcomOutBuffer[1] =  hexchars[(prog_exitstatus >> 4) % 16];
    remcomOutBuffer[2] =  hexchars[prog_exitstatus % 16];
    remcomOutBuffer[3] = 0;
    putpacket(remcomOutBuffer);
    flushDebugOutput();
    rm = l4_myself();
    rm.id.lthread = 0;
    rm_pgr_exit((sm_service_t)rm.dw, &exc);
    halt();
  }

  if (remote_debug) printf("vector=%d, sr=0x%x, pc=0x%x, thr=0x%x\n",
			    exceptionVector,
			    registers[ PS ],
			    registers[ PC ],
                            lthread);

  if (!first_breakpoint) {
    /* reply to host that an exception has occurred */
    remcomOutBuffer[0] = 'T';
    remcomOutBuffer[1] =  hexchars[sigval >> 4];
    remcomOutBuffer[2] =  hexchars[sigval % 16];
    strcpy (&remcomOutBuffer[3], "thread:");
    remcomOutBuffer[10] =  hexchars[lthread >> 4];
    remcomOutBuffer[11] =  hexchars[lthread % 16];
    strcpy (&remcomOutBuffer[12], ";");
    putpacket(remcomOutBuffer);
  } else {
    putDebugChar ('+');
    first_breakpoint = 0;
    otherops_thread = lthread;
  }

  while (1==1) {
    error = 0;
    remcomOutBuffer[0] = 0;
    getpacket(remcomInBuffer);
    switch (remcomInBuffer[0]) {
      case '?' : /* query: why are we stopped? */
                 remcomOutBuffer[0] = 'T';
                 remcomOutBuffer[1] =  hexchars[sigval >> 4];
                 remcomOutBuffer[2] =  hexchars[sigval % 16];
                 strcpy (&remcomOutBuffer[3], "thread:");
                 remcomOutBuffer[10] =  hexchars[lthread >> 4];
                 remcomOutBuffer[11] =  hexchars[lthread % 16];
                 strcpy (&remcomOutBuffer[12], ";");
                 break;
      case 'd' : remote_debug = !(remote_debug);  /* toggle debug flag */
                 break;
      case 'g' : /* return the value of the CPU registers */
                if (lthread == otherops_thread) {
                  mem2hex((char*) registers, remcomOutBuffer, NUMREGBYTES, 0);
                  break;
                } else {
                /* cannot get the other thread's registers without cancelling
                   any pending IPC operation that it's doing, so just get 
                   eip, esp and eflags and return rubbish for the rest */
                  unsigned oth_eip, oth_esp, oth_eflags, oth_ebp;
                  int i;
                  l4_threadid_t other, pager, preempter;
                  pager.dw = preempter.dw = -1;
                  other = threadid;
                  other.id.lthread = otherops_thread;
                  l4_thread_ex_regs (other, -1, -1, &pager, &preempter,
                                     &oth_eflags, &oth_eip, &oth_esp);
                  if (oth_eip == (unsigned) &waiting_for_exception_2b_handled)
                  {
                    /* thread is blocked waiting for an exception to be handled
                     * so all its registers have already been saved */
                    mem2hex((char*) &threadregisters[16*otherops_thread],
                            remcomOutBuffer, NUMREGBYTES, 0);
                    break;
                  }
                  for (i = 0; i < ESP*8; i++)
                     remcomOutBuffer[i] = '0';
	          mem_fault_routine = set_mem_err; /* might cause memfault */
                  mem2hex((char*) &oth_esp, &remcomOutBuffer[i], 4, 0); i+= 8;
                  if (*(unsigned char*)(oth_eip-2) == 0xcd
                      && (*(unsigned char*)(oth_eip-1) & 0xf0) == 0x30)
                    /* if instr = "int" */
                    oth_ebp = *(unsigned *) oth_esp; /* guess ebp!! */
                  else
                    oth_ebp = 0; /* don't even try to guess ebp */
                  mem_fault_routine = NULL; /* shouldn't cause memfault now */
                  mem2hex((char*) &oth_ebp, &remcomOutBuffer[i], 4, 1);
                  for (i = (EBP+1)*8; i < PC*8; i++)
                     remcomOutBuffer[i] = '0';
                  mem2hex((char*) &oth_eip, &remcomOutBuffer[i], 4, 0); i += 8;
                  mem2hex((char*) &oth_eflags, &remcomOutBuffer[i], 4, 0);
                  for (i = (PS+1)*8; i < 16*8; i++)
                     remcomOutBuffer[i] = '0';
                  remcomOutBuffer[i] = '\0';
                }
                
                break;
      case 'G' : /* set the value of the CPU registers - return OK */
                if (lthread == otherops_thread) {
                  hex2mem(&remcomInBuffer[1], (char*)registers, NUMREGBYTES,0);
                  strcpy(remcomOutBuffer,"OK");
                } else /* unimplemented */
                  strcpy(remcomOutBuffer,"E01");
                break;
      case 'P' : /* set the value of a single CPU register - return OK */
                {
                  int regno;

                  if (otherops_thread != lthread) {
                      /* hmm dangerous, the other thread is not stopped */
                      /* we only support changing eip or esp */
                      l4_threadid_t other, preempter, pager;
                      unsigned dum;
                      other = threadid;
                      other.id.lthread = otherops_thread;
                      preempter.dw = pager.dw = -1;
                      ptr = &remcomInBuffer[1];
                      if (hexToInt (&ptr, &regno) && *ptr++ == '=') {
                        if (regno == 8) {
                          hex2mem (ptr, (char *)a, 4, 0);
                          l4_thread_ex_regs (other, a, -1, &preempter, &pager,
                                             &dum, &dum, &dum);
                          break;
                        } else if (regno == 4) {
                          hex2mem (ptr, (char *)a, 4, 0);
                          l4_thread_ex_regs (other, -1, a, &preempter, &pager,
                                             &dum, &dum, &dum);
                          break;
                        }
                      }
                  } else {
                      ptr = &remcomInBuffer[1];
                      if (hexToInt (&ptr, &regno) && *ptr++ == '=') 
                      if (regno >= 0 && regno < NUMREGS)
                        {
                          hex2mem (ptr, (char *)&registers[regno], 4, 0);
                          strcpy(remcomOutBuffer,"OK");
                          break;
                        }

                      strcpy (remcomOutBuffer, "E01");
                      break;
                  }
                }

      /* mAA..AA,LLLL  Read LLLL bytes at address AA..AA */
      case 'm' :
		    /* TRY TO READ %x,%x.  IF SUCCEED, SET PTR = 0 */
                    ptr = &remcomInBuffer[1];
                    if (hexToInt(&ptr,&addr))
                        if (*(ptr++) == ',')
                            if (hexToInt(&ptr,&length))
                            {
                                ptr = 0;
				mem_err = 0;
                                mem2hex((char*) addr, remcomOutBuffer, length, 1);
				if (mem_err) {
				    strcpy (remcomOutBuffer, "E03");
				    debug_error ("memory fault");
				}
                            }

                    if (ptr)
                    {
		      strcpy(remcomOutBuffer,"E01");
		      debug_error("malformed read memory command: %s",remcomInBuffer);
		    }
	          break;

      /* MAA..AA,LLLL: Write LLLL bytes at address AA.AA return OK */
      case 'M' :
		    /* TRY TO READ '%x,%x:'.  IF SUCCEED, SET PTR = 0 */
                    ptr = &remcomInBuffer[1];
                    if (hexToInt(&ptr,&addr))
                        if (*(ptr++) == ',')
                            if (hexToInt(&ptr,&length))
                                if (*(ptr++) == ':')
                                {
				    mem_err = 0;
                                    hex2mem(ptr, (char*) addr, length, 1);

				    if (mem_err) {
					strcpy (remcomOutBuffer, "E03");
					debug_error ("memory fault");
				    } else {
				        strcpy(remcomOutBuffer,"OK");
				    }

                                    ptr = 0;
                                }
                    if (ptr)
                    {
		      strcpy(remcomOutBuffer,"E02");
		      debug_error("malformed write memory command: %s",remcomInBuffer);
		    }
                break;

     case 'B': /* (information) gonna set breakpoint */
         if (num_postponed_bpt_clears > MAX_POSTPONED_BPT_CLEARS-1) {
           strcpy (remcomOutBuffer,"E01");
           break;
         }
         ptr = &remcomInBuffer[1];
         if (!hexToInt (&ptr, &addr) || *ptr++ != ',') {
           strcpy (remcomOutBuffer,"E02");
           break;
         }
         if (!hexToInt (&ptr, &b)) {
           strcpy (remcomOutBuffer,"E03");
           break;
         }
         postponed_bpt_clears[num_postponed_bpt_clears].addr = addr;
         postponed_bpt_clears[num_postponed_bpt_clears++].origval = b;
         strcpy (remcomOutBuffer,"OK");
         break;
     case 'b': /* clear breakpoint */
         /* we don't actually clear the breakpoint yet, we clear it lazily
            at the next step/continue (if it hasn't been reset)
            cos we've got other threads in the bg which we want to still
            stop on this breakpoint */
         ptr = &remcomInBuffer[1];
         if (!hexToInt (&ptr, &addr) || *ptr++ != ',') {
           strcpy (remcomOutBuffer,"E02");
           break;
         }
         if (!hexToInt (&ptr, &b)) {
           strcpy (remcomOutBuffer,"E03");
           break;
         }
         for (a = 0; a < num_postponed_bpt_clears; a++)
           if (postponed_bpt_clears[a].addr == addr) {
             num_postponed_bpt_clears --;
             if (a < num_postponed_bpt_clears) {
               memcpy (&postponed_bpt_clears[a], &postponed_bpt_clears[a+1],
                       (num_postponed_bpt_clears - a)
                        * sizeof(postponed_bpt_clear_t));
               a--;
             }
           }
         strcpy (remcomOutBuffer,"OK");
         break;
     case 'c' : /* continue */
     case 's' : /* step one instruction */
         /* try to read optional parameter, pc unchanged if no parm */
         ptr = &remcomInBuffer[1];
         if (hexToInt(&ptr,&addr))
             registers[ PC ] = addr;

         newPC = registers[ PC];

         /* clear the trace bit */
         registers[ PS ] &= 0xfffffeff;

         /* set the trace bit if we're stepping */
         if (remcomInBuffer[0] == 's') registers[ PS ] |= 0x100;

         repair_bpt_early[lthread] = 0;
         for (a = 0; a < num_postponed_bpt_clears; a++) 
           if (postponed_bpt_clears[a].addr == registers[PC]
               && (registers[PS] & 0x100)) {
             /* we're stepping over a bpt - let's repair it ASAP */
             set_char ((unsigned char *) postponed_bpt_clears[a].addr,
                       postponed_bpt_clears[a].origval);
             repair_bpt_early[lthread] = postponed_bpt_clears[a].addr;
           }

         if (!repair_bpt_early[lthread]) {
           /* ok, remaining breakpoints really were meant to be cleared */
           for (a = 0; a < num_postponed_bpt_clears; a++)
             set_char ((unsigned char *) postponed_bpt_clears[a].addr,
                       postponed_bpt_clears[a].origval);
           num_postponed_bpt_clears = 0;
         }

         return; /* from exception */

      /* kill the program */
      case 'k' :
          {
              l4_threadid_t our_rm;
              sm_exc_t exc;
              our_rm = l4_myself();
              our_rm.id.lthread = 0;
              flushDebugOutput();
              rm_pgr_exit((sm_service_t)our_rm.dw, &exc);
              halt();
              break;
          }
      case 'T' : /* query whether particular thread is alive? */
          ptr = &remcomInBuffer[1];
          if (!hexToInt(&ptr,&a))
              strcpy (remcomOutBuffer, "E01");
          else if (a > 0 && a < 64) {
              if ((alive_threads & (1ULL << a)) != 0ULL)
                strcpy (remcomOutBuffer, "OK");
              else
                strcpy (remcomOutBuffer, "E02");
          } else
              strcpy (remcomOutBuffer, "E03");
          break;
      case 'H' : /* set thread */
          if (remcomInBuffer[1] == 'c') { /* set thr for step/continue */
              break; /* unimplemented */
          } else if (remcomInBuffer[1] == 'g') { /* set thr for other ops */
              ptr = &remcomInBuffer[2];
              if (!hexToInt(&ptr,&a))
                  strcpy (remcomOutBuffer, "E02");
              else if (a >= 0 && a < 64 && 
                       (!a||(alive_threads & (1ULL << a)) != 0ULL)) {
                  otherops_thread = a ? a : lthread;
                  strcpy (remcomOutBuffer, "OK");
              } else
                  strcpy (remcomOutBuffer, "E03");
          } else
            strcpy (remcomOutBuffer, "E01");
      case 'q' : /* query */
        switch(remcomInBuffer[1]) {
          case 'P': /* threadinfo */
#define TAG_THREADID 1      /* Echo the thread identifier */
#define TAG_EXISTS 2        /* Is this process defined enough to
                               fetch registers and its stack */
#define TAG_DISPLAY 4       /* A short thing maybe to put on a window */
#define TAG_THREADNAME 8    /* string, maps 1-to-1 with a thread is */
#define TAG_MOREDISPLAY 16  /* Whatever the kernel wants to say about 
                               the process*/
            ptr = &remcomInBuffer[2];
            hexToIntn(&ptr,&a,8);  /* TAG mask */
            hexToIntn(&ptr,&b,16); /* threadid */
            if (b <= 0 && b >= 64) {
              strcpy (remcomOutBuffer, "E01");
              break;
            }
            if (user_getinfo_fn == NULL)
              a &= TAG_EXISTS | TAG_THREADID;
            else
              a &= TAG_EXISTS | TAG_THREADID | TAG_DISPLAY | TAG_THREADNAME
                   | TAG_MOREDISPLAY;
            remcomOutBuffer[0] = 'q';
            remcomOutBuffer[1] = 'Q'; /* not sure what this char should be */
            intToHex(&remcomOutBuffer[2], a, 8);
            intToHex(&remcomOutBuffer[10], b, 16);
            thr = threadid;
            thr.id.lthread = b;
            ptr = &remcomOutBuffer[26];
            if (a & TAG_THREADID) {
              intToHex(ptr, TAG_THREADID, 8); ptr += 8;
              intToHex(ptr, 16, 2); ptr += 2;
              intToHex(ptr, b, 16); ptr += 16;
            }
            if (a & TAG_EXISTS) {
              intToHex(ptr, TAG_EXISTS, 8);  ptr += 8;
              intToHex(ptr, 1, 2); ptr += 2;
              intToHex(ptr, ((alive_threads & (1ULL << b)) != 0ULL), 1);
              ptr += 1;
            }
            if (a & TAG_DISPLAY) {
              const char *str;
              int len;
              intToHex(ptr, TAG_DISPLAY, 8);  ptr += 8;
              str = (*user_getinfo_fn) (thr, TAG_DISPLAY);
              len = strlen(str);
              if (ptr + len - remcomOutBuffer >  BUFMAX - 23)
                len = BUFMAX + remcomOutBuffer - ptr - 20;
              intToHex(ptr, len, 2);  ptr += 2;
              memcpy(ptr, str, len);  ptr += len;
            }
            if (a & TAG_THREADNAME) {
              const char *str;
              int len;
              intToHex(ptr, TAG_THREADNAME, 8);  ptr += 8;
              str = (*user_getinfo_fn) (thr, TAG_THREADNAME);
              len = strlen(str);
              if (ptr + len - remcomOutBuffer >  BUFMAX - 13)
                len = BUFMAX + remcomOutBuffer - ptr - 10;
              intToHex(ptr, len, 2);  ptr += 2;
              memcpy(ptr, str, len);  ptr += len;
            }
            if (a & TAG_MOREDISPLAY) {
              const char *str;
              int len;
              intToHex(ptr, TAG_MOREDISPLAY, 8);  ptr += 8;
              str = (*user_getinfo_fn) (thr, TAG_MOREDISPLAY);
              len = strlen(str);
              if (ptr + len - remcomOutBuffer >  BUFMAX - 3)
                len = BUFMAX + remcomOutBuffer - ptr - 1;
              intToHex(ptr, len, 2);  ptr += 2;
              memcpy(ptr, str, len);  ptr += len;
            }
            *ptr = '\0';
            break;
          case 'L': /* threadlist */
            ptr = &remcomInBuffer[2];
            hexToIntn(&ptr,&a,1); /* startflag, ignored */
            hexToIntn(&ptr,&a,2); /* max number of threads in our reply */
            hexToIntn(&ptr,&b,16); /* starting threadid */
            if (b < 0)  b = 0;
            remcomOutBuffer[0] = 'q';
            remcomOutBuffer[1] = 'M';
            ptr = &remcomOutBuffer[5];
            intToHex(ptr, b, 16);  ptr += 16;
            c = 0; /* count of active threads found */
            for (i = b; i < 64; i++) {
              if ((alive_threads & (1ULL << i)) != 0ULL) {
                intToHex(ptr, i, 16);  ptr += 16;
                if (++c >= a)  break; /* reply size limit reached */
              }
            }
            *ptr = '\0';
            intToHex(&remcomOutBuffer[2], c, 2);
            intToHex(&remcomOutBuffer[4], i==64, 1);
            break;
          case 'C': /* query current thread */
            remcomOutBuffer[0] = 'Q';
            remcomOutBuffer[1] = 'C';
            /* should this report lthread or otherops_thread? */
            intToHex(&remcomOutBuffer[2], lthread, 3);
            remcomOutBuffer[5] = '\0';
            break;
        } /* switch */
        break;
      } /* switch */

    /* reply to the request */
    putpacket(remcomOutBuffer);
    }
}

void set_debug_traps();
static void gdbstub_thread(int creator);

void gdbstub_thread_startstop_fn (l4_threadid_t threadid, int starting)
{
  static int initialised = 0;
  char need_to_init;
  asm ("btsl $0,%0\n\t"
       "setnc %1"
       : "=m"(initialised), "=r"(need_to_init) : "0"(initialised)
  );
  if (need_to_init) {
    l4_threadid_t gdbstub;
    l4_msgdope_t result;
    unsigned dum;
    create_thread_with_param (GDBSTUB_LTHREAD, gdbstub_thread, stackPtr,
                              L4_INVALID_ID, threadid.dw);
    gdbstub = threadid;
    gdbstub.id.lthread = GDBSTUB_LTHREAD;
    l4_i386_ipc_receive (gdbstub, L4_IPC_SHORT_MSG, &dum, &dum, &dum,
                         L4_IPC_NEVER, &result);
  }
  if (starting) {
    alive_threads |= (1ULL << threadid.id.lthread);
    set_debug_traps();
  } else
    alive_threads &=~ (1ULL << threadid.id.lthread);
}

void gdbstub_init_thread_debugging(unsigned *startstop_fn, unsigned getinfo_fn)
{
  *startstop_fn = (unsigned) gdbstub_thread_startstop_fn;
  user_getinfo_fn = (void *) getinfo_fn;
}

/* this function is used to set up exception handlers for tracing and
   breakpoints */
void set_debug_traps()
{
extern void _remcomHandler();

  stackPtr  = &remcomStack[STACKSIZE/sizeof(int) - 2];

  exceptionHandler (0, _catchException0);
  exceptionHandler (1, _catchException1);
  exceptionHandler (3, _catchException3);
  exceptionHandler (4, _catchException4);
  exceptionHandler (5, _catchException5);
  exceptionHandler (6, _catchException6);
  exceptionHandler (7, _catchException7);
  exceptionHandler (8, _catchException8);
  exceptionHandler (9, _catchException9);
  exceptionHandler (10, _catchException10);
  exceptionHandler (11, _catchException11);
  exceptionHandler (12, _catchException12);
  exceptionHandler (13, _catchException13);
  exceptionHandler (14, _catchException14);
  exceptionHandler (16, _catchException16);

  if (exceptionHook != _remcomHandler)
  {
      oldExceptionHook = exceptionHook;
      exceptionHook    = _remcomHandler;
  }

  /* In case GDB is started before us, ack any packets (presumably
     "$?#xx") sitting there.  */

  initialized = 1;

}

/* This function will generate a breakpoint exception.  It is used at the
   beginning of a program to sync up with a debugger and can be used
   otherwise as a quick means to stop program execution and "break" into
   the debugger. */

void breakpoint()
{
  if (initialized)
#if 0
    handle_exception(3);
#else
    BREAKPOINT();
#endif
  waitabit();
}

int waitlimit = 1000000;

void
waitabit()
{
  int i;
  for (i = 0; i < waitlimit; i++) ;
}
