#include <services/lib/threads.h>

queue_head_t monitors;

l4_threadid_t caller, serializer = L4_INVALID_ID;
dword_t snd_msg = L4_IPC_NIL_DESCRIPTOR;
l4_threadid_t l4_invthread = L4_INVALID_ID;
l4_threadid_t this_task, this_premp, this_pager;

/* For i386_threads.S */
#ifndef __L4_VERSION_X__
dword_t serializer_high, serializer_low;
#endif /* __L4_VERSION_X__ */

dword_t w0_glob, w1_glob;
dword_t dummy;
l4_msgdope_t dummydope;

int used_tcbs = 0,
    highest_used_tcb = -1;

/*
 * l4_threadid_t x thread_t
 */
l4_threadid_t
l4_thread(thread_t t) {
    this_task.id.lthread = t;
    return this_task;
}

#define thread(t) ((thread_t)(t).id.lthread)

/*
 * Serializer
 */
thread_t
thk_caller(void) {
    return thread(caller);
}

void
thk_let_caller_sleep(void) {
    snd_msg = L4_IPC_NIL_DESCRIPTOR;
    tcb(thk_caller()).eip = 0;
}

void
thk_return_to_caller(dword_t r0) {
    snd_msg = 0;
    w0_glob = r0;
}

void
thk_wakeup(thread_t t) {
    int rc;
    dword_t w0 = tcb(t).eip, w1 = 0, dummy;

#ifdef __L4_VERSION_X__
    rc = l4_i386_ipc_send(l4_thread(t), 0, w0, w1, dummy,
                          L4_IPC_NEVER, &dummydope);
#else /* __L4_VERSION_X__ */
    rc = l4_i386_ipc_send(l4_thread(t), 0, w0, w1,
                          L4_IPC_NEVER, &dummydope);
#endif /* __L4_VERSION_X__ */
    if (!(rc == 0 || rc == L4_IPC_SETIMEOUT)) {
        printf("\nthk_wakeup: send to 0x%x failed with 0x%x", t, rc);
        enter_kdebug("thk_wakeup");
    }
}

int
exp(int y, int x) {
    int result = y;

    while (--x) result *= y;
    return result;
}

void
calc_man_exp(int millis, int *m, int *e) {
    if (millis < 0)
        *e = 0;
    else if (millis <= 261)
        *e = 10;
    else if (millis <= 67 * 1000)
        *e = 6;
    else if (millis <= 55 * 360000)
        *e = 2;
    else
        *e = 0;

    if (*e > 0)
        *m = (millis * 1000) / exp(4, 15 - *e);
    else
        *m = 0;
}

dword_t
thk_call(void (*proc)(dword_t), dword_t param, int millis) {
    int rc, r_m, r_e;
    dword_t w0 = (dword_t)proc, w1 = param, dummy;

    calc_man_exp(millis, &r_m, &r_e);
#ifdef __L4_VERSION_X__
    rc = l4_i386_ipc_call(serializer, 0, w0, w1, dummy,
			  0, &w0, &dummy, &dummy,
                          L4_IPC_TIMEOUT(0, 0, r_m, r_e, 0, 0),
                          &dummydope);
#else /* __L4_VERSION_X__ */
    rc = l4_i386_ipc_call(serializer, 0, w0, w1, 0, &w0, &dummy,
                          L4_IPC_TIMEOUT(0, 0, r_m, r_e, 0, 0),
                          &dummydope);
#endif /* __L4_VERSION_X__ */
    if (proc == (void (*)(dword_t))thk_wait &&
        rc == L4_IPC_RETIMEOUT) {
        w0 = (dword_t)thk_enter_after_wait;
#ifdef __L4_VERSION_X__
        rc = l4_i386_ipc_call(serializer, 0, w0, w1, dummy,
			      0, &w0, &w1, &dummy,
                              L4_IPC_NEVER, &dummydope);
#else /* __L4_VERSION_X__ */
        rc = l4_i386_ipc_call(serializer, 0, w0, w1, 0, &w0, &w1,
                              L4_IPC_NEVER, &dummydope);
#endif /* __L4_VERSION_X__ */
        if (rc) {
            printf("\nltk_call: thk_enter_after_wait failed with 0x%x", rc);
            enter_kdebug("thk_call");
        }
    } else if (rc) {
        printf("\nthk_call: call failed with 0x%x", rc);
        enter_kdebug("thk_call");
    }
    return w0;
}

char *thk_func() {
    if (w0_glob == (dword_t)thk_create)
         return "thk_create";
    else if (w0_glob == (dword_t)thk_exit_self)
         return "thk_exit_self";
    else if (w0_glob == (dword_t)thk_suspend)
         return "thk_suspend";
    else if (w0_glob == (dword_t)thk_susp_carefully)
        return "thk_susp_carefully";
    else if (w0_glob == (dword_t)thk_resume)
        return "thk_resume";
    else if (w0_glob == (dword_t)thk_enter)
        return "thk_enter";
    else if (w0_glob == (dword_t)thk_enter_after_wait)
        return "thk_enter_after_wait";
    else if (w0_glob == (dword_t)thk_exit)
        return "thk_exit";
    else if (w0_glob == (dword_t)thk_wait)
        return "thk_wait";
    else {
        printf("\nthk_func: w0_glob = 0x%x, no such thk function", w0_glob);
        enter_kdebug("thk_func");
    }
}

void
dump_monitors() {
    mon_t mon;
    tcb_t tcb;

    printf("\nMonitor dump--monitor owner count (ext queue, int queue)");
    queue_iterate(&monitors, mon, mon_t, monq) {
        if (mon->owner == TH_NIL_ID &&
            mon->count == 0 &&
            queue_empty(&mon->ext_queue) &&
            queue_empty(&mon->int_queue))
            continue;
        printf("\n0x%07x 0x%08x %d", mon, mon->owner, mon->count);
        printf(" (");
        queue_iterate(&mon->ext_queue, tcb, tcb_t, monq) {
            printf("0x%x ", tcb_th(tcb));
        }
        printf(",");
        queue_iterate(&mon->int_queue, tcb, tcb_t, monq) {
            printf("0x%x ", tcb_th(tcb));
        }
        printf(")");
    }
}

void
thk_serialize(void) {
    int rc;
    dword_t dummy;

    snd_msg = L4_IPC_NIL_DESCRIPTOR;
    while (1) {
#ifdef __L4_VERSION_X__
        rc = l4_i386_ipc_reply_and_wait(
                 caller, (void *)snd_msg, w0_glob, w1_glob, dummy,
                 &caller, 0, &w0_glob, &w1_glob, &dummy,
                 L4_IPC_TIMEOUT(0, 1, 0, 0, 0, 0), /* snd 0, rcv inf */
                 &dummydope);
#else /* __L4_VERSION_X__ */
        rc = l4_i386_ipc_reply_and_wait(
                 caller, (void *)snd_msg, w0_glob, w1_glob,
                 &caller, 0, &w0_glob, &w1_glob,
                 L4_IPC_TIMEOUT(0, 1, 0, 0, 0, 0), /* snd 0, rcv inf */
                 &dummydope);
#endif /* __L4_VERSION_X__ */
        if (!rc) ((void (*)(dword_t))w0_glob)(w1_glob);
        else thk_let_caller_sleep();
    }
}

long long serializer_stack[L4_PAGESIZE / sizeof(long long)];
unsigned serializer_stack_size = sizeof(serializer_stack);
dword_t t_no = 0;
#ifdef __L4_VERSION_X__
dword_t premp_high, premp_low;
#else /* __L4_VERSION_X__ */
dword_t pager_high, pager_low;
#endif /* __L4_VERSION_X__ */

extern void _start, _end, _stack;

/*
 * threads API (th_*)
 */
thread_t
th_init() {
    int i;

    queue_init(&monitors);

    /* Initialize this_* globals */
    this_task = l4_myself();
    if (this_task.id.lthread != FIRST_THREAD) {
	printf("\nlt_init: Called by thread 0x%x?",
	       this_task.id.lthread);
        enter_kdebug("th_init");
    }
    this_premp = L4_INVALID_ID;
    this_pager = L4_INVALID_ID;
    l4_thread_ex_regs(this_task, 0xffffffff, 0xffffffff,
                      &this_premp, &this_pager,
                      &dummy, &dummy, &dummy);
#ifndef __L4_VERSION_X__
    premp_high = this_premp.lh.high;
    premp_low = this_premp.lh.low;
    pager_high = this_pager.lh.high;
    pager_low = this_pager.lh.low;
#endif /* __L4_VERSION_X__ */

    /* Initialize tcb array */
    for (i = 0; i < MAX_TCBS; i++)
        thk_tcb[i].susp_depth = -1;

    /* Initialize serializer's tcb */
    use_tcb((thread_t)SERIALIZER);
    serializer = l4_thread(SERIALIZER);
#ifndef __L4_VERSION_X__
    serializer_high = serializer.lh.high;
    serializer_low = serializer.lh.low;
#endif /* __L4_VERSION_X__ */
    tcb(SERIALIZER).stack_low = (dword_t)serializer_stack;
    tcb(SERIALIZER).stack_high = (dword_t)((char *)serializer_stack +
					   serializer_stack_size - 4);

    use_tcb((thread_t)REGION_MAPPER);
    use_tcb((thread_t)EMUL_LIB);
    use_tcb((thread_t)GDB_THREAD);
    use_tcb((thread_t)RESERVED_1);

    /* Initialize first thread's tcb.  Note the hardcoded stack!!!  */
    if (new_thread(TH_NIL_ID) != FIRST_THREAD) {
        printf("\nlt_init: Did not create thread %d", FIRST_THREAD);
        enter_kdebug("lt_init");
    }
    tcb(FIRST_THREAD).stack_low = (dword_t)&_stack - 4096;
    tcb(FIRST_THREAD).stack_high = (dword_t)&_stack;

    l4_thread_ex_regs(this_task, (dword_t)thk_serialize,
		      (dword_t)tcb(SERIALIZER).stack_high,
		      &this_premp, &this_pager,
		      &dummy, &dummy, &dummy);
    l4_thread_switch(serializer);

    return FIRST_THREAD;

#if 0
    /* Start first thread with lthread_ex_regs().  Now broken!!! */
    asm(
        "pushal                \n\t"
        "movl $1, %eax         \n\t" /* Thread 1 */
        "movl %esp, %ecx       \n\t" /* esp */
        "movl $1f, %edx        \n\t" /* eip */
#ifdef __L4_VERSION_X__
        "movl this_premp, %ebx \n\t" /* Preempter */
        "movl this_pager, %esi \n\t" /* Pager*/
#else /* __L4_VERSION_X__ */
        "movl premp_high, %ebp \n\t" /* Preempter */
        "movl premp_low, %ebx  \n\t"
        "movl pager_high, %edi \n\t" /* Pager */
        "movl pager_low, %esi  \n\t"
#endif /* __L4_VERSION_X__ */
        "int $0x35             \n\t"
        "1:                    \n\t"
        "cmpl $0, %eax         \n\t" /* Get the thread number without */
        "popal                 \n\t" /* touching the stack, thread 1's */
        "jne 2f                \n\t" /* eax will be 0 */
        "movl $1, t_no         \n\t"
        "2:                    \n\t"
    );

    if (t_no == SERIALIZER) {
        /* Switch to serializer's stack */
        asm("movl %0, %%esp" : : "r" (tcb(0).stack_high));
        ltk_serialize();
        enter_kdebug("lt_init: Serializer returned");
    } else {
        return (lthread_t)(FIRST_THREAD);
    }
#endif /* 0 */
}

thread_t
th_create(thread_t t, dword_t stack_low, dword_t stack_high,
          dword_t eip, dword_t back_ptr) {
    t = thk_call((void (*)(dword_t))thk_create, (dword_t)t, -1);
    /* alloc_stack() cannot be called by the serializer */
    if (t != TH_NIL_ID) {
        tcb(t).stack_low = stack_low;
        tcb(t).stack_high = stack_high;
        tcb(t).eip = eip;
        tcb(t).back_ptr = back_ptr;
        tcb(t).suspend = 0;
        tcb(t).susp_depth = 1;
        tcb(t).mon_wait = 0;
        tcb(t).mon_count = 1;
        tcb(t).priority = -1;
        if (tcb(t).stack_high == 0) alloc_stack(t, tcb(t).stack_low);
        /* Need to push the back pointer as an argument */
        *(dword_t *)(tcb(t).stack_high) = tcb(t).back_ptr;
        l4_thread_ex_regs(l4_thread(t), (dword_t)wait_for_resume,
                          tcb(t).stack_high - 4,
                          &this_premp, &this_pager,
                          &dummy, &dummy, &dummy);
        l4_thread_switch(l4_thread(t));
    }
    return t;
}

void
th_exit_self(void) {
    thk_call((void (*)(dword_t))thk_exit_self, (dword_t)0, -1);
}

int
th_suspend(thread_t t) {
    return thk_call((void (*)(dword_t))thk_suspend, (dword_t)t, -1);
}

int
th_susp_carefully(thread_t t) {
    return thk_call((void (*)(dword_t))thk_susp_carefully, (dword_t)t, -1);
}

int
th_suspended(thread_t t) {
    return (tcb(t).susp_depth > 0);
}

int
th_resume(thread_t t) {
    return thk_call((void (*)(dword_t))thk_resume, (dword_t)t, -1);
}

dword_t
th_stack(thread_t t, int type) {
    dword_t dummy, sp;

    if (!valid_used(t)) return SYS_ERR;

    if (type == STACK_POINTER) {
        l4_thread_ex_regs(l4_thread(t), 0xffffffff, 0xffffffff,
                          &l4_invthread, &l4_invthread,
                          &dummy, &dummy, &sp);
    } else if (type == STACK_LOW) {
        sp = tcb(t).stack_low;
    } else if (type == STACK_HIGH) {
        sp = tcb(t).stack_high;
    }

    return sp;
}

int
th_back_ptr(thread_t t, dword_t *bp) {
    if (!valid_used(t)) return SYS_ERR;
    if (*bp == 0xffffffff) *bp = tcb(t).back_ptr;
    else tcb(t).back_ptr = *bp;
    return SYS_OK;
}

void
th_yield(thread_t t) {
    l4_thread_switch((t == TH_NIL_ID) ? L4_NIL_ID : l4_thread(t));
}

int
set_l4_priority(l4_threadid_t tid, dword_t priority) {
    l4_threadid_t preempter;
    l4_sched_param_t schedparam;

    preempter = L4_INVALID_ID;
    schedparam.sched_param = -1;
    (void)l4_thread_schedule(tid, schedparam, &preempter, &preempter,
			     &schedparam);
    if (schedparam.sched_param == 0xffffffff) {
        printf("set_l4_priority: Get failed");
        enter_kdebug("set_l4_priority");
    }
    schedparam.sp.prio = priority;
    preempter = L4_INVALID_ID;
    (void)l4_thread_schedule(tid, schedparam, &preempter,
                             &preempter, &schedparam);
    if (schedparam.sched_param == 0xffffffff) {
        printf("set_l4_priority: Set failed");
        enter_kdebug("set_l4_priority");
    }
    return SYS_OK;
}

int
th_priority(thread_t t, dword_t *priority) {
    if (!valid_used(t)) return SYS_ERR;
    if (*priority == 0xffffffff) {
	*priority = tcb(t).priority;
    } else {
	tcb(t).priority = *priority;
#if 0
	(void)set_l4_priority(l4_thread(t), tcb(t).priority);
#endif /* 0 */
    }
    return SYS_OK;
}

/*
 * Begin thk serialized
 */
void
thk_create(thread_t t) {
    if (!valid_unused(t)) t = new_thread(t);
    thk_return_to_caller(t);
}

void
thk_exit_self(int dummy) {
    thread_t me = thk_caller();
    unuse_tcb(me);
    thk_let_caller_sleep();
}

void
thk_suspend(thread_t t) {
    if (!valid_used(t)) {
        thk_return_to_caller(SYS_ERR);
    } else if (++tcb(t).susp_depth == 1 && t != thk_caller()) {
        dword_t eip = (dword_t)wait_for_resume;

        l4_thread_ex_regs(l4_thread(t), eip, 0xffffffff,
                          &l4_invthread, &l4_invthread,
                          &dummy, &tcb(t).eip, &dummy);
    }
    thk_return_to_caller(tcb(t).susp_depth);
    if (t == thk_caller()) thk_let_caller_sleep();
}

void
thk_susp_carefully(thread_t t) {
    if (!valid_used(t)) {
        thk_return_to_caller(SYS_ERR);
    } else {
        thk_suspend(t);
        if (tcb(t).mon_depth > 0) {
            tcb(t).suspend = 1;
            thk_resume(t);
            thk_return_to_caller(SYS_DELAYED);
        }
    }
}

void
thk_resume(thread_t t) {
    if (!valid_used(t) ||
        t == thk_caller() ||
        tcb(t).susp_depth == 0) {
        thk_return_to_caller(SYS_ERR);
    } else {
        if (--tcb(t).susp_depth == 0) thk_wakeup(t);
        thk_return_to_caller(tcb(t).susp_depth);
    }
}
/*
 * End thk serialized
 */

int
valid_used(thread_t t) {
    return (t >= 0 &&
            t <= highest_used_tcb &&
            t != SERIALIZER &&
            tcb(t).susp_depth >= 0);
}

int
valid_unused(thread_t t) {
    return (t >= 0 &&
            t < MAX_TCBS &&
            (t > highest_used_tcb || tcb(t).susp_depth < 0));
}

thread_t
th_thread_self() {
    dword_t sp;
    int i = 0;
    l4_threadid_t l4_t;

    asm("movl %%esp, %0" : "=r" (sp));

    while (i <= highest_used_tcb) {
        if (valid_used(i) &&
            sp >= tcb(i).stack_low && sp <= tcb(i).stack_high) {
	    if (thread(l4_myself()) != i) enter_kdebug("th_thread_self");
            return (thread_t)i;
	}
        i++;
    }

    enter_kdebug("th_thread_self: Not a thread");
    return TH_NIL_ID;
}

thread_t
new_thread(thread_t t) {
    if (t == TH_NIL_ID) t = free_tcb();
    use_tcb(t);
    return t;
}

void
alloc_stack(thread_t t, int size) {
    tcb(t).stack_low = (dword_t)malloc(size);
    tcb(t).stack_high = tcb(t).stack_low + size - 4;
}

/*
 * TCB allocation
 */
thread_t
free_tcb() {
    int t, i;

    if (used_tcbs >= MAX_TCBS) return TH_NIL_ID;
    for (i = 0; i < MAX_TCBS; i++) {
        if (valid_unused(i)) {
            t = i;
            break;
        }
    }
    return (thread_t)t;
}

void
use_tcb(thread_t t) {
    int i;

    used_tcbs++;
    if ((int)t > highest_used_tcb) highest_used_tcb = t;
    tcb(t).susp_depth = 0;
}

void
unuse_tcb(thread_t t) {
    int i;

    used_tcbs--;
    tcb(t).susp_depth = -1;

    for (i = MAX_TCBS - 1; i > 0; i--) {
        if (tcb(i).susp_depth != -1) {
            highest_used_tcb = i;
            break;
        }
    }
}

/*
 * Monitor
 *
#define SERIALIZE 0
 */

#ifndef SERIALIZE
static inline int
AtomicCmpXchg(dword_t *x, dword_t y, dword_t z) {
    int rc = 0;

    asm volatile
       ("lock			\n\t"	/* Make sure we're SMP ready? */
        "cmpxchg %%ecx, (%%ebx) \n\t"
        "jne     1f             \n\t"
        "movl    $1, %%edx      \n\t"
        "1:                     \n\t"
        : "=b" (x), "=d" (rc)
        : "b"  (x), "c"  (z), "a" (y), "d" (rc));
    return rc;
}

static inline void
AtomicOr(dword_t *x, dword_t y) {
    asm volatile
       ("lock; orl %%ebx, (%%eax)"	/* Make sure we're SMP ready? */
        : "=a" (x)
        :  "a" (x), "b" (y));
}
#endif /* SERIALIZE */

void
mon_init(mon_t mon) {
    mon->owner = TH_NIL_ID;
    mon->count = 0;
    queue_init(&mon->ext_queue);
    queue_init(&mon->int_queue);
    queue_enter(&monitors, mon, mon_t, monq);
}

int
mon_enter(mon_t mon) {
#ifdef SERIALIZE
    return (thk_call((void (*)(dword_t))thk_enter, (dword_t)mon, -1));
#else /* SERIALIZE */
    thread_t me = th_thread_self();
 
    if ((mon->owner & OWNER_MASK) == me) {
        mon->count++;
    } else {
        if (!AtomicCmpXchg(&mon->owner, TH_NIL_ID, me)) {
            thk_call((void (*)(dword_t))thk_enter, (dword_t)mon, -1);
        } else {
            mon->count = 1;
        }
        if (mon->count != 1) enter_kdebug("mon_enter: count not 1");
    }
    return SYS_OK;
#endif /* SERIALIZE */
}

int
mon_entered(mon_t mon) {
    return ((mon->owner & OWNER_MASK) == th_thread_self());
}

int
mon_exit(mon_t mon) {
#ifdef SERIALIZE
    return (thk_call((void (*)(dword_t))thk_exit, (dword_t)mon, -1));
#else /* SERIALIZE */
    thread_t me = th_thread_self();
 
    if ((mon->owner & OWNER_MASK) != me) {
        enter_kdebug("mon_exit: Not owner");
        return SYS_ERR;
    }
 
    /* Are we really releasing the monitor? */
    mon->count--;
    if (mon->count < 0) enter_kdebug("mon_exit: Negative count");
    if (mon->count == 0) {
        /* If wait queue bit is set, wake up a thread */
        if (!AtomicCmpXchg(&mon->owner, me, TH_NIL_ID))
            thk_call((void (*)(dword_t))thk_exit, (dword_t)mon, -1);
    }

    return SYS_OK;
#endif /* SERIALIZE */
}

int
mon_notify(mon_t mon, int type) {
    thread_t me = th_thread_self();

    if ((mon->owner & OWNER_MASK) == me) {
        if (type == NOTIFY_ONE) mon->owner |= NOTIFY_ONE_MASK;
        else if (type == NOTIFY_ALL) mon->owner |=NOTIFY_ALL_MASK;
        else return SYS_ERR;
        return SYS_OK;
    } else {
        enter_kdebug("mon_notify: Not owner");
        return SYS_ERR;
    }
}

int
mon_wait(mon_t mon, int millis) {
    return thk_call((void (*)(dword_t))thk_wait, (dword_t)mon, millis);
}

/*
 * Begin thk serialized
 */
void
thk_enter(mon_t mon) {
    thread_t me = thk_caller();

#ifdef SERIALIZE
    if (mon->owner == TH_NIL_ID) {
        mon->owner = me;
        if (mon->count != 0)
            enter_kdebug("thk_enter: Monitor count not zero");
	mon->count = 1;
        thk_return_to_caller(SYS_OK);
    } else if ((mon->owner & OWNER_MASK) == me) {
        if (mon->count == 0)
            enter_kdebug("thk_enter: Monitor count zero");
        mon->count++;
        thk_return_to_caller(SYS_OK);
    } else {
        tcb_t me_tcb = &tcb(me);

        execute_notifications(mon);
        queue_enter(&mon->ext_queue, me_tcb, tcb_t, monq);
        me_tcb->queue = EXTERNAL;
        me_tcb->mon_count = 1;
        thk_let_caller_sleep();
    }
#else /* SERIALIZE */ 
    tcb_t tcb = &tcb(me);
 
    AtomicOr(&mon->owner, WAIT_QUEUE_MASK);
    execute_notifications(mon);
    queue_enter(&mon->ext_queue, tcb, tcb_t, monq);
    tcb->queue = EXTERNAL;
    tcb->mon_count = 1;
 
    /* Has monitor been released? */
    if (!AtomicCmpXchg(&mon->owner, TH_NIL_ID, me)) {
        thk_let_caller_sleep();
    } else {
        queue_remove_first(&mon->ext_queue, tcb, tcb_t, monq);
	if (tcb != &tcb(me)) enter_kdebug("mon_enter: Wrong thread");
        tcb->queue = NO_QUEUE;
        mon->owner = queue_empty(&mon->ext_queue) ?
                     me : me | WAIT_QUEUE_MASK;
        mon->count = tcb->mon_count;
        tcb->mon_count = 1;
        thk_return_to_caller(SYS_OK);
    }
#endif /* SERIALIZE */ 
}

void
thk_enter_after_wait(mon_t mon) {
    thread_t me = thk_caller();
    tcb_t me_tcb = &tcb(me);

    if ((mon->owner & OWNER_MASK == me) || me_tcb->queue != INTERNAL) {
        thk_return_to_caller(SYS_OK);
    } else {
#ifdef SERIALIZE
        queue_remove(&mon->int_queue, me_tcb, tcb_t, monq);
        me_tcb->queue = NO_QUEUE;
        me_tcb->mon_wait = 0;

        if (queue_empty(&mon->ext_queue) && mon->owner == TH_NIL_ID) {
            mon->owner = me;
            mon->count = tcb(me).mon_count;
            tcb(me).mon_count = 1;
            ltk_return_to_caller(SYS_OK);
        } else {
            execute_notifications(mon);
            queue_enter(&mon->ext_queue, me_tcb, tcb_t, monq);
            me_tcb->queue = EXTERNAL;
            thk_let_caller_sleep();
        }
#else /* SERIALIZE */
        AtomicOr(&mon->owner, WAIT_QUEUE_MASK);
        execute_notifications(mon);
        queue_remove(&mon->int_queue, me_tcb, tcb_t, monq);
        me_tcb->queue = NO_QUEUE;
        queue_enter(&mon->ext_queue, me_tcb, tcb_t, monq);
        me_tcb->queue = EXTERNAL;
 
        /* Has monitor been released? */
        if (!AtomicCmpXchg(&mon->owner, TH_NIL_ID, me)) {
            thk_let_caller_sleep();
        } else {
            queue_remove_first(&mon->ext_queue, me_tcb, tcb_t, monq);
	    if (me_tcb != &tcb(me))
		enter_kdebug("thk_enter_after_wait: Wrong thread");
            me_tcb->queue = NO_QUEUE;
            mon->count = me_tcb->mon_count;
            me_tcb->mon_count = 1;
            mon->owner = queue_empty(&mon->ext_queue) ?
                         me : me | WAIT_QUEUE_MASK;
            thk_return_to_caller(SYS_OK);
        }
#endif /* SERIALIZE */
    }
}

void
thk_exit(mon_t mon) {
#ifdef SERIALIZE
    thread_t me = thk_caller();

    if ((mon->owner & OWNER_MASK) != me) {
        printf("\nthk_exit: me = 0x%x, mon = 0x%x", me, mon);
        dump_monitors();
        enter_kdebug("ltk_exit");
    }

    mon->count--;
    if (mon->count < 0) {
        printf("\nthk_exit: me = 0x%x", me);
        enter_kdebug("thk_exit, 0: Monitor count negative");
    } else if (mon->count == 0) {
        thread_t t = TH_NIL_ID;
        tcb_t t_tcb;

        execute_notifications(mon);
        if (!queue_empty(&mon->ext_queue)) {
            queue_remove_first(&mon->ext_queue, t_tcb, tcb_t, monq);
            t_tcb->queue = NO_QUEUE;
            t = tcb_th(t_tcb);
        }
        if (t != TH_NIL_ID) {
            mon->count = tcb(t).mon_count;
            if (mon->count < 0)
                enter_kdebug("thk_exit, 1: Monitor count negative");
            tcb(t).mon_count = 1;
        } else {
            mon->count = 0;
        }
        mon->owner = t;
        if (t != TH_NIL_ID) thk_wakeup(t);
    }
    thk_return_to_caller(SYS_OK);
#else /* SERIALIZE */
    thread_t t = TH_NIL_ID;
    tcb_t tcb;
 
    execute_notifications(mon);
    if (!queue_empty(&mon->ext_queue)) {
        queue_remove_first(&mon->ext_queue, tcb, tcb_t, monq);
        tcb->queue = NO_QUEUE;
        t = tcb_th(tcb);
    }
    if (t != TH_NIL_ID) {
        mon->count = tcb(t).mon_count;
        tcb(t).mon_count = 1;
    } else {
        mon->count = 0;
    }
    mon->owner = queue_empty(&mon->ext_queue) ?
                 t : t | WAIT_QUEUE_MASK;
    if (t != TH_NIL_ID) thk_wakeup(t);
    thk_return_to_caller(SYS_OK);
#endif /* SERIALIZE */
}

void
execute_notifications(mon_t mon) {
    tcb_t tcb;

    if (mon->owner == TH_NIL_ID) return;

    if (mon->owner & NOTIFY_ONE_MASK) {
        mon->owner &= ~NOTIFY_ONE_MASK;
        if (!queue_empty(&mon->int_queue)) {
            queue_remove_first(&mon->int_queue, tcb, tcb_t, monq);
            tcb->queue = NO_QUEUE;
            tcb->mon_wait = 0;
            queue_enter(&mon->ext_queue, tcb, tcb_t, monq);
            tcb->queue = EXTERNAL;
        }
    } else if (mon->owner & NOTIFY_ALL_MASK) {
        mon->owner &= ~NOTIFY_ALL_MASK;
        while (!queue_empty(&mon->int_queue)) {
	    queue_remove_first(&mon->int_queue, tcb, tcb_t, monq);
            tcb->queue = NO_QUEUE;
            tcb->mon_wait = 0;
            queue_enter(&mon->ext_queue, tcb, tcb_t, monq);
            tcb->queue = EXTERNAL;
        }
    }
}

void
thk_wait(mon_t mon) {
    thread_t me = thk_caller(), t = TH_NIL_ID;
    tcb_t t_tcb;

    if ((mon->owner & OWNER_MASK) != me)
        enter_kdebug("thk_wait: Not owner");

    tcb(me).mon_count = mon->count;
    if (tcb(me).mon_count <= 0)
        enter_kdebug("thk_wait: Stored monitor count negative/zero");
    queue_enter(&mon->int_queue, &tcb(me), tcb_t, monq);
    tcb(me).queue = INTERNAL;
    tcb(me).mon_wait = (int)mon;
    thk_let_caller_sleep();

    if (!queue_empty(&mon->ext_queue)) {
        queue_remove_first(&mon->ext_queue, t_tcb, tcb_t, monq);
        t_tcb->queue = NO_QUEUE;
        t_tcb->mon_wait = 0;
        t = tcb_th(t_tcb);
    }
    if (t != TH_NIL_ID) {
        mon->count = tcb(t).mon_count;
        tcb(t).mon_count = 1;
    } else {
        mon->count = 0;
    }
#ifdef SERIALIZE
    mon->owner = t;
#else /* SERIALIZE */
    mon->owner = queue_empty(&mon->ext_queue) ?
                 t : t | WAIT_QUEUE_MASK;
#endif /* SERIALIZE */
    if (t != TH_NIL_ID) thk_wakeup(t);
}
/*
 * End thk serialized
 */
