/* $Id: l4_proc.c,v 1.1.1.1 1999/04/06 19:20:57 yoonho Exp $ */

#include <linux/stat.h>
#include <linux/proc_fs.h>
#include <linux/mm.h>

#include <asm/segment.h>
#include <asm/ioctls.h>

#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>

#include "../include/config.h"
#include "../include/exclpage.h"
#include "../include/task.h"
#include "../include/rmgr.h"
#include "../include/drivers.h"

extern int emulib_idt_descr, emulib_debug_idt_descr;

#define NAME_DIR "l4"
#define MODE_DIR 0555

#define NAME_CTL "ctl"
#define MODE_CTL 0666

#define NAME_IDS "ids"
#define MODE_IDS 0444

#define NAME_PERF "perf"
#define MODE_PERF 0444

#define NAME_UNSELECT "unselect"
#define MODE_UNSELECT 0444

/* a number for each special file... */

unsigned long print_switch=0, debug_msr=0; 
enum 
{
  FILE_CTL = 1,
  FILE_IDS,
  FILE_PERF,
  FILE_UNSELECT
};

/* helper routines */

static int lookup_file(struct inode * i)
{
  if (i->i_mode & S_IFDIR)
    return -EISDIR;
  
  if (! i->u.generic_ip)
    return -EISDIR;

  return (int) ((struct proc_dir_entry *) i->u.generic_ip)->data;
}

/* XXX here we allow user tasks to alloc system call numbers, but we never 
       free them after the user dies */

static int alloc_taskid(l4_taskid_t *id)
{
  int no;

  no = task_no_alloc();
  if (no < 0)
    return -EBUSY;

  *id = get_taskid(current->tss.user_thread_id);

  id->id.task = no;

  /* make the user task the chief of the alloced task id */
  *id = l4_task_new(*id, get_taskid(current->tss.user_thread_id).lh.low,
                   0, 0, L4_NIL_ID);

  if (thread_equal(L4_NIL_ID, *id))
    {
      task_no_free(no);

      return -EIO;
    }

  id->id.nest++;
  id->id.chief = get_taskid(current->tss.user_thread_id).id.task;
  return 0;
}

static int gettask(unsigned long *newid)
{
  int error;
  l4_taskid_t id;

  printk("sys_l4_alloc_taskid called\n");

  error = verify_area(VERIFY_WRITE, newid, 8);
  if (error)
    return error;
  
  error = alloc_taskid(&id);
  if (error)
    return error;

  put_fs_long(id.lh.low, newid);
  put_fs_long(id.lh.high, newid + 1);

  printk("sys_l4_alloc_taskid succeeded; returns %x:%x\n",
         id.lh.high, id.lh.low);

  return 0;
}

void user_level_pager();
static int get_emu_info(unsigned long *emu_info)
{
  int error;
#ifdef DEBUG
  printk("sys_l4_get_emu_info called\n");
#endif
  error = verify_area(VERIFY_WRITE, emu_info, 3*sizeof(unsigned long));
  if (error)
    return error;
  
  put_fs_long((unsigned long)&emulib_idt_descr, emu_info);
  put_fs_long((unsigned long)&emulib_debug_idt_descr, emu_info + 1);
  put_fs_long(EMU_LOCAL_LOCK_ADDRESS, emu_info + 2);
  put_fs_long((unsigned long)user_level_pager, emu_info + 3);
#ifdef DEBUG
  printk("sys_l4_get_emu_info succeeded; returns %p, %p, %lx, %p\n",
         &emulib_idt_descr, &emulib_debug_idt_descr, EMU_LOCAL_LOCK_ADDRESS,
	 user_level_pager);
#endif
  return 0;
}

static int frame(unsigned long arg)
{
  void l4_handle_special_region(unsigned long arg);
  l4_handle_special_region(arg);
  enter_kdebug("special region");
  return 0;
}
extern unsigned long syscall_count;
static int
print_perf(char *buffer)
{
  int printed = 0;
  printed += sprintf(buffer + printed, "NrSysCalls: %ld\n", syscall_count);

  return printed;
}
static int print_ids(char *buffer)
{
  int printed = 0,
    i;
  struct task_struct *p;

  for (i = 0; i < NR_TASKS ; i++)
    {
      if (! (p = task[i]))
	continue;

      if (printed > PAGE_SIZE - 200)
	{
	  printed += sprintf(buffer, "[MORE]\n");
	  break;
	}
 
      printed += sprintf(buffer + printed,
			 "%-8s %5d %08x\n",
			 p->comm, p->pid, p->tss.user_thread_id.lh.low);
    }

  return printed;
}

/* file implementation */

static int proc_l4_read(struct inode * i, struct file * f, 
			char * buf, int count)
{
  int file, error;
  unsigned long length;
  char *page;

  file = lookup_file(i);
  if (file < 0)
    return file;

  if (! (page = (char *) __get_free_page(GFP_KERNEL)))
    return -ENOMEM;

  switch (file)
    {
    case FILE_IDS:
      length = print_ids(page);
      if (length <= 0)
	{
	  error = length;
	  break;
	}

      if (f->f_pos >= length)
	{
	  error = 0;
	  break;
	}

      if (count + f->f_pos > length)
	count = length - f->f_pos;

      memcpy_tofs(buf, page + f->f_pos, count);
      f->f_pos += count;

      error = count;
      break;

    case FILE_PERF:
      length = print_perf(page);
      if (length <= 0)
	{
	  error = length;
	  break;
	}

      if (f->f_pos >= length)
	{
	  error = 0;
	  break;
	}

      if (count + f->f_pos > length)
	count = length - f->f_pos;

      memcpy_tofs(buf, page + f->f_pos, count);
      f->f_pos += count;

      error = count;
      break;
    default:
      error = -EINVAL;
    }

  free_page((unsigned long) page);
  return error;
}

static int proc_l4_ioctl(struct inode *i, struct file *f, unsigned int cmd, 
			 unsigned long arg)
{
  int file;
  pid_t pid;
  struct task_struct *p;
  void enter_kernel_debugger(void);

  file = lookup_file(i);
  if (file < 0)
    return file;

  switch (file)
    {
    case FILE_CTL:
      switch (cmd)
	{
	case L4GETTASK:
	  return gettask((unsigned long *) arg);
	case L4GETEMUINFO:
	  return get_emu_info((unsigned long *) arg);
	case L4FRAME:
	  return frame(arg);
	case L4KDEBUG:
	  enter_kernel_debugger();
	  return 0;
	case L4PRINT_SWITCH:
	  print_switch = arg;
	  return 0;
	case L4DEBUG_MSR:
	  debug_msr = arg;
	  return 0;
	case L4PID2TID:
	  pid = (pid_t)arg;
	  for_each_task(p) {
	    if (p && p->pid == pid)
	      return p->tss.user_thread_id.lh.low; 
	  }
	  return 0;
	case L4RAISEEXCEPTION:
	  asm("pushf;"
	      "pusha;"
	      "movl $0xaaaaaaaa, %eax;"
	      "movl $0xbbbbbbbb, %ebx;"
	      "movl $0xcccccccc, %ecx;"
	      "movl $0xdddddddd, %edx;"
	      "movl $0xeeeeeeee, %esi;"
	      "movl $0xffffffff, %edi;"
	      "int $79;"
	      "popa;"
	      "popf;");
	  return 0;
	case L4TRAPINIT:
	  {
	    void trap_init(void);
	    trap_init();
	    return 0;
	  }
	case L4NULLPTR:
	  *(int *)0=0;
	  return 0;
#ifdef USE_SMALL_SPACES
	case L4MKSMALL:
	  {
	    static int next_small_space = 3; /* XXX should be configurable */
	    
	    pid = (pid_t) arg;
	    for_each_task(p)
	      {
		if (p && p->pid == pid)
		  {
#if 1 /* XXX */
		    next_small_space = 3;
#endif

		    rmgr_set_small_space(p->tss.user_thread_id, 
					 next_small_space);

		    if (++next_small_space >= 512/USE_SMALL_SPACES)
		      next_small_space = 3;

		    return 0;
		  }
	      }  

	    return -ESRCH;
	  }
#endif /* USE_SMALL_SPACES */
	default:
	  return -EINVAL;
	}

    default:
      return -EINVAL;
    }

  return 0;
}

static int proc_l4_select(struct inode *inode, struct file *f, 
			  int sel_type, select_table * wait)
{
  int file = lookup_file(inode);
  if (file < 0) 
    return file;

  if (file != FILE_UNSELECT)
    return -EINVAL;

  if (sel_type != SEL_IN)
    return 0;

  select_wait(&current->tss.unselect_wait_queue, wait);
  return 0;
}

/* file data structures */

static struct file_operations fops =
{
  NULL,				/* lseek   */
  proc_l4_read,			/* read    */
  NULL,				/* write   */
  NULL,				/* readdir */
  proc_l4_select,		/* select  */
  proc_l4_ioctl,		/* ioctl   */
  NULL,				/* mmap    */
  NULL,				/* no special open code    */
  NULL,				/* no special release code */
  NULL				/* can't fsync */
};


static struct inode_operations iops =
{
  &fops,
  NULL,				/* create */
  NULL,				/* lookup */
  NULL,				/* link */
  NULL,				/* unlink */
  NULL,				/* symlink */
  NULL,				/* mkdir */
  NULL,				/* rmdir */
  NULL,				/* mknod */
  NULL,				/* rename */
  NULL,				/* readlink */
  NULL,				/* follow_link */
  NULL,				/* readpage */
  NULL,				/* writepage */
  NULL,				/* bmap */
  NULL,				/* truncate */
  NULL				/* permission */
};

/* procfs data structures */

static struct proc_dir_entry entry_dir =
{
  0, sizeof(NAME_DIR) - 1, NAME_DIR, MODE_DIR | S_IFDIR, 2, 0, 0, 0, 
  &proc_dir_inode_operations, 0, 0, 0, &proc_root, 0, 0    
};

static struct proc_dir_entry entry_ctl =
{
  0, sizeof(NAME_CTL) - 1, NAME_CTL, MODE_CTL | S_IFREG, 1, 0, 0, 0, 
  &iops, 0, 0, 0, &entry_dir, 0, (void *) FILE_CTL
};

static struct proc_dir_entry entry_ids =
{
  0, sizeof(NAME_IDS) - 1, NAME_IDS, MODE_IDS | S_IFREG, 1, 0, 0, 0, 
  &iops, 0, 0, 0, &entry_dir, 0, (void *) FILE_IDS
};

static struct proc_dir_entry entry_perf =
{
  0, sizeof(NAME_PERF) - 1, NAME_PERF, MODE_PERF | S_IFREG, 1, 0, 0, 0, 
  &iops, 0, 0, 0, &entry_dir, 0, (void *) FILE_PERF
};

static struct proc_dir_entry entry_unselect =
{
  0, sizeof(NAME_UNSELECT) - 1, NAME_UNSELECT, 
  MODE_UNSELECT | S_IFREG, 1, 0, 0, 0, 
  &iops, 0, 0, 0, &entry_dir, 0, (void *) FILE_IDS
};

/* the ``unselect'' thread */

static void unselect_thread(void)
{
  int code, ok;
  dword_t pid, dummy;
  l4_threadid_t sender;
  l4_msgdope_t dummydope;
  struct task_struct *client, *target, *p;

  for (;;)
    {
      code = l4_i386_ipc_wait(&sender, 0,
			      &pid, &dummy,
			      L4_IPC_NEVER, &dummydope);
      while (code == 0);
	{
	  target = 0;
	  for_each_task(p)
	    {
	      if (p->pid == pid)
		{
		  target = p;
		  break;
		}
	    }

	  client = task_to_proc[sender.id.task];

	  if (client && target &&
	      client->mm == target->mm)	/* threads in same address space? */
	    {
	      wake_up_interruptible(&target->tss.unselect_wait_queue);
	      ok = 1;
	    }
	  else
	    ok = 0;

	  code = l4_i386_ipc_reply_and_wait(sender, 0,
					    ok, 0,
					    &sender, 0,
					    &pid, &dummy,
					    L4_IPC_TIMEOUT(0, 1, 0, 0, 0, 0),
					    &dummydope);
	}
    }
}

/* initialization */

int proc_l4_setup(void)
{
  static int unselect_stack[STACK_SIZE_PROC_UNSELECT];
  dword_t dummy;
  l4_threadid_t unselect_id;
  l4_threadid_t my_preempter, my_pager;
  l4_threadid_t me;

  /* register the /proc entries */
  proc_register_dynamic(&proc_root, &entry_dir);
  proc_register_dynamic(&entry_dir, &entry_ctl);
  proc_register_dynamic(&entry_dir, &entry_ids);
  proc_register_dynamic(&entry_dir, &entry_perf);
  proc_register_dynamic(&entry_dir, &entry_unselect);

  /* now spawn a thread used for ``unselecting''. */
  me = l4_myself();
  my_preempter = L4_INVALID_ID;
  my_pager = L4_INVALID_ID;
  l4_thread_ex_regs(me, (dword_t) -1, (dword_t) -1,
                    &my_preempter,
                    &my_pager,
                    &dummy,
                    &dummy,
                    &dummy);
  unselect_id = me;
  unselect_id.id.lthread = LTHREAD_NO_PROC_UNSELECT;

  l4_thread_ex_regs(unselect_id, (dword_t) unselect_thread, 
		    ((dword_t) unselect_stack) + STACK_SIZE_PROC_UNSELECT,
                    &my_preempter,
                    &my_pager,
                    &dummy,
                    &dummy,
                    &dummy);

  return 0;
}
