/* ptrace.c */
/* By Ross Biro 1/23/92 */
/* edited by Linus Torvalds */

#include <linux/head.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/ptrace.h>
#include <linux/user.h>
#include <linux/debugreg.h>

#include <asm/segment.h>
#include <asm/pgtable.h>
#include <asm/system.h>
#include "../include/exclpage.h"
/*
 * does not yet catch signals sent when the child dies.
 * in exit.c or in signal.c.
 */

/* determines which flags the user has access to. */
/* 1 = access 0 = no access */
#define FLAG_MASK 0x00044dd5

/* set's the trap flag. */
#define TRAP_FLAG 0x100

/*
 * this is the number to subtract from the top of the stack. To find
 * the local frame.
 */
#define MAGICNUMBER 68
/* #define DEBUG_PTRACE */
#ifdef DEBUG_PTRACE
static char reg_names[17][9]={"EBX", "ECX", "EDX", "ESI", "EDI", "EBP", "EAX", 
				"DS", "ES", "FS", "GS", "ORIG_EAX", 
				"EIP", "CS", "EFL", "UESP", "SS"};
#endif
/* change a pid into a task struct. */
static inline struct task_struct * get_task(int pid)
{
	int i;

	for (i = 1; i < NR_TASKS; i++) {
		if (task[i] != NULL && (task[i]->pid == pid))
			return task[i];
	}
	return NULL;
}

/*
 * This routine gets a long from any process space by following the page
 * tables. NOTE! You should check that the long isn't on a page boundary,
 * and that it is in the task area before calling this: this routine does
 * no checking.
 */
static unsigned long get_long(struct task_struct * tsk, 
	struct vm_area_struct * vma, unsigned long addr)
{
	pgd_t * pgdir;
	pmd_t * pgmiddle;
	pte_t * pgtable;
	unsigned long page;

repeat:
	pgdir = pgd_offset(vma->vm_mm, addr);
	if (pgd_none(*pgdir)) {
		do_no_page(tsk, vma, addr, 0);
		goto repeat;
	}
	if (pgd_bad(*pgdir)) {
		printk("ptrace: bad page directory %08lx\n", pgd_val(*pgdir));
		pgd_clear(pgdir);
		return 0;
	}
	pgmiddle = pmd_offset(pgdir, addr);
	if (pmd_none(*pgmiddle)) {
		do_no_page(tsk, vma, addr, 0);
		goto repeat;
	}
	if (pmd_bad(*pgmiddle)) {
		printk("ptrace: bad page middle %08lx\n", pmd_val(*pgmiddle));
		pmd_clear(pgmiddle);
		return 0;
	}
	pgtable = pte_offset(pgmiddle, addr);
	if (!pte_present(*pgtable)) {
		do_no_page(tsk, vma, addr, 0);
		goto repeat;
	}
	page = pte_page(*pgtable);
/* this is a hack for non-kernel-mapped video buffers and similar */
	if (page >= high_memory)
		return 0;
	page += addr & ~PAGE_MASK;
	return *(unsigned long *) page;
}

/*
 * This routine puts a long into any process space by following the page
 * tables. NOTE! You should check that the long isn't on a page boundary,
 * and that it is in the task area before calling this: this routine does
 * no checking.
 *
 * Now keeps R/W state of page so that a text page stays readonly
 * even if a debugger scribbles breakpoints into it.  -M.U-
 */
static void put_long(struct task_struct * tsk, struct vm_area_struct * vma, unsigned long addr,
	unsigned long data)
{
	pgd_t *pgdir;
	pmd_t *pgmiddle;
	pte_t *pgtable;
	unsigned long page;

repeat:
	pgdir = pgd_offset(vma->vm_mm, addr);
	if (!pgd_present(*pgdir)) {
		do_no_page(tsk, vma, addr, 1);
		goto repeat;
	}
	if (pgd_bad(*pgdir)) {
		printk("ptrace: bad page directory %08lx\n", pgd_val(*pgdir));
		pgd_clear(pgdir);
		return;
	}
	pgmiddle = pmd_offset(pgdir, addr);
	if (pmd_none(*pgmiddle)) {
		do_no_page(tsk, vma, addr, 1);
		goto repeat;
	}
	if (pmd_bad(*pgmiddle)) {
		printk("ptrace: bad page middle %08lx\n", pmd_val(*pgmiddle));
		pmd_clear(pgmiddle);
		return;
	}
	pgtable = pte_offset(pgmiddle, addr);
	if (!pte_present(*pgtable)) {
		do_no_page(tsk, vma, addr, 1);
		goto repeat;
	}
	page = pte_page(*pgtable);
	if (!pte_write(*pgtable)) {
		do_wp_page(tsk, vma, addr, 1);
		goto repeat;
	}
/* this is a hack for non-kernel-mapped video buffers and similar */
	if (page < high_memory)
		*(unsigned long *) (page + (addr & ~PAGE_MASK)) = data;
/* we're bypassing pagetables, so we have to set the dirty bit ourselves */
/* this should also re-instate whatever read-only mode there was before */
	set_pte(pgtable, pte_mkdirty(mk_pte(page, vma->vm_page_prot)));
	flush_tlb();
}

static struct vm_area_struct * find_extend_vma(struct task_struct * tsk, unsigned long addr)
{
	struct vm_area_struct * vma;

	addr &= PAGE_MASK;
	vma = find_vma(tsk->mm, addr);
	if (!vma)
		return NULL;
	if (vma->vm_start <= addr)
		return vma;
	if (!(vma->vm_flags & VM_GROWSDOWN))
		return NULL;
	if (vma->vm_end - addr > tsk->rlim[RLIMIT_STACK].rlim_cur)
		return NULL;
	vma->vm_offset -= vma->vm_start - addr;
	vma->vm_start = addr;
	return vma;
}

/*
 * This routine checks the page boundaries, and that the offset is
 * within the task area. It then calls get_long() to read a long.
 */
static int read_long(struct task_struct * tsk, unsigned long addr,
	unsigned long * result)
{
	struct vm_area_struct * vma = find_extend_vma(tsk, addr);

	if (!vma)
		return -EIO;
	if ((addr & ~PAGE_MASK) > PAGE_SIZE-sizeof(long)) {
		unsigned long low,high;
		struct vm_area_struct * vma_high = vma;

		if (addr + sizeof(long) >= vma->vm_end) {
			vma_high = vma->vm_next;
			if (!vma_high || vma_high->vm_start != vma->vm_end)
				return -EIO;
		}
		low = get_long(tsk, vma, addr & ~(sizeof(long)-1));
		high = get_long(tsk, vma_high, (addr+sizeof(long)) & ~(sizeof(long)-1));
		switch (addr & (sizeof(long)-1)) {
			case 1:
				low >>= 8;
				low |= high << 24;
				break;
			case 2:
				low >>= 16;
				low |= high << 16;
				break;
			case 3:
				low >>= 24;
				low |= high << 8;
				break;
		}
		*result = low;
	} else
		*result = get_long(tsk, vma, addr);
	return 0;
}

/*
 * This routine checks the page boundaries, and that the offset is
 * within the task area. It then calls put_long() to write a long.
 */
static int write_long(struct task_struct * tsk, unsigned long addr,
	unsigned long data)
{
	struct vm_area_struct * vma = find_extend_vma(tsk, addr);

	if (!vma)
		return -EIO;
	if ((addr & ~PAGE_MASK) > PAGE_SIZE-sizeof(long)) {
		unsigned long low,high;
		struct vm_area_struct * vma_high = vma;

		if (addr + sizeof(long) >= vma->vm_end) {
			vma_high = vma->vm_next;
			if (!vma_high || vma_high->vm_start != vma->vm_end)
				return -EIO;
		}
		low = get_long(tsk, vma, addr & ~(sizeof(long)-1));
		high = get_long(tsk, vma_high, (addr+sizeof(long)) & ~(sizeof(long)-1));
		switch (addr & (sizeof(long)-1)) {
			case 0: /* shouldn't happen, but safety first */
				low = data;
				break;
			case 1:
				low &= 0x000000ff;
				low |= data << 8;
				high &= ~0xff;
				high |= data >> 24;
				break;
			case 2:
				low &= 0x0000ffff;
				low |= data << 16;
				high &= ~0xffff;
				high |= data >> 16;
				break;
			case 3:
				low &= 0x00ffffff;
				low |= data << 24;
				high &= ~0xffffff;
				high |= data >> 8;
				break;
		}
		put_long(tsk, vma, addr & ~(sizeof(long)-1),low);
		put_long(tsk, vma_high, (addr+sizeof(long)) & ~(sizeof(long)-1),high);
	} else
		put_long(tsk, vma, addr, data);
	return 0;
}

#ifdef DEBUG_PTRACE

asmlinkage int _sys_ptrace(long request, long pid, long addr, long data);
asmlinkage int sys_ptrace(long request, long pid, long addr, long data)
{
  int ret;
#if 0
  struct task_struct *child = get_task(pid);
#endif
  herc_printf("sys_ptrace: pid: %lx, addr: %lx, data: %lx\n",
	 pid, addr, data);
  switch (request) {
  case PTRACE_TRACEME: 
    herc_printf("   request: PTRACE_TRACEME ");
    break;
  case PTRACE_ATTACH: 
    herc_printf("   request: PTRACE_ATTACH");
    break;
  case PTRACE_KILL:
    herc_printf("   request: PTRACE_KILL");
    break;
  case PTRACE_PEEKTEXT:
    herc_printf("   request: PTRACE_PEEKTEXT ");
    break;
  case PTRACE_PEEKDATA:
    herc_printf("   request: PTRACE_PEEKDATA  ");
    break;
  case PTRACE_PEEKUSR:
    herc_printf("   request: PTRACE_PEEKUSR  ");
    break;
  case PTRACE_SYSCALL:
    herc_printf("   request: PTRACE_SYSCALL  ");
    break;
  case PTRACE_CONT:
    herc_printf("   request: PTRACE_CONT  ");
    break;
  case PTRACE_SINGLESTEP:
    herc_printf("   request: PTRACE_SINGLESTEP  ");
    break;
  case PTRACE_DETACH:
    herc_printf("   request: PTRACE_DETACH  ");
    break;
  case PTRACE_POKETEXT:
    herc_printf("   request: PTRACE_POKETEXT  ");
    break;
  case PTRACE_POKEDATA:
    herc_printf("   request: PTRACE_POKEDATA  ");
    break;
  case PTRACE_POKEUSR:
    herc_printf("   request: PTRACE_POKEUSR  ");
    break;
  }
  ret = _sys_ptrace(request, pid, addr, data);
  herc_printf("ret: %x\n", ret);
  return ret;
}
asmlinkage int _sys_ptrace(long request, long pid, long addr, long data)
#else
asmlinkage int sys_ptrace(long request, long pid, long addr, long data)
#endif
{
	struct task_struct *child;
	struct user * dummy;
#if 0
	int i;
#endif
	dummy = NULL;

	if (request == PTRACE_TRACEME) {
		/* are we already being traced? */
		if (current->flags & PF_PTRACED)
			return -EPERM;
		/* set the ptrace bit in the process flags. */
		current->flags |= PF_PTRACED;
		return 0;
	}
	if (pid == 1)		/* you may not mess with init */
		return -EPERM;
	if (!(child = get_task(pid)))
		return -ESRCH;
	if (request == PTRACE_ATTACH) {
		if (child == current)
			return -EPERM;
		if ((!child->dumpable ||
		    (current->uid != child->euid) ||
		    (current->uid != child->uid) ||
	 	    (current->gid != child->egid) ||
	 	    (current->gid != child->gid)) && !suser())
			return -EPERM;
		/* the same process cannot be attached many times */
		if (child->flags & PF_PTRACED)
			return -EPERM;
		child->flags |= PF_PTRACED;
		if (child->p_pptr != current) {
			REMOVE_LINKS(child);
			child->p_pptr = current;
			SET_LINKS(child);
		}
		send_sig(SIGSTOP, child, 1);
		return 0;
	}
	if (!(child->flags & PF_PTRACED))
		return -ESRCH;
	if (child->state != TASK_STOPPED) {
		if (request != PTRACE_KILL)
			return -ESRCH;
	}
	if (child->p_pptr != current)
		return -ESRCH;

	switch (request) {
	/* when I and D space are separate, these will need to be fixed. */
		case PTRACE_PEEKTEXT: /* read word at location addr. */ 
		case PTRACE_PEEKDATA: {
			unsigned long tmp;
			int res;

			res = read_long(child, addr, &tmp);
			if (res < 0)
				return res;
			res = verify_area(VERIFY_WRITE, (void *) data, sizeof(long));
			if (!res)
				put_fs_long(tmp,(unsigned long *) data);
			return res;
		}

	/* read the word at location addr in the USER area. */
	/* XXX Hmm, that seems to be a wrong comment. PEEKUSR reads a word from
	 * XXX the privileged stack of the process and there you can find only 
	 * XXX the thread_struct. I will implement it using our own tss. 
	 * XXX There is no treatment of debug registers and segment registers because
	 * XXX of we don't have them at the moment or at all. Jean */
		case PTRACE_PEEKUSR: {
			unsigned long tmp;
			int res;

			if ((addr & 3) || addr < 0 || 
			    addr > sizeof(struct user) - 3)
				return -EIO;

			res = verify_area(VERIFY_WRITE, (void *) data, sizeof(long));
			if (res)
				return res;
			tmp = 0;  /* Default return condition */
			if(addr < 17*sizeof(long)) {
			  addr = addr >> 2; /* temporary hack. */

			  switch (addr) {
			  case OLD_EBX:
			    tmp = child->tss.ex_page->regs.ebx ;
			    break;
			  case OLD_ECX:
			    tmp = child->tss.ex_page->regs.ecx ;
			    break;
			  case OLD_EDX:
			    tmp = child->tss.ex_page->regs.edx ;
			    break;
			  case OLD_ESI:
			    tmp = child->tss.ex_page->regs.esi ;
			    break;
			  case OLD_EDI:
			    tmp = child->tss.ex_page->regs.edi ;
			    break;
			  case OLD_EBP:
			    tmp = child->tss.ex_page->regs.ebp ;
			    break;
			  case OLD_EAX:
			    tmp = child->tss.ex_page->regs.eax ;
			    break;
			  case OLD_ORIG_EAX:
			    tmp = child->tss.ex_page->regs.orig_eax ;
			    break;
			  case OLD_EIP: 
			    tmp = child->tss.ex_page->regs.eip ;
			    break;
			  case OLD_EFL:
			    tmp = child->tss.ex_page->regs.eflags ;
			    break;
			  case OLD_UESP: 
			    tmp = child->tss.ex_page->regs.esp ;
			    break;
			  }
			};
#ifdef DEBUG_PTRACE
			herc_printf("peekusr: reading %s (%lx)", 
			       reg_names[addr], tmp);
#endif
			put_fs_long(tmp,(unsigned long *) data);
			return 0;
		}

      /* when I and D space are separate, this will have to be fixed. */
		case PTRACE_POKETEXT: /* write the word at location addr. */
		case PTRACE_POKEDATA:
#ifdef DEBUG_PTRACE
		        herc_printf("writing %lx to address %lx", data, addr);
#endif
			return write_long(child,addr,data);
	/* write the word at location addr in the USER area */
	/* XXX Look at PEEKUSR for comments about reading and writing the u_area */ 
		case PTRACE_POKEUSR: 
			if ((addr & 3) || addr < 0 || 
			    addr > sizeof(struct user) - 3)
				return -EIO;

			addr = addr >> 2; /* temporary hack. */

			if (addr == OLD_ORIG_EAX)
				return -EIO;
			if (addr == OLD_DS || addr == OLD_ES ||
			    addr == OLD_FS || addr == OLD_GS ||
			    addr == OLD_CS || addr ==OLD_SS) {
			    	data &= 0xffff;
			    	if (data && (data & 3) != 3)
					return -EIO;
			}
			if (addr == OLD_EFL) {   /* flags. */
				data &= FLAG_MASK;
				data |= child->tss.ex_page->regs.eflags;
			}
		  /* Do not allow the user to set the debug register for kernel
		     address space */
		  if(addr < 17){
			  switch (addr) {
			  case OLD_EBX:
			    child->tss.ex_page->regs.ebx = data ;
			    break;
			  case OLD_ECX:
			    child->tss.ex_page->regs.ecx = data;
			    break;
			  case OLD_EDX:
			    child->tss.ex_page->regs.edx = data;
			    break;
			  case OLD_ESI:
			    child->tss.ex_page->regs.esi = data;
			    break;
			  case OLD_EDI:
			    child->tss.ex_page->regs.edi = data;
			    break;
			  case OLD_EBP:
			    child->tss.ex_page->regs.ebp = data;
			    break;
			  case OLD_EAX:
			    child->tss.ex_page->regs.eax = data;
			    break;
			  case OLD_ORIG_EAX:
			    child->tss.ex_page->regs.orig_eax = data;
			    break;
			  case OLD_EIP: 
			    child->tss.ex_page->regs.eip = data;
			    break;
			  case OLD_EFL:
			    child->tss.ex_page->regs.eflags = data;
			    break;
			  case OLD_UESP: 
			    child->tss.ex_page->regs.esp = data;
			    break;
			  }
#ifdef DEBUG_PTRACE
			herc_printf("pokeusr: writing %s (%lx)", 
			       reg_names[addr], data);
#endif
			return 0;
			};

		  /* XXX I don't want to handle debug registers at the moment. Jean */
		  return -EIO;
#if 0
		  /* We need to be very careful here.  We implicitly
		     want to modify a portion of the task_struct, and we
		     have to be selective about what portions we allow someone
		     to modify. */

		  addr = addr << 2;  /* Convert back again */
		  if(addr >= (long) &dummy->u_debugreg[0] &&
		     addr <= (long) &dummy->u_debugreg[7]){

			  if(addr == (long) &dummy->u_debugreg[4]) return -EIO;
			  if(addr == (long) &dummy->u_debugreg[5]) return -EIO;
			  if(addr < (long) &dummy->u_debugreg[4] &&
			     ((unsigned long) data) >= 0xbffffffd) return -EIO;
			  
			  if(addr == (long) &dummy->u_debugreg[7]) {
				  data &= ~DR_CONTROL_RESERVED;
				  for(i=0; i<4; i++)
					  if ((0x5f54 >> ((data >> (16 + 4*i)) & 0xf)) & 1)
						  return -EIO;
			  };

			  addr -= (long) &dummy->u_debugreg;
			  addr = addr >> 2;
			  child->debugreg[addr] = data;
			  return 0;
		  };
		  return -EIO;
#endif
		case PTRACE_SYSCALL: /* continue and stop at next (return from) syscall */
		case PTRACE_CONT: { /* restart after signal. */
			if ((unsigned long) data > NSIG)
				return -EIO;
			if (request == PTRACE_SYSCALL)
				child->flags |= PF_TRACESYS;
			else
				child->flags &= ~PF_TRACESYS;
			child->exit_code = data;
			wake_up_process(child);
	/* make sure the single step bit is not set. */
			child->tss.ex_page->regs.eflags &= ~TRAP_FLAG;
			return 0;
		}

/*
 * make the child exit.  Best I can do is send it a sigkill. 
 * perhaps it should be put in the status that it wants to 
 * exit.
 */
		case PTRACE_KILL: {
			if (child->state == TASK_ZOMBIE)	/* already dead */
				return 0;
			wake_up_process(child);
			child->exit_code = SIGKILL;
	/* make sure the single step bit is not set. */
			child->tss.ex_page->regs.eflags &= ~TRAP_FLAG;
			return 0;
		}

		case PTRACE_SINGLESTEP: {  /* set the trap flag. */
			if ((unsigned long) data > NSIG)
				return -EIO;
			child->flags &= ~PF_TRACESYS;
			child->tss.ex_page->regs.eflags |= TRAP_FLAG;
			wake_up_process(child);
			child->exit_code = data;
	/* give it a chance to run. */
			return 0;
		}

		case PTRACE_DETACH: { /* detach a process that was attached. */
			if ((unsigned long) data > NSIG)
				return -EIO;
			child->flags &= ~(PF_PTRACED|PF_TRACESYS);
			wake_up_process(child);
			child->exit_code = data;
			REMOVE_LINKS(child);
			child->p_pptr = child->p_opptr;
			SET_LINKS(child);
			/* make sure the single step bit is not set. */
			child->tss.ex_page->regs.eflags &= ~TRAP_FLAG;
			return 0;
		}

		default:
			return -EIO;
	}
}

asmlinkage void syscall_trace(void)
{
	if ((current->flags & (PF_PTRACED|PF_TRACESYS))
			!= (PF_PTRACED|PF_TRACESYS))
		return;
	current->exit_code = SIGTRAP;
	current->state = TASK_STOPPED;
	/* Need a signal arg (YHP) */
	notify_parent(current, SIGTRAP);
	schedule();
	/*
	 * this isn't the same as continuing with a signal, but it will do
	 * for normal use.  strace only continues with a signal if the
	 * stopping signal is not SIGTRAP.  -brl
	 */
	if (current->exit_code)
		current->signal |= (1 << (current->exit_code - 1));
	current->exit_code = 0;
}
