#include <linux/mm.h>
#include <asm/segment.h>
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>

#include "../include/l4_memory.h"
#include "../include/task.h"
#include "../include/config.h"
#include "../include/dispatch.h"
#include "../include/exclpage.h"
#include "../include/lock.h"

/* #define DEBUG */
#include "../include/debug.h"
#define ROOT_NO_REPLY 0x29051970

static int root_pager_stack[256];

unsigned long root_pager_start_low_mem, root_pager_memory_start, 
  root_pager_memory_end;
l4_threadid_t root_pager_id, root_preempter_id;
extern volatile int null_pointer_check;
static int null_pointer_check_done = 0;

void 
start_root_pager(void)
{
  dword_t old_eflags, old_eip, old_esp;
  l4_threadid_t myself, mypager = L4_INVALID_ID;
  root_preempter_id = L4_INVALID_ID;

  /* get information about myself */
  myself = root_pager_id = l4_myself();
  l4_thread_ex_regs(myself, (dword_t) 0xffffffff, (dword_t) 0xffffffff, 
		    &root_preempter_id, &mypager, 
		    &old_eflags, &old_eip, &old_esp);

  /* create root_pager */
  root_pager_id.id.lthread = LTHREAD_NO_ROOT_PAGER;
  l4_thread_ex_regs(root_pager_id, 
		    (dword_t) root_pager, (dword_t) &root_pager_stack[255], 
		    &root_preempter_id, &mypager, 
		    &old_eflags, &old_eip, &old_esp);
}


int
page_fault_request(l4_threadid_t src, 
		   dword_t fault_address, dword_t faulting_eip, 
		   l4_fpage_t *fpage)
{
  pte_t pte, *ptep;

  herc_printf("Hmm, strange, that shouldn't happen anymore\n");
  ptep = lookup_pte(swapper_pg_dir, VMALLOC_VMADDR(fault_address));
  
  if (! ptep)
    {
      /* Ohh..  the kernel touched a non-present page! */
      herc_printf("Root: pte not present for addr %lx (virt: %lx) "
		  "and pdir %p at eip: %lx\n",
		  (unsigned long)fault_address, 
		  (unsigned long)VMALLOC_VMADDR(fault_address),
		  swapper_pg_dir, (unsigned long)faulting_eip);
      enter_kdebug("root_pager: pte not present");

      return -1;		/* wait for next message */
    }

  pte = *ptep;

  if (pte_val(pte) > 0x80000000U)
    {
      /* OK, this is a vremap()'d area. */
	      
      fault_address &= ~(0x400000 - 1);
      if (!request_mem_area_from_sigma0(fault_address, 
					pte_val(*ptep)))
	{
	  *fpage = l4_fpage(fault_address, L4_LOG2_4MB, 
			    (pte_write(*ptep) ? L4_FPAGE_RW : L4_FPAGE_RO), 
			    L4_FPAGE_MAP);
	  return 0;
	}
      else
	{
	  enter_kdebug("request region in kernel failed");
	  return -1;
	}
    }
	  
  if ((fault_address & _PAGE_RW) && !pte_write(pte)) 
    {
      /* access check failed */ 
      herc_printf("Root, Wrong Access in %x.%x, addr: %lx (%s), "
		  "eip; %lx\n",
		  src.id.task, src.id.lthread,
		  (unsigned long)fault_address, 
		  ((fault_address & _PAGE_RW) ? "write" : "read"),
		  (unsigned long)faulting_eip);
      enter_kdebug("wrong access");
      return -1;
    }

  /* check for null pointer */
  if ((unsigned long)fault_address < (unsigned long)0x1000L)
    {
      herc_printf("Root: null pointer deref in %x.%x "
		  "at addr: %lx and eip: %lx (pte: %lx)\n",
		  src.id.task, src.id.lthread, 
		  (unsigned long)fault_address, 
		  (unsigned long)faulting_eip,  pte_val(*ptep));
      if (null_pointer_check && !null_pointer_check_done)
	{
	  *(int *)0 = 0;
/* 	  enter_kdebug("touched zero page"); */
	  null_pointer_check = 0;
	  null_pointer_check_done = 1;
	}
      else
	{
	  return -1;
	}
    }
  else
    if (fault_address < root_pager_memory_end )
      {
	herc_printf("\nRoot: pfa (%lx) < rp_mem_end (%lx)"
		    " in %x.%x at eip %lx\n",
		    (unsigned long)fault_address, 
		    root_pager_memory_end, 
		    src.id.task, src.id.lthread, 
		    (unsigned long)faulting_eip);
	enter_kdebug("root: ram-pf; why?");
      }
	  
  /* emulate dirty and accessed bits of page_tables */
  if (fault_address & _PAGE_RW)
    pte_val(*ptep) |= _PAGE_DIRTY | _PAGE_ACCESSED;
  else
    pte_val(*ptep) |= _PAGE_ACCESSED;
	  
  /* Ok, the kernel accessed a valid page. Map it */
  *fpage = l4_fpage(pte_page(pte), L4_LOG2_PAGESIZE, L4_FPAGE_RW, 
		    /* XXX pte_write(pte) ? L4_FPAGE_RW : L4_FPAGE_RO, */
		    L4_FPAGE_MAP);
  return 0;
}

int send_sig_segv(void);
void 
root_pager(void)
{
  dword_t error, fault_address, faulting_eip;
  l4_threadid_t src;
  l4_msgdope_t result;
  l4_fpage_t fpage;

  /* Setup a default idt */
  set_default_idt();

  /* Wait for the very first page fault of our special init task and map
     it the whole RAM space */
  error = l4_i386_ipc_wait(&src, L4_IPC_SHORT_MSG, 
			   &fault_address, &faulting_eip,
			   L4_IPC_NEVER, &result);
  if (!error)
    {
      error = l4_i386_ipc_send(src, L4_IPC_SHORT_FPAGE,
			       0, /* send base */
			       l4_fpage(0, L4_LOG2_HALF_GIG,
					L4_FPAGE_RW, L4_FPAGE_MAP).fpage,
			       L4_IPC_TIMEOUT(0, 1, 0, 0, 0, 0), 
			       &result);
      if (error)
	enter_kdebug("error while mapping");
    }
  else
    {
      enter_kdebug("error while waiting for special init");
    }

  /* Enter endless loop and wait for requests or page faults (there
     shouldn't be any page faults anymore */
  for(;;)
    {
      error = l4_i386_ipc_wait(&src, L4_IPC_SHORT_MSG, 
			       &fault_address, &faulting_eip,
			       L4_IPC_NEVER, &result);
      while(!error) 
	{
	  KO('r');
#ifdef DEBUG
	  herc_printf("root: rcvd %lx, %lx\n", fault_address, faulting_eip);
#endif
	  herc_printf("root: rcvd %lx, %lx\n", 
		      (unsigned long)fault_address, 
		      (unsigned long)faulting_eip);
	  error = page_fault_request(src, fault_address, faulting_eip, 
				     &fpage);
	  if (error)
	    {
	      send_sig_segv();
	      error = ROOT_NO_REPLY;
	    }
	  if (!error)
	    {
	      KO('R');
	      error =
		l4_i386_ipc_reply_and_wait(src,
					   L4_IPC_SHORT_FPAGE,
					   fault_address, fpage.fpage,
					   &src, 
					   L4_IPC_SHORT_MSG, 
					   &fault_address, &faulting_eip,
					   L4_IPC_TIMEOUT(0,1,0,0,0,0), 
					   &result);
	    }
	}
      if (error && (error != ROOT_NO_REPLY))
	enter_kdebug("root: error!?");
    }
}



/* page_present looks into the page directory of the user process and
     tries to find and entry for the page. It returns (-1) if successful
     or an error code according to the intel manuals (a combination of
     PF_EUSER, (PF_EREAD or PF_EWRITE) and (PF_ENOTPRESENT or
     PF_EPROTECTION)).
   
     XXX Optimization: touch the virtual address in USER_VM space
     instead of the physical page to reduce the number of page faults in
     the case of copy_in/out.  */

static inline int 
page_present(void **msg, dword_t *fault_address, 
	     dword_t *faulting_eip)
{
  static dword_t map_addr = 0;

  pte_t *ptep, pte;

  ptep = lookup_pte(current->tss.page_dir, *fault_address);
#ifdef DEBUG
  herc_printf("(%x%x)lookup_pte (\"%s\"): pte_t*: %p\n",
	      l4_myself().id.task, l4_myself().id.lthread, current->comm,
	      ptep);
#endif


  if (!ptep || !pte_present((pte = *ptep)))
    {
      if (*fault_address & _PAGE_RW)
	return PF_EUSER | PF_EWRITE | PF_ENOTPRESENT;
      else
	return PF_EUSER | PF_EREAD | PF_ENOTPRESENT;
    }  

  /* XXX Check for special regions, we map it read/write because of
     we expect the permissions to be so */
  if (pte_page(pte) > 0x80000000U)
    {
      herc_printf("pte_page(pte) > 0x80000000U, val: %lx, pf: %x, eip: %x\n",
		  pte_page(pte), *fault_address, *faulting_eip);

      if (! map_addr)
	{
	  extern void * vreserve(unsigned long size, unsigned long align);

	  /* get an 4MB-aligned empty VM area */
	  map_addr = (dword_t) vreserve(0x400000L, 0x400000L);

	  if (!map_addr)
	    {
	      /* Oh lord...  couldn't get a temporary mapping area! */
	      enter_kdebug("no map_area");
		  
	      /* XXX */
	      return PF_EUSER | PF_ENOTPRESENT
		| ((*fault_address & _PAGE_RW) ? PF_EWRITE : PF_EREAD);
	    }
	}

      /* OK, we have a temporary region.  now request a mapping from
	 sigma0 */
      if (!request_mem_area_from_sigma0(map_addr, pte_page(pte)))
	{
	  /* OK, temporarily map this area in the kernel, then
	     grant it to the user */
	      
	  *faulting_eip = l4_fpage(map_addr,
				   L4_LOG2_4MB, 
				   pte_write(pte) ? L4_FPAGE_RW : L4_FPAGE_RO, 
				   L4_FPAGE_GRANT).fpage;
	  *msg = L4_IPC_SHORT_FPAGE;
#if 0
	  enter_kdebug("region requested");
#endif
	  return -1;
	}
      else
	{
	  enter_kdebug("request region failed");
	  /* XXX */
	  return PF_EUSER | PF_ENOTPRESENT
	    | ((*fault_address & _PAGE_RW) ? PF_EWRITE : PF_EREAD);
	}
    }

  /* ...so it is an ordinary page fault, right? */

  if (! (*fault_address & _PAGE_RW)) 
    {
      /* Read access. Since we should still have the page, we don't
	 need to request them anymore. */
	      
      /* emulate dirty and accessed bits */
      pte_val(*ptep) |= _PAGE_ACCESSED;
	  
      *faulting_eip = l4_fpage(pte_page(pte),
			       L4_LOG2_PAGESIZE, 
			       L4_FPAGE_RO, L4_FPAGE_MAP).fpage;
      *msg = L4_IPC_SHORT_FPAGE;
      return -1;
    }
  else if (pte_write(pte)) 
    {
      /* Write access. Since we should still have the page, we don't
	 need to request them anymore. */

      /* emulate dirty and accessed bits */
      pte_val(*ptep) |= _PAGE_DIRTY | _PAGE_ACCESSED;
	  
      *faulting_eip= l4_fpage(pte_page(pte),
			      L4_LOG2_PAGESIZE, 
			      L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
      *msg = L4_IPC_SHORT_FPAGE;
      return -1;
    }

  /* oha, unallowed write access! */
  return PF_EUSER | PF_EWRITE | PF_EPROTECTION;
}

/* use by l4_idle for debugging */
void check_lookup(dword_t fault_addr, dword_t fpage)
{
  int error;
  dword_t faddr, feip;
  void *msg;
  faddr = fault_addr;
  ko('x');
  error = page_present(&msg, &faddr, &feip);
#ifdef DEBUG
  if (current->comm[0]=='e')
    {
      herc_printf("sending %lx (%lx)\n", 
		  (unsigned long)fpage, (unsigned long)feip);
    }
#endif
  if (error == -1)
    {
      if (fpage != feip)
	{
	  printk("wrong lookup: asm: %lx, c: %lx", 
		 (unsigned long)fpage, (unsigned long)feip);
	  enter_kdebug("wrong lookup");
	}
    }
  else
    {
      fault_addr &=0xfffff000;
      if ((fault_addr != 0xa0008000) && (fault_addr != 0xa0000000))
	{
	  herc_printf("fault at %lx, pdir: %lx (%lx), error : %x\n",
		      (unsigned long)fault_addr, current_pdir, 
		      (unsigned long)current->tss.page_dir, error);
	  enter_kdebug("no page found");
	}
    }
}
int
l4_handle_special_region(pte_t pte)
{
  static dword_t special_map_addr = 0;

  if (pte_page(pte) > 0x80000000U)
    {
      printk("pte_page(pte) > 0x80000000U, val: %lx\n", pte_page(pte));

      if (! special_map_addr)
	{
	  extern void * vreserve(unsigned long size, unsigned long align);

	  /* get an 4MB-aligned empty VM area */
	  special_map_addr = (dword_t) vreserve(0x400000L, 0x400000L);

	  if (!special_map_addr)
	    {
	      /* Oh lord...  couldn't get a temporary mapping area! */
	      enter_kdebug("no map_area");
		  
	      /* XXX */
	      return 0;
	    }
	}

      /* OK, we have a temporary region.  now request a mapping from
	 sigma0 */
      if (!request_mem_area_from_sigma0(special_map_addr, pte_page(pte)))
	{
	  /* OK, temporarily map this area in the kernel, then
	     grant it to the user */
	      
#if 0
	  enter_kdebug("region requested");
#endif
	  return l4_fpage(special_map_addr,
			  L4_LOG2_4MB, 
			  pte_write(pte) ? L4_FPAGE_RW : L4_FPAGE_RO, 
			  L4_FPAGE_GRANT).fpage;
	}
      else
	{
	  enter_kdebug("request region failed");
	  /* XXX */
	  return 0;
	}
    }
  else
    {
      enter_kdebug("pte_page(pte) <= 0x80000000U");
      return 0;
    }
}
