/* communication with the L4 RMGR */

#include <l4/syscalls.h>
#include <l4/server/rmgr.h>

#include "../include/l4_memory.h"
#include "../include/config.h"

/* the interface we implement */
#include "../include/rmgr.h"

int have_rmgr;
l4_threadid_t rmgr_id;

extern inline int rmgr_call(dword_t request, dword_t arg, dword_t *out)
{
  extern unsigned long irqs_in_progress;
  l4_msgdope_t result;
  dword_t r1, r2;
  int err;

  if (irqs_in_progress != 0) {
    printk("irqs_in_progress before rmgr_call: %lx\n", irqs_in_progress);
    enter_kdebug("irq activ before rmgr_call");
  }
  err = l4_i386_ipc_call(rmgr_id, 
			 L4_IPC_SHORT_MSG, request, arg,
			 L4_IPC_SHORT_MSG, &r1, &r2,
			 L4_IPC_NEVER, &result);
  
  if (irqs_in_progress != 0) {
    printk("irqs_in_progress after rmgr_call: %lx\n", irqs_in_progress);
    enter_kdebug("irq activ after rmgr_call");
  }

  if (err) return err;

  if (out) *out = r2;
  return r1;
}

int rmgr_init(void)
{
  dword_t r;

  rmgr_id = sigma0_id;
  rmgr_id.id.lthread = RMGR_LTHREAD_SUPER;
  
  have_rmgr = (rmgr_call(RMGR_RMGR_MSG(RMGR_RMGR_PING, 0xbeef), 0, &r) == 0
	       && r == ~(0xbeef));

  return have_rmgr;
}

int rmgr_set_small_space(l4_threadid_t dest, int num)
{
#ifdef USE_SMALL_SPACES
  l4_sched_param_t schedparam;
  l4_threadid_t foo_id;

  if (have_rmgr)
    return rmgr_call(RMGR_TASK_MSG(RMGR_TASK_SET_SMALL, num), dest.lh.low, 0);

  foo_id = L4_INVALID_ID;
  l4_thread_schedule(dest, L4_INVALID_SCHED_PARAM, &foo_id, &foo_id, 
		     &schedparam);
  foo_id = L4_INVALID_ID;
  schedparam.sp.small = L4_SMALL_SPACE(USE_SMALL_SPACES, num);
  l4_thread_schedule(dest, schedparam, &foo_id, &foo_id, &schedparam);
#endif

  return 0;
}

int rmgr_set_prio(l4_threadid_t dest, int num)
{
#ifdef USE_PRIOS
  l4_sched_param_t schedparam;
  l4_threadid_t foo_id;

  static short mcp_below = 0xff;

  /* try to set directly first */
  if (num <= mcp_below)
    {
      foo_id = L4_INVALID_ID;
      l4_thread_schedule(dest, L4_INVALID_SCHED_PARAM, &foo_id, &foo_id, 
			 &schedparam);
      if (!l4_is_invalid_sched_param(schedparam))
	{
	  schedparam.sp.prio = num;
	  foo_id = L4_INVALID_ID;
	  l4_thread_schedule(dest, schedparam, &foo_id, &foo_id, &schedparam);
	  if (schedparam.sched_param != 0xffffffff)
	    return 0;
 
	  mcp_below = num - 1;
	  if (mcp_below < 0) mcp_below = 0;
	}
      else
	{
	  /* even failed to query the schedparam word... */
	  mcp_below = 0;
	}
    }

  /* failed to set directly -- use the RMGR */
  if (have_rmgr)
    return rmgr_call(RMGR_TASK_MSG(RMGR_TASK_SET_PRIO, num), dest.lh.low, 0);
  else
    return 1;			/* what happened ??? */

#else
  return 0;
#endif
}

int rmgr_get_task(int num)
{
#ifndef USE_SUPER_CLAN
  if (have_rmgr)
    return rmgr_call(RMGR_TASK_MSG(RMGR_TASK_GET, num), 0, 0);

  return 0;			/* if we don't have a supervisor, *we*
				   are the supervisoer ourselves and
				   already own all task create rights */
#else
  l4_taskid_t t;
  int err = rmgr_call(RMGR_TASK_MSG(RMGR_TASK_GET, num), 0, 0);

  if (err)
    return err;

  /* this is a valid task number; however, as we create tasks through
     RMGR, just return the task create right to RMGR */
  t = l4_myself();
  t.id.chief = t.id.task;
  t.id.task = num;
  t.id.nest++;
  l4_task_new(t, rmgr_id.lh.low, 0, 0, L4_NIL_ID);

  return 0;
#endif
}

int rmgr_get_irq(int num)
{
  if (have_rmgr)
    return rmgr_call(RMGR_IRQ_MSG(RMGR_IRQ_GET, num), 0, 0);

  return 0;
}

#ifdef USE_SUPER_CLAN
l4_taskid_t rmgr_task_new(l4_taskid_t dest, dword_t mcp_or_new_chief,
			  dword_t esp, dword_t eip, l4_threadid_t pager)
{
  l4_msgdope_t result;
  dword_t r1, r2;
  int err;

  static struct {
	dword_t fp;
	l4_msgdope_t size_dope;
	l4_msgdope_t send_dope;
	dword_t      d1, d2;
	l4_strdope_t data;
  } msg = {0, L4_IPC_DOPE(2, 1), L4_IPC_DOPE(2, 1), 0, 0, 
	   {16, 0, 0, 0}};

  int deleting = l4_is_nil_id(pager);

  msg.data.snd_str = (dword_t) &esp;

  if (deleting)
    err = l4_i386_ipc_call(rmgr_id, 
			   L4_IPC_SHORT_MSG, 
			   RMGR_TASK_MSG(RMGR_TASK_DELETE, dest.id.task),
			   0,
			   L4_IPC_SHORT_MSG, &r1, &r2,
			   L4_IPC_NEVER, &result);
  else
    err = l4_i386_ipc_call(rmgr_id, 
			   &msg, 
			   RMGR_TASK_MSG(RMGR_TASK_CREATE, mcp_or_new_chief),
			   dest.lh.low,
			   L4_IPC_SHORT_MSG, &r1, &r2,
			   L4_IPC_NEVER, &result);

  if (err)
    return L4_NIL_ID;

  return ((l4_threadid_t){lh: {r1, r2}});
}

l4_taskid_t rmgr_task_new_with_prio(l4_taskid_t dest, dword_t mcp_or_new_chief,
				    dword_t esp, dword_t eip, 
				    l4_threadid_t pager, 
				    l4_sched_param_t sched_param)
{
  l4_msgdope_t result;
  dword_t r1, r2;
  int err;

  static struct {
	dword_t fp;
	l4_msgdope_t size_dope;
	l4_msgdope_t send_dope;
	dword_t      d1, d2;
	l4_strdope_t data;
  } msg = {0, L4_IPC_DOPE(2, 1), L4_IPC_DOPE(2, 1), 0, 0, 
	   {20, 0, 0, 0}};

  int deleting = l4_is_nil_id(pager);

  msg.data.snd_str = (dword_t) &esp;

  if (deleting)
    err = l4_i386_ipc_call(rmgr_id, 
			   L4_IPC_SHORT_MSG, 
			   RMGR_TASK_MSG(RMGR_TASK_DELETE, dest.id.task),
			   0,
			   L4_IPC_SHORT_MSG, &r1, &r2,
			   L4_IPC_NEVER, &result);
  else {
    err = l4_i386_ipc_call(rmgr_id, 
			   &msg, 
			   RMGR_TASK_MSG(RMGR_TASK_CREATE_WITH_PRIO,
					 mcp_or_new_chief),
			   dest.lh.low,
			   L4_IPC_SHORT_MSG, &r1, &r2,
			   L4_IPC_NEVER, &result);
  }

  if (err)
    return L4_NIL_ID;

  return ((l4_threadid_t){lh: {r1, r2}});
}
#endif
