/*
 *  linux/arch/i386/mm/fault.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 */

#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/head.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/pgtable.h>

#include "../include/l4_memory.h"
#include "../include/exclpage.h"
#include "../include/fake_interrupt.h"
#include "../include/debug.h"

extern void die_if_kernel(const char *,struct pt_regs *,long);

/*
 * This routine handles page faults.  It determines the address,
 * and the problem, and then passes it off to one of the appropriate
 * routines.
 *
 * error_code:
 *	bit 0 == 0 means no page found, 1 means protection fault
 *	bit 1 == 0 means read, 1 means write
 *	bit 2 == 0 means kernel, 1 means user-mode
int do_page_fault(struct pt_regs *regs, unsigned long address, 
		  unsigned long error_code)
 */
int l4_do_page_fault(unsigned long address, unsigned long error_code)
{
  struct vm_area_struct * vma;
  unsigned long page;

#ifdef DEBUG
  printk("(%x.%x)l4_do_pagefault called for pf %lx with error %lx from %lx\n",
	 l4_myself().id.task, l4_myself().id.lthread, 
	 address, error_code, *((unsigned long *)&address - 1)); 
#endif
  vma = find_vma(current->mm, address);
  if (!vma)
    { 
      printk("No vma found for %lx\n", address);
      goto bad_area;
    }
  if (vma->vm_start <= address)
    goto good_area;
  if (!(vma->vm_flags & VM_GROWSDOWN))
    {
#if 0
      printk("area grows up, starts at %lx and goes up to %lx\n",
	     vma->vm_start, vma->vm_end);
#endif
      goto bad_area;
    }
#if 0
  /* XXX I don't have esp at hand, therefore I can't check stack access */
  if (error_code & PF_EUSER) 
    {
      /*
       * accessing the stack below %esp is always a bug.
       * The "+ 32" is there due to some instructions (like
       * pusha) doing pre-decrement on the stack and that
       * doesn't show up until later..
       */
      if (address + 32 < regs->esp)
	goto bad_area;
    }
#endif
  if (expand_stack(vma, address))
    goto bad_area;
  /*
   * Ok, we have a good vm_area for this memory access, so
   * we can handle it..
   */
good_area:
  /*
   * was it a write?
   */
  if (error_code & PF_EWRITE) 
    {
      if (!(vma->vm_flags & VM_WRITE))
	{
#if 0
	  printk("write access to read only area\n");
#endif
	  goto bad_area;
	}
    } 
  else 
    {
      /* read with protection fault? */
      if (error_code & PF_EPROTECTION)
	{
	  printk("protection fault\n");
	  goto bad_area;
	}
      if (!(vma->vm_flags & (VM_READ | VM_EXEC)))
	{
	  printk("!(vma->vm_flags & (VM_READ | VM_EXEC))\n");
	  goto bad_area;
	}
    }
#if 0
  /* XXX don't know anything about vm86
   * Did it hit the DOS screen memory VA from vm86 mode?
   */
  if (regs->eflags & VM_MASK) 
    {
      unsigned long bit = (address - 0xA0000) >> PAGE_SHIFT;
      if (bit < 32)
	current->tss.screen_bitmap |= 1 << bit;
    }
#endif
  /* Protection fault ? */
  if (error_code & PF_EPROTECTION) 
    {
#ifdef TEST_VERIFY_AREA
      if (regs->cs == KERNEL_CS)
	printk("WP fault at %08x\n", regs->eip);
#endif
      do_wp_page(current, vma, address, error_code & PF_EWRITE);
      return 0;
    }
  do_no_page(current, vma, address, error_code & PF_EWRITE);
  return 0;
  /*
   * Something tried to access memory that isn't in our memory map..
   * Fix it, but check if it's kernel or user first..
   */
bad_area:
#if 0
  printk("l4_do_pagefault (%s): bad area, addr %lx, error %lx:(\n",
	 current->comm, address, error_code);
  enter_kdebug("bad access");
#endif
  if (error_code & PF_EUSER) 
    {
      current->tss.page_fault_addr = address;
      current->tss.ex_page->error_code = error_code;
      current->tss.ex_page->exception_number = 14;
      send_sig(SIGSEGV, current, 1);
      generate_fake_interrupt(current);
      return -1;
    }
  /*
   * Oops. The kernel tried to access some bad page. We'll have to
   * terminate things with extreme prejudice.
   */
  if ((unsigned long) (address-TASK_SIZE) < PAGE_SIZE) 
    {
      printk(KERN_ALERT "Unable to handle kernel NULL pointer dereference");
      pg0[0] = pte_val(mk_pte(0, PAGE_SHARED));
    } 
  else
    printk(KERN_ALERT "Unable to handle kernel paging request");
  printk(" at virtual address %08lx\n",address);
  page = (int)current->tss.page_dir;
  page = ((unsigned long *) page)[address >> 22];
  printk(KERN_ALERT "*pde = %08lx\n", page);
  if (page & PF_EPROTECTION) 
    {
      page &= PAGE_MASK;
      address &= 0x003ff000;
      page = ((unsigned long *) page)[address >> PAGE_SHIFT];
      printk(KERN_ALERT "*pte = %08lx\n", page);
    }
  panic("call to die_if_kernel(\"Oops\", regs, error_code);\n"
	"do_exit(SIGKILL);\n");
  return -1;
}

void seg_fault_above_task_size(unsigned long address)
{
  herc_printf("sfats %lx ", address);
  current->tss.page_fault_addr = address;
  current->tss.ex_page->error_code = PF_EUSER | PF_ENOTPRESENT | 
    ((address & PF_EWRITE) ? PF_EWRITE : PF_EREAD) ;
  current->tss.ex_page->exception_number = 14;
  send_sig(SIGSEGV, current, 1);
  generate_fake_interrupt(current);
  herc_printf("sfats_end\n");
}
