/*
 * Serial interface GDB stub
 *
 * Written (hacked together) by David Grothe (dave@gcom.com)
 *
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial.h>
#include <linux/serial_reg.h>
#include <linux/config.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/segment.h>
#include <asm/bitops.h>
#include <asm/system.h>

#define IRQ_T(info) ((info->flags & ASYNC_SHARE_IRQ) ? SA_SHIRQ : SA_INTERRUPT)


#define	GDB_BUF_SIZE	512		/* power of 2, please */

static char	gdb_buf[GDB_BUF_SIZE] ;
static int	gdb_buf_in_inx ;
static int	gdb_buf_in_cnt ;
static int	gdb_buf_out_inx ;

struct async_struct	*gdb_async_info ;
static int	gdb_async_irq ;

extern void	set_debug_traps(void) ;		/* GDB routine */
extern void	breakpoint(void) ;		/* GDB routine */


/*
 * Get a byte from the hardware data buffer and return it
 */
static int	read_data_bfr(struct async_struct *info)
{
    if (inb(info->port + UART_LSR) & UART_LSR_DR)
	return(inb(info->port + UART_RX));

    return( -1 ) ;

} /* read_data_bfr */


/*
 * Get a char if available, return -1 if nothing available.
 * Empty the receive buffer first, then look at the interface hardware.
 */
static int	read_char(struct async_struct *info)
{
    if (gdb_buf_in_cnt != 0)		/* intr routine has q'd chars */
    {
	int		chr ;

	chr = gdb_buf[gdb_buf_out_inx++] ;
	gdb_buf_out_inx &= (GDB_BUF_SIZE - 1) ;
	gdb_buf_in_cnt-- ;
	return(chr) ;
    }

    return(read_data_bfr(info)) ;	/* read from hardware */

} /* read_char */

/*
 * Wait until the interface can accept a char, then write it.
 */
static void	write_char(struct async_struct *info, int chr)
{
    while ( !(inb(info->port + UART_LSR) & UART_LSR_THRE) ) ;

    outb(chr, info->port+UART_TX);

} /* write_char */

/*
 * This is the receiver interrupt routine for the GDB stub.
 * It will receive a limited number of characters of input
 * from the gdb  host machine and save them up in a buffer.
 *
 * When the gdb stub routine getDebugChar() is called it
 * draws characters out of the buffer until it is empty and
 * then reads directly from the serial port.
 *
 * We do not attempt to write chars from the interrupt routine
 * since the stubs do all of that via putDebugChar() which
 * writes one byte after waiting for the interface to become
 * ready.
 *
 * The debug stubs like to run with interrupts disabled since,
 * after all, they run as a consequence of a breakpoint in
 * the kernel.
 *
 * Perhaps someone who knows more about the tty driver than I
 * care to learn can make this work for any low level serial
 * driver.
 */
static void gdb_interrupt(int irq, void *dev_id, struct pt_regs * regs)
{
    int			 chr ;
    struct async_struct	*info;
    
    info = gdb_async_info ;
    if (!info || !info->tty || irq != gdb_async_irq)
	    return;

    for (;;)
    {
	chr = read_data_bfr(info) ;
	if (chr < 0) break ;

	if (gdb_buf_in_cnt >= GDB_BUF_SIZE)
	{				/* buffer overflow, clear it */
	    gdb_buf_in_inx = 0 ;
	    gdb_buf_in_cnt = 0 ;
	    gdb_buf_out_inx = 0 ;
	    break ;
	}

	gdb_buf[gdb_buf_in_inx++] = chr ;
	gdb_buf_in_inx &= (GDB_BUF_SIZE - 1) ;
	gdb_buf_in_cnt++ ;
    }

} /* gdb_interrupt */

/*
 * Just a NULL routine for testing.
 */
void gdb_null(void)
{
} /* gdb_null */

/*
 * Hook an IRQ for GDB.
 *
 * This routine is called from the ioctl routine of the serial driver
 * to hand the interface over to the GDB stub code.
 */
int	gdb_hook_interrupt(struct async_struct *info)
{
#ifdef LINUX_ON_L4
    int		retval ;

    free_irq(info->irq, NULL);
    retval = request_irq(info->irq,
			 gdb_interrupt,
			 IRQ_T(info),
			 "GDB-stub", NULL);
    if (retval == 0)
	gdb_async_info = info ;
    else
	gdb_async_info = NULL ;

    if (retval)				/* error */
	return(retval) ;

    gdb_async_irq = info->irq ;
    info->IER = UART_IER_RDI;			/* rcvr intr only */
    outb(info->port + UART_IER, info->IER);	/* enable interrupts */

    /*
     * Call GDB routine to setup the exception vectors for the debugger
     */
    printk("gdb_hook_interrupt: set_debug_traps()\n") ;
    set_debug_traps() ;

    /*
     * Call the breakpoint() routine in GDB to start the debugging
     * session.
     */
    printk("gdb_hook_interrupt: breakpoint()\n") ;
    breakpoint() ;
    gdb_null() ;

    return(0) ;
#endif
} /* gdb_hook_interrupt */

/*
 * getDebugChar
 *
 * This is a GDB stub routine.  It waits for a character from the
 * serial interface and then returns it.  If there is no serial
 * interface connection then it returns a bogus value which will
 * almost certainly cause the system to hang.
 */
int	getDebugChar(void)
{
    volatile int	chr ;

#if PRNT
    printk("getDebugChar: ") ;
#endif

    if (gdb_async_info == NULL)
	chr = 0xFF ;
    else
    {
	while ( (chr = read_char(gdb_async_info)) < 0 ) ;
    }


#if PRNT
    printk("%c", chr > ' ' && chr < 0x7F ? chr : ' ') ;
#endif
    return(chr) ;

} /* getDebugChar */

/*
 * putDebugChar
 *
 * This is a GDB stub routine.  It waits until the interface is ready
 * to transmit a char and then sends it.  If there is no serial
 * interface connection then it simply returns to its caller, having
 * pretended to send the char.
 */
void	putDebugChar(int chr)
{
#if PRNT
    printk("putDebugChar: chr=%02x '%c', info_ptr=%lx\n", chr,
		chr > ' ' && chr < 0x7F ? chr : ' ', gdb_async_info) ;
#endif

    if (gdb_async_info == NULL) return ;

    write_char(gdb_async_info, chr) ;	/* this routine will wait */

} /* putDebugChar */
