/* 
        paride.c  (c) 1997  Grant R. Guenther <grant@torque.net>
                            Under the terms of the GNU public license.

	This is the base module for the family of device drivers
        that support parallel port IDE devices.  

*/

#define PI_VERSION      "0.94"

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/string.h>

#include "paride.h"

#define MAX_PROTOS	32

static struct pi_protocol	*protocols[MAX_PROTOS];


void pi_write_regr( PIA *pi, int cont, int regr, int val)

{	pi->proto->write_regr(pi,cont,regr,val);
}

int pi_read_regr( PIA *pi, int cont, int regr)

{	return pi->proto->read_regr(pi,cont,regr);
}

void pi_write_block( PIA *pi, char * buf, int count)

{	pi->proto->write_block(pi,buf,count);
}

void pi_read_block( PIA *pi, char * buf, int count)

{       pi->proto->read_block(pi,buf,count);
}

void pi_connect( PIA *pi)

{	pi->proto->connect(pi);
}

void pi_disconnect( PIA *pi)

{	pi->proto->disconnect(pi);
}

void pi_release( PIA *pi)

{	if (pi->reserved) release_region(pi->port,pi->reserved);
	pi->proto->dec_use();
}

#define WR(r,v)         pi_write_regr(pi,0,r,v)
#define RR(r)           (pi_read_regr(pi,0,r))

static int pi_test_proto( PIA *pi, char * scratch, int verbose )

{       int     j, k;
        int     e[2] = {0,0};

        pi_connect(pi);

        for (j=0;j<2;j++) {
                WR(6,0xa0+j*0x10);
                for (k=0;k<256;k++) {
                        WR(2,k^0xaa);
                        WR(3,k^0x55);
                        if (RR(2) != (k^0xaa)) e[j]++;
                        }
                }

        pi_disconnect(pi);

        if (verbose)
                printk("%s: %s: port 0x%x, mode  %d, test=(%d,%d)\n",
                        pi->device,pi->proto->name,pi->port,
			pi->mode,e[0],e[1]);

        return (e[0] && e[1]);  /* not here if both > 0 */
}

int  pi_register( PIP *pr)

{	int k;

	for (k=0;k<MAX_PROTOS;k++) 
	   if (protocols[k] && !strcmp(pr->name,protocols[k]->name)) {
		printk("paride: %s protocol already registered\n",pr->name);
		return 0;
	   }
	k = 0;
	while((k<MAX_PROTOS) && (protocols[k])) k++;
	if (k == MAX_PROTOS) {
		printk("paride: protocol table full\n");
		return 0;
	}
	MOD_INC_USE_COUNT;
	protocols[k] = pr;
	pr->index = k;
	if (!pr->test_proto) pr->test_proto = pi_test_proto;
	printk("paride: %s registered as protocol %d\n",pr->name,k);
	return 1;
}	

void pi_unregister( PIP *pr)

{	if (!pr) return;
	if (protocols[pr->index] != pr) {
		printk("paride: %s not registered\n",pr->name);
		return;
	} 
	protocols[pr->index] = 0;
	MOD_DEC_USE_COUNT;
}

static int pi_probe_mode( PIA *pi, int max, char * scratch, int verbose)

{	int	best, range;

	if (pi->mode != -1) {
		if (pi->mode >= max) return 0;
		range = 3;
		if (pi->mode >= pi->proto->epp_first) range = 8;
		if ((range == 8) && (pi->port % 8)) return 0;
		if (check_region(pi->port,range)) return 0;
		pi->reserved = range;
		return (!pi->proto->test_proto(pi,scratch,verbose));
	}
	best = -1;
	for(pi->mode=0;pi->mode<max;pi->mode++) {
		range = 3;
		if (pi->mode >= pi->proto->epp_first) range = 8;
		if ((range == 8) && (pi->port % 8)) break;
		if (check_region(pi->port,range)) break;
		pi->reserved = range;
		if (!pi->proto->test_proto(pi,scratch,verbose)) 
			best = pi->mode;
	}
	pi->mode = best;
	return (best > -1);
}

static int pi_probe_unit( PIA *pi, int unit, char * scratch, int verbose)

{	int max,s,e;

	s = unit; e = s+1;

	if (s == -1) { 
		s = 0; 
		e = pi->proto->max_units; 
	}

	if (pi->proto->test_port)
		max = pi->proto->test_port(pi);
	else max = pi->proto->max_mode;

	if (pi->proto->probe_unit) {
	   for (pi->unit=s;pi->unit<e;pi->unit++)
	      if (pi->proto->probe_unit(pi)) {
		 if (pi_probe_mode(pi,max,scratch,verbose)) return 1;
		 return 0;
	         }
	   return 0;
	   }

	return pi_probe_mode(pi,max,scratch,verbose);
}

int pi_init(PIA *pi, int autoprobe, int port, int mode, 
	    int unit, int protocol, int delay, char * scratch, 
	    int devtype, int verbose, char *device )

{	int p,k,s,e;
	int lpts[7] = {0x3bc,0x378,0x278,0x268,0x27c,0x26c,0};

	s = protocol; e = s+1;

	if (autoprobe) {
		s = 0; 
		e = MAX_PROTOS;
	} else if ((s < 0) || (s >= MAX_PROTOS) || (port <= 0) ||
		    (!protocols[s]) || (unit < 0) || 
		    (unit >= protocols[s]->max_units)) {
			printk("%s: Invalid parameters\n",device);
			return 0;
		        }

	for (p=s;p<e;p++) {
	  if (protocols[p]) {
		pi->proto = protocols[p];
		pi->proto->inc_use();
		if (delay == -1) pi->delay = pi->proto->default_delay;
		  else pi->delay = delay;
		pi->devtype = devtype;
		pi->device = device;
		pi->private = 0;
	        
		pi->mode = mode;
		if (port != -1) {
			pi->port = port;
			if (pi_probe_unit(pi,unit,scratch,verbose)) break;
			pi->port = 0;
		} else { 
			k = 0;
			while ((pi->port = lpts[k++]))
			   if (pi_probe_unit(pi,unit,scratch,verbose)) break;
      			if (pi->port) break;
		}
		pi->proto->dec_use();
	  }
	}

	if (!pi->port) {
		if (autoprobe) printk("%s: Autoprobe failed\n",device);
		else printk("%s: Adapter not found\n",device);
		return 0;
	}

	request_region(pi->port,pi->reserved,pi->device);

	pi->proto->log_adapter(pi,scratch,verbose);

	return 1;
}

#ifdef MODULE

int	init_module(void)

{	int k;

	for (k=0;k<MAX_PROTOS;k++) protocols[k] = 0;
	printk("paride: version %s installed\n",PI_VERSION);
	return 0;
}

void	cleanup_module(void)

{
}

#endif

/* end of paride.c */
