/*
 *  linux/mm/vmalloc.c
 *
 *  Copyright (C) 1993  Linus Torvalds
 */

#include <asm/system.h>

#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/head.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/malloc.h>
#include <linux/mm.h>

#include <asm/segment.h>
#include <asm/pgtable.h>

#ifdef LINUX_ON_L4
#include "../arch/l4-i386/include/l4_memory.h"
#include "../arch/l4-i386/include/debug.h"
#endif

struct vm_struct {
	unsigned long flags;
	void * addr;
#ifdef LINUX_ON_L4
	void * subaddr;
#endif
	unsigned long size;
	struct vm_struct * next;
};

static struct vm_struct * vmlist = NULL;

static inline void set_pgdir(unsigned long address, pgd_t entry)
{
	struct task_struct * p;

	for_each_task(p) {
		if (!p->mm)
			continue;
		*pgd_offset(p->mm,address) = entry;
	}
}

#ifdef LINUX_ON_L4
static inline void free_area_pte(pmd_t * pmd, unsigned long address, unsigned long size, unsigned long vm_address)
#else
static inline void free_area_pte(pmd_t * pmd, unsigned long address, unsigned long size)
#endif
{
	pte_t * pte;
	unsigned long end;

	if (pmd_none(*pmd))
		return;
	if (pmd_bad(*pmd)) {
		printk("free_area_pte: bad pmd (%08lx)\n", pmd_val(*pmd));
		pmd_clear(pmd);
		return;
	}
	pte = pte_offset(pmd, address);
	address &= ~PMD_MASK;
	end = address + size;
	if (end > PMD_SIZE)
		end = PMD_SIZE;
	while (address < end) {
		pte_t page = *pte;
#ifdef LINUX_ON_L4
		vm_pte_clear(pte, vm_address);
		vm_address += PAGE_SIZE;
#else
		pte_clear(pte);
#endif
		address += PAGE_SIZE;
		pte++;
		if (pte_none(page))
			continue;
		if (pte_present(page)) {
			free_page(pte_page(page));
			continue;
		}
		printk("Whee.. Swapped out page in kernel page table\n");
	}
}

static inline void free_area_pmd(pgd_t * dir, unsigned long address, unsigned long size)
{
	pmd_t * pmd;
	unsigned long end;
#ifdef LINUX_ON_L4
	unsigned long vm_address = address;
#endif

	if (pgd_none(*dir))
		return;
	if (pgd_bad(*dir)) {
		printk("free_area_pmd: bad pgd (%08lx)\n", pgd_val(*dir));
		pgd_clear(dir);
		return;
	}
	pmd = pmd_offset(dir, address);
	address &= ~PGDIR_MASK;
	end = address + size;
	if (end > PGDIR_SIZE)
		end = PGDIR_SIZE;
	while (address < end) {
#ifdef LINUX_ON_L4
		free_area_pte(pmd, address, end - address, vm_address);
		vm_address = (vm_address + PMD_SIZE) & PMD_MASK;
#else
		free_area_pte(pmd, address, end - address);
#endif
		address = (address + PMD_SIZE) & PMD_MASK;
		pmd++;
	}
}

static void free_area_pages(unsigned long address, unsigned long size)
{
	pgd_t * dir;
	unsigned long end = address + size;

	dir = pgd_offset(&init_mm, address);
	flush_cache_all();
	while (address < end) {
		free_area_pmd(dir, address, end - address);
		address = (address + PGDIR_SIZE) & PGDIR_MASK;
		dir++;
	}
	flush_tlb_all();
}

#ifdef LINUX_ON_L4
static inline int alloc_area_pte(pte_t * pte, unsigned long address, unsigned long size, unsigned long vm_address)
#else
static inline int alloc_area_pte(pte_t * pte, unsigned long address, unsigned long size)
#endif
{
	unsigned long end;

	address &= ~PMD_MASK;
	end = address + size;
	if (end > PMD_SIZE)
		end = PMD_SIZE;
	while (address < end) {
		unsigned long page;
		if (!pte_none(*pte))
			printk("alloc_area_pte: page already exists\n");
#ifdef LINUX_ON_L4
		page = __get_free_4mb_compatible_page(GFP_KERNEL, 0);
#else
		page = __get_free_page(GFP_KERNEL);
#endif
		if (!page)
			return -ENOMEM;
#ifdef LINUX_ON_L4
		vm_set_pte(pte, mk_pte(page, PAGE_KERNEL), vm_address);
		vm_address += PAGE_SIZE;
#else
		set_pte(pte, mk_pte(page, PAGE_KERNEL));
#endif
		address += PAGE_SIZE;
		pte++;
	}
	return 0;
}

static inline int alloc_area_pmd(pmd_t * pmd, unsigned long address, unsigned long size)
{
	unsigned long end;
#ifdef LINUX_ON_L4
	unsigned long vm_address = address;
#endif

	address &= ~PGDIR_MASK;
	end = address + size;
	if (end > PGDIR_SIZE)
		end = PGDIR_SIZE;
	while (address < end) {
		pte_t * pte = pte_alloc_kernel(pmd, address);
		if (!pte)
			return -ENOMEM;
#ifdef LINUX_ON_L4
		if (alloc_area_pte(pte, address, end - address, vm_address))
			return -ENOMEM;
		vm_address = (vm_address + PMD_SIZE) & PMD_MASK;
#else
		if (alloc_area_pte(pte, address, end - address))
			return -ENOMEM;
#endif
		address = (address + PMD_SIZE) & PMD_MASK;
		pmd++;
	}
	return 0;
}

static int alloc_area_pages(unsigned long address, unsigned long size)
{
	pgd_t * dir;
	unsigned long end = address + size;

	dir = pgd_offset(&init_mm, address);
	flush_cache_all();
	while (address < end) {
		pmd_t *pmd = pmd_alloc_kernel(dir, address);
		if (!pmd)
			return -ENOMEM;
		if (alloc_area_pmd(pmd, address, end - address))
			return -ENOMEM;
		set_pgdir(address, *dir);
		address = (address + PGDIR_SIZE) & PGDIR_MASK;
		dir++;
	}
	flush_tlb_all();
	return 0;
}

static inline void remap_area_pte(pte_t * pte, unsigned long address, unsigned long size,
	unsigned long offset)
{
	unsigned long end;

	address &= ~PMD_MASK;
	end = address + size;
	if (end > PMD_SIZE)
		end = PMD_SIZE;
	do {
		if (!pte_none(*pte))
			printk("remap_area_pte: page already exists\n");
#ifdef LINUX_ON_L4
		/* Don't try to map something, it will be done in
		   vremap after the page tables are set up, because we
		   have to handle 4 MB pages for adapter areas */
		vm_set_pte(pte, mk_pte(offset, PAGE_KERNEL), VM_PTE_NO_MAP);
#else
		set_pte(pte, mk_pte(offset, PAGE_KERNEL));
#endif
		address += PAGE_SIZE;
		offset += PAGE_SIZE;
		pte++;
	} while (address < end);
}

static inline int remap_area_pmd(pmd_t * pmd, unsigned long address, unsigned long size,
	unsigned long offset)
{
	unsigned long end;

	address &= ~PGDIR_MASK;
	end = address + size;
	if (end > PGDIR_SIZE)
		end = PGDIR_SIZE;
	offset -= address;
	do {
		pte_t * pte = pte_alloc_kernel(pmd, address);
		if (!pte)
			return -ENOMEM;
		remap_area_pte(pte, address, end - address, address + offset);
		address = (address + PMD_SIZE) & PMD_MASK;
		pmd++;
	} while (address < end);
	return 0;
}

static int remap_area_pages(unsigned long address, unsigned long offset, unsigned long size)
{
	pgd_t * dir;
	unsigned long end = address + size;

	offset -= address;
	dir = pgd_offset(&init_mm, address);
	flush_cache_all();
	while (address < end) {
		pmd_t *pmd = pmd_alloc_kernel(dir, address);
		if (!pmd)
			return -ENOMEM;
		if (remap_area_pmd(pmd, address, end - address, offset + address))
			return -ENOMEM;
		set_pgdir(address, *dir);
		address = (address + PGDIR_SIZE) & PGDIR_MASK;
		dir++;
	}
	flush_tlb_all();
	return 0;
}

#ifdef LINUX_ON_L4
/* If align != 0, align (round up) *both* size and returned address */
static struct vm_struct * get_vm_area(unsigned long size, unsigned long align)
#else
static struct vm_struct * get_vm_area(unsigned long size)
#endif
{
	void *addr;
	struct vm_struct **p, *tmp, *area;

	area = (struct vm_struct *) kmalloc(sizeof(*area), GFP_KERNEL);
	if (!area)
		return NULL;
	addr = (void *) VMALLOC_START;
#ifdef LINUX_ON_L4
	/* Round up according to alignment */
	if (align) {
		size = (size + align - 1) & ~(align - 1);
		addr = (void *)(((unsigned long)addr + align - 1)
				& ~(align - 1));
	}
	area->subaddr = NULL;
#endif
	area->size = size + PAGE_SIZE;
	area->next = NULL;
	for (p = &vmlist; (tmp = *p) ; p = &tmp->next) {
		if (size + (unsigned long) addr < (unsigned long) tmp->addr)
			break;
		addr = (void *) (tmp->size + (unsigned long) tmp->addr);
#ifdef LINUX_ON_L4
		if (align)
			addr = (void *)(((unsigned long)addr + align - 1)
					& ~(align - 1));
#endif
	}
	area->addr = addr;
	area->next = *p;
	*p = area;
	return area;
}

void vfree(void * addr)
{
	struct vm_struct **p, *tmp;

	if (!addr)
		return;
	if ((PAGE_SIZE-1) & (unsigned long) addr) {
		printk("Trying to vfree() bad address (%p)\n", addr);
		return;
	}
	for (p = &vmlist ; (tmp = *p) ; p = &tmp->next) {
#ifdef LINUX_ON_L4
		if (tmp->addr == addr || tmp->subaddr == addr) {
#else
		if (tmp->addr == addr) {
#endif
#warning WARNING!!!  Loosing virtual and physical memory. (YHP)
			*p = tmp->next;
			free_area_pages(VMALLOC_VMADDR(tmp->addr), tmp->size);
			kfree(tmp);
#if 0
#endif
			return;
		}
	}
	printk("Trying to vfree() nonexistent vm area (%p)\n", addr);
}

void * vmalloc(unsigned long size)
{
	void * addr;
	struct vm_struct *area;

	size = PAGE_ALIGN(size);
	if (!size || size > (MAP_NR(high_memory) << PAGE_SHIFT))
		return NULL;
#ifdef LINUX_ON_L4
	area = get_vm_area(size, 0);
#else
	area = get_vm_area(size);
#endif
	if (!area)
		return NULL;
	addr = area->addr;
	if (alloc_area_pages(VMALLOC_VMADDR(addr), size)) {
		vfree(addr);
		return NULL;
	}
	return addr;
}

#ifdef LINUX_ON_L4
void * vreserve(unsigned long size, unsigned long align)
{
	struct vm_struct *area;

	size = PAGE_ALIGN(size);
	if (!size)
		return NULL;
	area = get_vm_area(size, align);
	if (!area)
		return NULL;
	/* XXX I guess it's OK not to allocate any memory here?  I'm
	   somewhat unsure whether we must setup some page tables or
	   something like that. */
	return area->addr;
}
#endif

/*
 * Remap an arbitrary physical address space into the kernel virtual
 * address space. Needed when the kernel wants to access high addresses
 * directly.
 */
void * vremap(unsigned long offset, unsigned long size)
{
	void * addr;
	struct vm_struct * area;

	if (MAP_NR(offset) < MAP_NR(high_memory))
		return NULL;
	if (offset & ~PAGE_MASK)
		return NULL;
	size = PAGE_ALIGN(size);
	if (!size || size > offset + size)
		return NULL;
#ifndef LINUX_ON_L4
	area = get_vm_area(size);
	if (!area)
		return NULL;
	addr = area->addr;
#else
	/* XXX waste 4 MB for every vremap() request... :-( */
	area = get_vm_area(size, 0x400000);
	if (!area)
		return NULL;
	addr = (void *) ((unsigned long)(area->addr)
			 + (offset & (0x400000 - 1)));
	area->subaddr = addr;
#endif
	if (remap_area_pages(VMALLOC_VMADDR(addr), offset, size)) {
		vfree(addr);
		return NULL;
	}
#ifdef LINUX_ON_L4
	{
	  unsigned long end = offset + size;
	  unsigned long dest = (unsigned long)area->addr;
	  if (adapter_page(offset))
	    while(offset < end)
	      {
		if (dest & (0x400000 - 1))
		  {
		    printk("area->addr = %lx\n", dest);
		    enter_kdebug("vremap: area not 4MB aligned");
		  }
		dest &= ~(0x400000 - 1);
		if (request_mem_area_from_sigma0(dest, offset))
		  {
		    enter_kdebug("request region in kernel failed");
		  }
#ifdef DEBUG
		else
		  enter_kdebug("vremap: after request region");
#endif
		offset += 0x400000;
		dest += 0x400000;
	      }
	  else
	    while(offset < end)
	      {
		request_physical_page_from_sigma0(offset, L4_WRITE_ACCESS, dest);
		offset += PAGE_SIZE;
		dest += PAGE_SIZE;
	      }
	}
#endif
	return addr;
}

int vread(char *buf, char *addr, int count)
{
	struct vm_struct **p, *tmp;
	char *vaddr, *buf_start = buf;
	int n;

	for (p = &vmlist; (tmp = *p) ; p = &tmp->next) {
		vaddr = (char *) tmp->addr;
		while (addr < vaddr) {
			if (count == 0)
				goto finished;
			put_user('\0', buf++), addr++, count--;
		}
		n = tmp->size - PAGE_SIZE;
		if (addr > vaddr)
			n -= addr - vaddr;
		while (--n >= 0) {
			if (count == 0)
				goto finished;
			put_user(*addr++, buf++), count--;
		}
	}
finished:
	return buf - buf_start;
}
