/* $Id: process.c,v 1.5 1999/05/25 17:23:51 volkmar Exp $
 *
 *  linux/arch/l4-i386/kernel/process.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 */

/*
 * This file handles the architecture-dependent parts of process handling..
 */

#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>

#include <linux/ptrace.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/a.out.h>
#include <linux/user.h>
#include <linux/smp_lock.h>
#include <asm/segment.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/processor.h>
#include <asm/ptrace.h>
#include <asm/uaccess.h>

#include "../include/config.h"
#include "../include/task.h"
#include "../include/sched.h"
#include "../include/user.h"
#include "../include/shared_data.h"
#include "../include/assert.h"
#include "../include/l4_memory.h"

#define __KERNEL_SYSCALLS__
#include <linux/unistd.h>

#ifdef USE_SUPER_CLAN
#include "../include/rmgr.h"
# define l4_task_new rmgr_task_new
#endif

#define DEBUG_STACK
/* #define DEBUG */
/* msg transfer */

int idle_sleeping = 0;

asmlinkage int sys_idle(void)
{
  void l4_idle(void *kernel_sp); /* in l4_idle.S */

  if (current->pid != 0)
    return -EPERM;

  current->counter = -100;
  l4_idle(&current->tss.kernel_sp);
  return 0;			/* never reached? */
}

asmlinkage int sys_enosys(void)
{
  return -ENOSYS;
}

/*
 * This routine reboots the machine by asking the keyboard
 * controller to pulse the reset-line low. We try that for a while,
 * and if it doesn't work, we do some other stupid things.
 */
#if 0
static long no_idt[2] = {0, 0};
#endif

static inline void kb_wait(void)
{
	int i;

	for (i=0; i<0x10000; i++)
		if ((inb_p(0x64) & 0x02) == 0)
			break;
}

void machine_restart(char *unused)
{
	int i, j;

	sti();
#ifdef DEBUG
	enter_kdebug("hard_reset_now");
#endif
/* rebooting needs to touch the page at absolute addr 0 */
	pg0[0] = 7;		/* XXX make pg0 writable */
	request_zero_page_from_sigma0();
	*((unsigned short *)0x472) = 0x1234;
	for (;;) {
		for (i=0; i<100; i++) {
			kb_wait();
			for(j = 0; j < 100000 ; j++)
				/* nothing */;
			outb(0xfe,0x64);	 /* pulse reset low */
		}
#if 0
		/* XXX generate triple fault */
		__asm__ __volatile__("\tlidt %0": "=m" (no_idt));
#endif
	}
}

void machine_halt(void)
{
}

void machine_power_off(void)
{
#if defined(CONFIG_APM) && defined(CONFIG_APM_POWER_OFF)
	apm_power_off();
#endif
}


void show_regs(struct pt_regs * regs)
{
  printk("\n");

  if (! regs)
    {
      printk("Can't print regs from interrupt handler: &pt_regs == 0!");
      return;
    }

  printk("EIP: %08lx", regs->eip);
  printk(" ESP: %08lx", regs->esp);
  printk(" EFLAGS: %08lx\n",regs->eflags);
  printk("EAX: %08lx EBX: %08lx ECX: %08lx EDX: %08lx\n",
	 regs->eax,regs->ebx,regs->ecx,regs->edx);
  printk("ESI: %08lx EDI: %08lx EBP: %08lx\n",
	 regs->esi, regs->edi, regs->ebp);
}


/*
 * Allocation and freeing of basic task resources.
 *
 * NOTE! The task struct and the stack go together
 *
 * The task structure is a two-page thing, and as such
 * not reliable to allocate using the basic page alloc
 * functions. We have a small cache of structures for
 * when the allocations fail..
 *
 * This extra buffer essentially acts to make for less
 * "jitter" in the allocations..
 *
 * On SMP we don't do this right now because:
 *  - we aren't holding any locks when called, and we might
 *    as well just depend on the generic memory management
 *    to do proper locking for us instead of complicating it
 *    here.
 *  - if you use SMP you have a beefy enough machine that
 *    this shouldn't matter..
 */
#ifndef __SMP__
#define EXTRA_TASK_STRUCT	16
static struct task_struct * task_struct_stack[EXTRA_TASK_STRUCT];
static int task_struct_stack_ptr = -1;
#endif

struct task_struct * alloc_task_struct(void)
{
#ifndef EXTRA_TASK_STRUCT
	return (struct task_struct *) __get_free_pages(GFP_KERNEL,1);
#else
	int index;
	struct task_struct *ret;

	index = task_struct_stack_ptr;
	if (index >= EXTRA_TASK_STRUCT/2)
		goto use_cache;
	ret = (struct task_struct *) __get_free_pages(GFP_KERNEL,1);
	if (!ret) {
		index = task_struct_stack_ptr;
		if (index >= 0) {
use_cache:
			ret = task_struct_stack[index];
			task_struct_stack_ptr = index-1;
		}
	}
	return ret;
#endif
}

void free_task_struct(struct task_struct *p)
{
#ifdef EXTRA_TASK_STRUCT
	int index = task_struct_stack_ptr+1;

	if (index < EXTRA_TASK_STRUCT) {
		task_struct_stack[index] = p;
		task_struct_stack_ptr = index;
	} else
#endif
		free_pages((unsigned long) p, 1);
}


/* we don't have segments (hardware segments), so we simply provide
 * some dummy routine to be able to link the kernel. */

void copy_segments(int nr, struct task_struct *p, struct mm_struct *new_mm)
{
  /* If new_mm is NULL, we're being called to set up the LDT
   * descriptor * for a clone task. Each clone must have a separate
   * entry in the GDT.  */
}

void release_segments(struct mm_struct *mm)
{
}

void forget_segments(void)
{
}

/*
 * Create a kernel thread
 */
pid_t kernel_thread(int (*fn)(void *), void * arg, 
		    unsigned long flags)
{
  pid_t pid;

#if (48 != PTREGS_SIZE)
# error "adapt me"
#endif

  /* we don't care about errno, that's why we call the system calls
     directly (sys_*() routines). */
  flags |= CLONE_L4_KERNEL;

  __asm__ __volatile__
    (
     /* clone() expects the caller's struct pt_regs on the stack
	which will become the new task's state.  we fake that struct
	here. */

     "pushl $0			\n\t" /* esp -- fake */
     "pushfl			\n\t" /* eflags */
     "pushl $0			\n\t" /* cs */
     "pushl $1f			\n\t" /* eip -- child return address */
     "pushl %%eax		\n\t" /* orig_eax */
     "pushl %%eax		\n\t" /* eax */
     "pushl %%ebp		\n\t" /* ebp */
     "pushl %%edi		\n\t" /* edi */
     "pushl %%esi		\n\t" /* esi */
     "pushl %%edx		\n\t" /* edx */
     "pushl $0			\n\t" /* ecx -- user stack arg */
     "pushl %%ebx		\n\t" /* ebx -- clone flags */
     "call " SYMBOL_NAME_STR(clone) "\n\t" /* call clone */
     "jmp   0f			\n"   /* parent */

     /* this is the point where the child returns, with a new stack
        (so don't junk the clone() args in this case) */

   "1: pushl %3			\n\t" /* child: push arg */
     "call  *%2			\n\t" /* child: call fn */
     "movl  $0,(%%esp)		\n\t" /* child: push arg for sys_exit */
     "call " SYMBOL_NAME_STR(sys_exit) "\n" /* child: call sys_exit */

     /* parent */

   "0: leal 48(%%esp), %%esp	\n\t" /* parent: junk sys_clone args */

     : "=a" (pid)
     : "0" (__NR_clone), "r" (fn), "r" (arg),
       "b" (flags | CLONE_VM)
     : "ecx", "edx"
     );

  return pid;
}

/*
 * Allocate/free current thread data structures etc..
 */

/* helpers routines  */

extern int l4_task_version;

l4_taskid_t template_id;
l4_sched_param_t template_schedparam;

/* create a new task number for a new user task */
static int get_new_taskno(l4_taskid_t *task)
{
  int n = task_no_alloc();

  if (n < 0)
    return n;

  *task = template_id;
  task->id.lthread = 0;
  task->id.task = n;
  task->id.version_low = l4_task_version;

#ifdef DEBUG
  printk("get_new_taskno: allocated id: %x\n", task->lh.low);
#endif
  return 0;
}

/* delete a task */
extern inline int 
task_delete(l4_threadid_t task)
{
  l4_task_new(task, kernel_thread_id.lh.low, 0, 0, L4_NIL_ID);
  return 0;
}

/* called by do_exit(); kills the Linux user thread */
void exit_thread(void)
{
  /* check if we were a kernel-internal thread (i.e., have no
     user-space partner */
  if (! thread_equal(current->tss.user_thread_id, L4_NIL_ID))
    {
      l4_taskid_t task;

      /* kill the associated user task */
      task = get_taskid(current->tss.user_thread_id);
#ifdef DEBUG
      printk("exit_thread: trying to delete %x\n", 
	     task.lh.low);
#if 0
      enter_kdebug("exit_thread");
#endif
#endif

      if (task_delete(task)
	  < 0)
	{
	  panic("exit_thread: root_task_execute(task_delete) failed");
	}

      task_no_free(task.id.task);
      clear_task_to_proc_entry(task.id.task);
    }

  free_page((unsigned long) current->tss.shared_data);

  /* we can't delete the L4 thread running in kernel mode associated
     with this Linux task because _we_ are the thread in question... 
     That's why, leave the work to release_thread(). */
}

/* called by release(); should free any data structures still held
   after exit_thread() */
void release_thread(struct task_struct *dead_task)
{
}

/* helpers for copy_thread() */

/* initialize a new kernel thread created with kernel_thread().  This
   is the first routine running in the new context.  It is invoked
   with the first switch_to() to this process.  */
static void
kernel_thread_init(struct task_struct *p, unsigned long clone_flags)
{
#ifdef DEBUG
  printk("(%x.%x)kernel_thread_init: called with task_struct %x, "
	 "clone_flags: %x\n",
	 l4_myself().id.task, l4_myself().id.lthread,
	 (unsigned int)p, (unsigned int)clone_flags);
  if (p->tss.cr3 == swapper_pg_dir)
    printk("kernel_thread_init: p->tss.cr3 == swapper_pg_dir\n");
  else
    printk("kernel_thread_init: p->tss.cr3 (%lx) "
	   "!= swapper_pg_dir (%lx)\n", (unsigned long)p->tss.cr3, 
	   (unsigned long)swapper_pg_dir);
  show_regs(&p->tss.ex_page->regs);
#endif

  struct pt_regs *pt_regsp = &p->tss.shared_data->regs;
  
  if (p != current)
    panic("kernel_thread_init: i'm not current");

  if (p->tss.cr3 != swapper_pg_dir)
    panic("kernel_thread_init: not a kernel thread");
  
  /* restore registers and continue within new thread */
#define __MOV_REG(offs,reg) "movl (4*" #offs ")(%%ebx)," reg "\n\t"
#define MOV_REG(offs,reg) __MOV_REG(offs, "%%" #reg)
  asm volatile
    (
     "pushl	%%eax	\n\t" /* push the return address (new control 
				 flow */
     "pushl	%%ecx	\n\t" /* push the value of eflags */
     "pushl	%%edx	\n\t" /* push the value of ebx */
     
     MOV_REG(EAX, eax)
     MOV_REG(ECX, ecx)
     MOV_REG(EDX, edx)
     MOV_REG(ESI, esi)
     MOV_REG(EDI, edi)
     MOV_REG(EBP, ebp)
     
     "popl		%%ebx	\n\t" /* restore ebx */
     "popfl 		\n\t" /* restore eflags */
     /* asm_enter_kdebug("kernel_thread_init: returning to ptregs.eip")  */
     "ret			\n\t" /* return to original control flow */
     : 
     /* No Output */
     : 
     "b" (pt_regsp),	/* address of pt regs structure */
     "a" (pt_regsp->eip),	/* EIP */
     "c" (pt_regsp->eflags), /* EFLAGS */
     "d" (pt_regsp->ebx)	/* EBX */
     );

  panic("kernel_thread_init: return after jump to new control flow");
}

/* create an L4 task (user mode) for the new task_struct;
   copy_thread() will invoke this routine in the Linux server's root
   task */
static int 
thread_create_user(struct thread_struct *t)
{
  l4_taskid_t task, foo_id;
  unsigned long *sp;
  extern l4_sched_param_t template_schedparam;
  l4_sched_param_t schedparam;

  /* warning: sp is an address in user space; don't try to dereference
     it in the kernel */
  sp = (unsigned long *)(USIGNAL_STACK_INIT);
  atomic_set(&t->under_kernel_control, 0);
  atomic_set(&t->shared_data->emu_local_lock, 1);

#ifdef USE_KEVS_TASKSERV
  rmm_pgr_get_rm_start (rmmservice, &rmmstart, &exc);
  init_ds.dsm = l4_myself(); /* should optimise this */
  strdope.snd_size = sizeof(init_ds);
  strdope.snd_str = (dword_t) &init_ds;
  strdope.rcv_size = 0;
  strdope.rcv_str = 0;
  taskserv_def_create (taskservice, rmmservice.dw, strdope, rmmstart, 0, 6,
                       8192, &task.lh.low, &exc);
  if (exc._type != exc_l4_no_exception)
    task = L4_NIL_ID;
#else
#ifndef USE_SUPER_CLAN
  task = l4_task_new(t->user_thread_id, 0, (dword_t) sp, 
		     (dword_t) user_task_init, kernel_thread_id); 
#else
  task = rmgr_task_new_with_prio(t->user_thread_id, 0, 
				 (dword_t) sp, (dword_t) user_task_init, 
				 kernel_thread_id, template_schedparam); 
#endif /* USE_SUPER_CLAN */
#endif /* USE_KEVS_TASKSERV */

  if (thread_equal(task, L4_NIL_ID)) 
    {
      printk("thread_create_user: Failed to create user task\n");
      return -EBUSY;
    }
#ifndef USE_SUPER_CLAN
  /* set prio of user task */
  foo_id = L4_INVALID_ID;
  l4_thread_schedule(task, template_schedparam, &foo_id, &foo_id, 
		     &schedparam);
#endif  

  nr_running_user++;

  return 0;
}

/* Create the kernel context for a new process.  Our main duty here is
   to fill in p->tss, the arch-specific part of the process'
   task_struct */
static int 
thread_create(struct task_struct *p,
	      unsigned long clone_flags)
{
  struct thread_struct *t = &p->tss;
  int code;
  unsigned long *sp;

  /* first, allocate task id for  client task */
  if (!(clone_flags & CLONE_L4_KERNEL)) /* is this a user process? */
    {
      if (get_new_taskno(&t->user_thread_id) < 0)
	{
	  printk("thread_create: No task no left for user\n"); 
	  return -EBUSY;
	}
    }
  else
    {
      /* we're a kernel process */
	t->cr3 = swapper_pg_dir;
	t->user_thread_id = L4_NIL_ID;
    }

  /* first, create the kernel part of the new task pair */
  t->shared_data->kernel_id = get_l4_id_from_stack();
  t->shared_data->user_id = t->user_thread_id; /* XXX this is just sanity */

  /* compute pointer to end of stack */
  sp = (unsigned long *) ((unsigned long)p + sizeof(union task_union));
  /* leave room for stack id and 4 extra bytes used by l4_idle.S */
  sp -= L4_IDLE_STACK_USAGE / sizeof(*sp);
  /* put thread id to top of stack */
  put_l4_id_to_stack((unsigned)sp, get_l4_id_from_stack());
  
  /* Emulate function call to kernel_thread_init() with two parameters
     on stack.  We will actually only be scheduled (and call
     kernel_thread_init() if this is a kernel-internal thread, but we
     put the routine call there anyway to trap errors. */
  *(--sp) = clone_flags;
  *(--sp) = (unsigned long) p;
  *(--sp) = 0;			/* fake return address */

  /* switch_to will expect the new program pointer on the stack */
  *(--sp) = (unsigned long) kernel_thread_init;

  t->kernel_sp = (unsigned long) sp;

  /* if creating a kernel-internal thread, return at this point */
  if (t->cr3 == swapper_pg_dir)
    {
      /* cause this thread to be scheduled in schedule() */
      atomic_set(&t->under_kernel_control, 1);
/*       need_resched = 1; */
      return 0;
    }

  /* now create the user part of the new task */
  if ((code = thread_create_user(t)) < 0)
    {
      /* kill kernel task */
      task_no_free(t->user_thread_id.id.task);
      
      return code;
    }
  /* setup task-to-proc_struct map entry */
  set_task_to_proc_entry(t->user_thread_id.id.task, p);			

  return 0;
}

/* set up a new task_struct p's tss/ldt/gdt with registers from regs
   (containing the do_fork() caller's regs).  create an L4 user task
   for the new task_struct */
int copy_thread(int nr, unsigned long clone_flags, unsigned long esp,
		struct task_struct * p, struct pt_regs * regs)
{
  int retval;
  struct pt_regs * childregs;

  /* initialize tss */

  /* allocate a new exclusive page */
  p->tss.shared_data = (struct shared_data *) get_free_page(GFP_KERNEL);
  if (! p->tss.shared_data)
    {
      return -EBUSY;
    }

  /* set up regs for child */
  childregs = & p->tss.shared_data->regs;
  *childregs = *regs;
  childregs->eax = 0;
  childregs->esp = esp;
  childregs->eflags |= 0x200;	/* sanity: set EI flag */

#ifdef DEBUG
  printk("copy_thread: esp: %x on_page: %x\n",(int)esp,(int)&childregs->esp);
#endif

  /* initialize fs to the old task's value.  this way, we make sure
     new kernel threads get KERNEL_DS, while new user threads get
     USER_DS in here.  when we go from kernel to user space in
     execve(), we change the value from KERNEL_DS to USER_DS (see
     below) */

  /* V: unfortunatly this assumption was wrong, since user tasks can also
     force a kernel_thread (like a mount of an nfs-drive creates the 
     rpcio-daemon. Therefore we have another (L4-specific) clone flag.
     We use this flag to identify, if the new thread should become
     a user or a kernel thread */
     
  set_tsk_fs(p, get_fs());

  /* create the user task */
  retval = thread_create(p, clone_flags);
  if (retval < 0)
    {
      free_page((unsigned long) p->tss.shared_data);
      return retval;
    }

  return 0;
}

/* called by do_execve()/.../flush_old_exec(); should recycle the thread
   so that a new process can run in it */
void flush_thread(void)
{
  l4_taskid_t task = get_taskid(current->tss.user_thread_id);

  if ((! thread_equal(task, L4_NIL_ID))
      && task_delete(task) < 0)
    {
      /* XXX what should we do here? */
      enter_kdebug("flush_thread: delete failed");
    }
}

/* called by do_execve() */
void start_thread(struct pt_regs * regs, unsigned long eip, 
		  unsigned long esp)
{
  regs->eip = eip;
  regs->esp = esp;

  atomic_set(&current->tss.under_kernel_control, 2);

  if (eip > TASK_SIZE)
    send_sig(SIGSEGV, current, 0);
}

/* called by l4_idle.S when activating a freshly-started thread, and
   by the (kernel-internal) execve() when we want to convert a
   kernel-internal thread to a user process. */
void start_thread_really(void)
{
  struct thread_struct *t = &current->tss;

  if (atomic_read(&t->under_kernel_control) != 2)
    enter_kdebug("ukk != 2");

  /* create the task */

  if (thread_create_user(t) < 0)
    {
      task_no_free(t->user_thread_id.id.task);

      /* XXX what should we do here? */
      enter_kdebug("start thread failed");

      sig_kill_received();
    }

  /* Note: thread_create_user() will have the side effect of
     cleaning the processes under_kernel_control flag -- which is okay */
}


/* core dump routines */

int dump_fpu (struct user_i387_struct *foo)
{
  return 0;			/* XXX can't dump fpu state */
}

/* next is from arch/i386/kernel/process.c and adjusted to new pt_regs
   structure */
void dump_thread(struct pt_regs * regs, struct user * dump)
{
  int i;

  /* changed the size calculations - should hopefully work better. lbt */
  dump->magic = CMAGIC;
  dump->start_code = 0;
  dump->start_stack = regs->esp & ~(PAGE_SIZE - 1);
  dump->u_tsize = ((unsigned long) current->mm->end_code)
    >> PAGE_SHIFT;
  dump->u_dsize = ((unsigned long) (current->mm->brk + (PAGE_SIZE-1))) 
    >> PAGE_SHIFT;
  dump->u_dsize -= dump->u_tsize;
  dump->u_ssize = 0;
#if 0				/* XXX */
  for (i = 0; i < 8; i++)
    dump->u_debugreg[i] = current->debugreg[i];  
#else
  for (i = 0; i < 8; i++)
    dump->u_debugreg[i] = 0;
#endif

  if (dump->start_stack < TASK_SIZE)
    dump->u_ssize = ((unsigned long) (TASK_SIZE - dump->start_stack)) 
      >> PAGE_SHIFT;

  /* copy register contents */
  memset((void *)&dump->regs, 0, (size_t)sizeof(dump->regs));
  dump->regs.ebx = regs->ebx; 
  dump->regs.ecx = regs->ecx; 
  dump->regs.edx = regs->edx; 
  dump->regs.esi = regs->esi; 
  dump->regs.edi = regs->edi; 
  dump->regs.ebp = regs->ebp; 
  dump->regs.eax = regs->eax; 
  dump->regs.orig_eax = regs->orig_eax; 
  dump->regs.eip = regs->eip; 
  dump->regs.eflags = regs->eflags; 
  dump->regs.esp = regs->esp;
  
  dump->u_fpvalid = dump_fpu (&dump->i387);
}

/* fork/exec system calls (copied from arch/i386/kernel/process.c) */

asmlinkage int sys_fork(void)
{
        struct pt_regs *regs = &current->tss.shared_data->regs;
	return do_fork(SIGCHLD, regs->esp, regs);
}

/* user-visible clone system call */
/* sys_*(ebx, ecx, edx, esi, edi); */
asmlinkage int sys_clone(unsigned long clone_flags, unsigned long newsp)
{
	struct pt_regs *regs = &current->tss.shared_data->regs;

	if (!newsp)
		newsp = regs->esp;

	clone_flags &= ~CLONE_L4_KERNEL;

	return do_fork(clone_flags, newsp, regs);
}

/* kernel-internal clone (called from kernel_thread()) */
asmlinkage int clone(struct pt_regs regs)
{

	unsigned long clone_flags;
	unsigned long newsp;

	clone_flags = regs.ebx;
	newsp = regs.ecx;
	if (!newsp)
		newsp = regs.esp;

	return do_fork(clone_flags, newsp, &regs);
}

/*
 * This is trivial, and on the face of it looks like it
 * could equally well be done in user mode.
 *
 * Not so, for quite unobvious reasons - register pressure.
 * In user mode vfork() cannot have a stack frame, and if
 * done by calling the "clone()" system call directly, you
 * do not have enough call-clobbered registers to hold all
 * the information you need.
 */
asmlinkage int sys_vfork(struct pt_regs regs)
{
	return do_fork(CLONE_VFORK | CLONE_VM | SIGCHLD, regs.esp, &regs);
}

/*
 * sys_execve() executes a new program.
 */
/* sys_*(ebx, ecx, edx, esi, edi); */
asmlinkage int sys_execve(char *name, char **argv, char **envp)
{
	int error;
	char * filename;

	lock_kernel();
	filename = getname(name);
	error = PTR_ERR(filename);
	if (IS_ERR(filename))
		goto out;
	error = do_execve(filename, argv, envp, 
			  &current->tss.shared_data->regs);
	if (error == 0)
		current->flags &= ~PF_DTRACE;
	putname(filename);
out:
	unlock_kernel();
	return error;
}

/* kernel-internal execve() */
int execve(char * file, char ** argv, char ** envp)
{
  int error;

  ASSERT(thread_equal(current->tss.user_thread_id, L4_NIL_ID));

  /* we are going to become a real user task now, so prepare a real
     pt_regs structure.
     this assumes the kernel doesn't use shared_data->regs for
     kernel-internal tasks except for clone()ing.
     we don't need to zero-initalize the registers because
     get_free_page() already did it for us */
  current->tss.shared_data->regs.eflags = 0x200; /* XXX hardcoded */

  /* we're about to exec, so get a task id now */
  if (get_new_taskno(&current->tss.user_thread_id) < 0)
    {
      printk("execve: No task no left for user\n"); 
      errno = -EBUSY;

      return -1;
    }

  /* now the fun part: we're switching to user space, so set fs to
     point to user space */
  ASSERT(segment_eq(get_fs(), KERNEL_DS));
  set_fs(USER_DS);

  /* do_execve() will create the user task for us in
     start_thread(). */

  lock_kernel();
  error = do_execve(file, argv, envp, & current->tss.shared_data->regs);

  if (error)
    {
      /* we failed -- become a kernel thread again */
      task_no_free(current->tss.user_thread_id.id.task);
      set_fs(KERNEL_DS);
      current->tss.user_thread_id = L4_NIL_ID;

      errno = -error;

      return -1;
    }

  start_thread_really();	/* implicitly sets under_kernel_control = 0 */
  set_task_to_proc_entry(current->tss.user_thread_id.id.task, current);

  unlock_kernel();

  /* now become a service thread */
  schedule();			/* sleep and wait for user activation */

#if 1
  enter_kdebug("kexecve: returned from schedule");
#endif
  sig_kill_received();

  /* not reached */
  enter_kdebug("not reached");
  return 0;
}
