/*
 *  linux/arch/i386/traps.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

/*
 * 'Traps.c' handles hardware traps and faults after we have saved some
 * state in 'asm.s'.
 */
#include <linux/config.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/ptrace.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/delay.h>

#ifdef CONFIG_MCA
#include <linux/mca.h>
#include <asm/processor.h>
#endif

#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/spinlock.h>
#include <asm/atomic.h>
#include <asm/smp.h>

#include "../include/shared_data.h"
#include "../include/debug.h"

static inline void console_verbose(void)
{
	extern int console_loglevel;
	console_loglevel = 15;
}

#define DO_ERROR(trapnr, signr, str, name, tsk) \
void do_##name(struct pt_regs * regs, long error_code) \
{ \
	tsk->tss.error_code = error_code; \
	tsk->tss.trap_no = trapnr; \
	force_sig(signr, tsk); \
	die_if_no_fixup(str,regs,error_code); \
}

void divide_error(void);
void debug(void);
void nmi(void);
void int3(void);
void overflow(void);
void bounds(void);
void invalid_op(void);
void device_not_available(void);
void double_fault(void);
void coprocessor_segment_overrun(void);
void invalid_TSS(void);
void segment_not_present(void);
void stack_segment(void);
void general_protection(void);
void page_fault(void);
void coprocessor_error(void);
void reserved(void);
void alignment_check(void);
void spurious_interrupt_bug(void);

int kstack_depth_to_print = 24;

/*
 * These constants are for searching for possible module text
 * segments.  VMALLOC_OFFSET comes from mm/vmalloc.c; MODULE_RANGE is
 * a guess of how much space is likely to be vmalloced.
 */
#define VMALLOC_OFFSET (8*1024*1024)
#define MODULE_RANGE (8*1024*1024)

static void show_registers(struct pt_regs *regs)
{
	int i;
	int in_kernel = 1;
	unsigned long esp;
	unsigned short ss;
	unsigned long *stack, addr, module_start, module_end;
	extern char _stext, _etext;

	esp = (unsigned long) (1+regs);
	ss = 0;

	printk("CPU:    %d\nEIP:    %04x:[<%08lx>]\nEFLAGS: %08lx\n",
		smp_processor_id(), 0, regs->eip, regs->eflags);
	printk("eax: %08lx   ebx: %08lx   ecx: %08lx   edx: %08lx\n",
		regs->eax, regs->ebx, regs->ecx, regs->edx);
	printk("esi: %08lx   edi: %08lx   ebp: %08lx   esp: %08lx\n",
		regs->esi, regs->edi, regs->ebp, esp);
	printk("ds: %04x   es: %04x   ss: %04x\n",
		0, 0, ss);
	printk("Process %s (pid: %d, stackpage=%08lx)",
		current->comm, current->pid, 4096+(unsigned long)current);

	/*
	 * When in-kernel, we also print out the stack and code at the
	 * time of the fault..
	 */
	if (in_kernel) {
		printk("\nStack: ");
		stack = (unsigned long *) esp;
		for(i=0; i < kstack_depth_to_print; i++) {
			if (((long) stack & 4095) == 0)
				break;
			if (i && ((i % 8) == 0))
				printk("\n       ");
			printk("%08lx ", *stack++);
		}
		printk("\nCall Trace: ");
		stack = (unsigned long *) esp;
		i = 1;
		module_start = PAGE_OFFSET + (max_mapnr << PAGE_SHIFT);
		module_start = ((module_start + VMALLOC_OFFSET) & ~(VMALLOC_OFFSET-1));
		module_end = module_start + MODULE_RANGE;
		while (((long) stack & 4095) != 0) {
			addr = *stack++;
			/*
			 * If the address is either in the text segment of the
			 * kernel, or in the region which contains vmalloc'ed
			 * memory, it *may* be the address of a calling
			 * routine; if so, print it so that someone tracing
			 * down the cause of the crash will be able to figure
			 * out the call path that was taken.
			 */
			if (((addr >= (unsigned long) &_stext) &&
			     (addr <= (unsigned long) &_etext)) ||
			    ((addr >= module_start) && (addr <= module_end))) {
				if (i && ((i % 8) == 0))
					printk("\n       ");
				printk("[<%08lx>] ", addr);
				i++;
			}
		}
		printk("\nCode: ");
		for(i=0;i<20;i++)
			printk("%02x ", ((unsigned char *)regs->eip)[i]);
	}
	printk("\n");
}	

spinlock_t die_lock;

void die(const char * str, struct pt_regs * regs, long err)
{
	console_verbose();
	spin_lock_irq(&die_lock);
	printk("%s: %04lx\n", str, err & 0xffff);
	show_registers(regs);
	spin_unlock_irq(&die_lock);
	do_exit(SIGSEGV);
}

static inline void die_if_kernel(const char * str, struct pt_regs * regs, long err)
{
  	die(str, regs, err);
}

static void die_if_no_fixup(const char * str, struct pt_regs * regs, long err)
{
  	die(str, regs, err);
}

DO_ERROR( 0, SIGFPE,  "divide error", divide_error, current)
DO_ERROR( 3, SIGTRAP, "int3", int3, current)
DO_ERROR( 4, SIGSEGV, "overflow", overflow, current)
DO_ERROR( 5, SIGSEGV, "bounds", bounds, current)
DO_ERROR( 6, SIGILL,  "invalid operand", invalid_op, current)
DO_ERROR( 7, SIGSEGV, "device not available", device_not_available, current)
DO_ERROR( 8, SIGSEGV, "double fault", double_fault, current)
DO_ERROR( 9, SIGFPE,  "coprocessor segment overrun", coprocessor_segment_overrun, current)
DO_ERROR(10, SIGSEGV, "invalid TSS", invalid_TSS, current)
DO_ERROR(11, SIGBUS,  "segment not present", segment_not_present, current)
DO_ERROR(12, SIGBUS,  "stack segment", stack_segment, current)
DO_ERROR(17, SIGSEGV, "alignment check", alignment_check, current)
DO_ERROR(18, SIGSEGV, "reserved", reserved, current)

void do_general_protection(struct pt_regs * regs, long error_code)
{
	current->tss.error_code = error_code;
	current->tss.trap_no = 13;
	force_sig(SIGSEGV, current);
	return;
}

void do_nmi(struct pt_regs * regs, long error_code)
{
}

/*
 * Careful - we must not do a lock-kernel until we have checked that the
 * debug fault happened in user mode. Getting debug exceptions while
 * in the kernel has to be handled without locking, to avoid deadlocks..
 *
 * Being careful here means that we don't have to be as careful in a
 * lot of more complicated places (task switching can be a bit lazy
 * about restoring all the debug state, and ptrace doesn't have to
 * find every occurrence of the TF bit that could be saved away even
 * by user code - and we don't have to be careful about what values
 * can be written to the debug registers because there are no really
 * bad cases).
 */
 void do_debug(struct pt_regs * regs, long error_code)
{
	struct task_struct *tsk = current;

	tsk->tss.trap_no = 1;
	tsk->tss.error_code = error_code;
	force_sig(SIGTRAP, tsk);
	return;
}

/*
 * Note that we play around with the 'TS' bit in an attempt to get
 * the correct behaviour even in the presence of the asynchronous
 * IRQ13 behaviour
 */
void math_error(void)
{
	struct task_struct * task;

	/*
	 * Save the info for the exception handler
	 * (this will also clear the error)
	 */
	task = current;
	task->tss.trap_no = 16;
	task->tss.error_code = 0;
	force_sig(SIGFPE, task);
}

 void do_coprocessor_error(struct pt_regs * regs, long error_code)
{
	math_error();
}

void __init trap_init(void)
{
}


asmlinkage int FASTCALL(do_signal(struct pt_regs *regs, sigset_t *oldset));

void
check_for_exception_within_emulation_library(void)
{
  extern int emulib_start, emulib_end;
  if ((current->tss.shared_data->regs.eip >= (unsigned long)&emulib_start) &&
      (current->tss.shared_data->regs.eip <= (unsigned long)&emulib_end))
    {
      printk("exception within emulation library: eip: %lx\n",
	     current->tss.shared_data->regs.eip);
      herc_printf("exception %d within emulation library: "
		  "eip: %lx, task: %x\n",
		  current->tss.shared_data->exception_number,
		  current->tss.shared_data->regs.eip,
		  current->tss.user_thread_id.id.task);
    }
}

int 
deliver_signal(void)
{
  /* use exception entry points within kernel 
     e.g. do_divide_error( struct ptregs * regs, long error);
     */
  
  current->tss.shared_data->regs.orig_eax = -1;

  check_for_exception_within_emulation_library();

  switch(current->tss.shared_data->exception_number)
    {
    case 0:  
      do_divide_error(&current->tss.shared_data->regs,
		      current->tss.shared_data->error_code);    
      break;
    case 1:
      do_debug(&current->tss.shared_data->regs,
	       current->tss.shared_data->error_code);    
      break;
    case 2:
      do_nmi(&current->tss.shared_data->regs,
	     current->tss.shared_data->error_code);    
      break;
    case 3:
      do_int3(&current->tss.shared_data->regs,
	      current->tss.shared_data->error_code);    
      break;
    case 4:
      do_overflow(&current->tss.shared_data->regs,
		  current->tss.shared_data->error_code);    
      break;  
    case 5:
      do_bounds(&current->tss.shared_data->regs,
		current->tss.shared_data->error_code);    
      break;  
    case 6:
      do_invalid_op(&current->tss.shared_data->regs,
		    current->tss.shared_data->error_code);    
      break;  
    case 7:			/* no */
      do_device_not_available(&current->tss.shared_data->regs,
			      current->tss.shared_data->error_code);    
      break; 
    case 8:			/* no */
      do_double_fault(&current->tss.shared_data->regs,
		      current->tss.shared_data->error_code);    
      break;
    case 9:			/* no */
      do_coprocessor_segment_overrun(&current->tss.shared_data->regs,
				     current->tss.shared_data->error_code);    
      break;  
    case 10:			/* no */
      do_invalid_TSS(&current->tss.shared_data->regs,
		     current->tss.shared_data->error_code);    
      break;
    case 11:
      do_segment_not_present(&current->tss.shared_data->regs,
			     current->tss.shared_data->error_code);    
      break;
    case 12:
      do_stack_segment(&current->tss.shared_data->regs,
		       current->tss.shared_data->error_code);    
      break;
    case 13:
      do_general_protection(&current->tss.shared_data->regs,
			    current->tss.shared_data->error_code);    
      break;
    case 14:
      enter_kdebug("page fault trap???");
      break; 
    case 15:
      do_coprocessor_error(&current->tss.shared_data->regs,
			   current->tss.shared_data->error_code);    
      break;  
    case 16:
      do_reserved(&current->tss.shared_data->regs,
		  current->tss.shared_data->error_code);    
      break;
    case 17:
      do_alignment_check(&current->tss.shared_data->regs,
			 current->tss.shared_data->error_code);    
      break; 
    case 19:
      break; 
    default:
      enter_kdebug("deliver_signal:unknown exception");
      break;
    }

  printk("(%s, %d): deliver signal calls do_signal",
	 current->comm, current->pid);

  if (current->sigpending)
    do_signal(&current->tss.shared_data->regs, NULL);
      
  return 0;
}

extern idt_entry_t kernel_idt[NUM_DEFAULT_INT_HANDLER];

void enter_kernel_debugger(void)
{
  idt_entry_t int3_entry;

  int3_entry = kernel_idt[3];
  kernel_idt[3] = (idt_entry_t){ 0, 0, 0, 0, 0, 0, 0 };
  printk("enter_kernel_debugger\n");
  enter_kdebug("proc_io_ctrl");
  kernel_idt[3] = int3_entry ;
}
