/*
 * Functions for communication with sigma 0
 */

#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>
#include "../include/l4_memory.h"
/* #define DEBUG */

static l4_threadid_t sigma0_id = L4_INVALID_ID;
static int sigma0_initialized = 0;
l4_kernel_info_t *l4_kernel_info = 0;

#ifdef CONFIG_M4_PAGES
int mb4_pages_dont_work = 0;
#endif

void
check_pager(void)
{
  /* get pager id if required */
  if (!sigma0_initialized)
    {
      l4_threadid_t pager = L4_INVALID_ID, preempter = L4_INVALID_ID;
      dword_t eip = -1, esp = -1, eflags;
      l4_thread_ex_regs(l4_myself(), eip, esp, 
		 &preempter, &pager, &eflags, &eip, &esp);
      if (!l4_is_invalid_id(pager))
	{
	  sigma0_id = pager;
	  sigma0_initialized = 1;
	}
      else
	enter_kdebug("check_pager_id: invalid pager");
    }
  
}

int 
call_pager(dword_t dw0, dword_t dw1, 
	   l4_msgdope_t *map_msg, dword_t *reply_dw0, dword_t *reply_dw1, 
	   l4_timeout_t timeout, l4_msgdope_t *result)
{
  check_pager();
  return  l4_i386_ipc_call(sigma0_id, 
			   L4_IPC_SHORT_MSG, dw0, dw1,
			   map_msg, reply_dw0, reply_dw1,
			   timeout, result);
}


int setup_kernel_info_page(void)
{
  l4_msgdope_t result;
  l4_snd_fpage_t fpage;
  int error;

  l4_kernel_info = (l4_kernel_info_t *)0;
  /* request kernel info page page from sigma 0 */
  error = call_pager(1, 1,
		     L4_IPC_MAPMSG(KERNEL_INFO_PAGE_ADDRESS, L4_LOG2_PAGESIZE), 
		     &fpage.snd_base, &fpage.fpage.fpage,  
		     L4_IPC_NEVER, &result);
  if (!error)
    {
      if (l4_ipc_fpage_received(result))
	{
	  l4_kernel_info = (l4_kernel_info_t *)KERNEL_INFO_PAGE_ADDRESS;
	  return 0;
	}
      else
	panic("setup_kernel_info_page:" 
	       " sigma 0 denied mapping of kernel info page\n");
    }
  else
    panic("setup_kernel_info_page:"
	   " ipc error %x while requesting kernel info page\n", error);
  return 1; 
}

int check_error(int error, l4_msgdope_t result, l4_snd_fpage_t fpage, 
		 int address, int size)
{
  if (!error)
    {
      if (l4_ipc_fpage_received(result))
	{
	  if (l4_ipc_is_fpage_writable(fpage.fpage))
	    {
	      return RP_WRITABLE;
	    }
	  else
	    {
	      enter_kdebug("sigma0 mapped page read only");
	      return RP_NOT_WRITABLE;
	    }
	}
      else
	{
	  if ((address < ((unsigned long)&_stext & PAGE_MASK)) 
	      || (address > (unsigned long)&_end))
	    {
	      return RP_NOT_MAPPED;
	    }
	  else
	    {
	      return RP_WRITABLE;
	    }
	}
    }
  else
    {
      kd_display("request_page_from_sigma0: ipc error: ");
      outhex32(error);
      enter_kdebug("error while calling sigma 0");
    }
  return RP_NOT_MAPPED;

} 

int 
request_page_from_sigma0(unsigned long address)
{
  l4_msgdope_t result;
  l4_snd_fpage_t fpage;
  int error;

  error = call_pager(address, 0,
		     L4_IPC_MAPMSG(address, L4_LOG2_PAGESIZE), 
		     &fpage.snd_base, &fpage.fpage.fpage,  
		     L4_IPC_NEVER, &result);
  return check_error(error, result, fpage, address, 0);
}

int 
request_virtual_page_from_sigma0(unsigned long vm_addr, unsigned long page)
{
  l4_msgdope_t result;
  l4_snd_fpage_t fpage;
  int error;
  
  error = call_pager(page, 0,
		     L4_IPC_MAPMSG(vm_addr, L4_LOG2_PAGESIZE), 
		     &fpage.snd_base, &fpage.fpage.fpage,  
		     L4_IPC_NEVER, &result);
  error = check_error(error, result, fpage, page, 0);
  if (error != RP_WRITABLE) {
    herc_printf("sigma0 denied writable mapping, map code: %x\n", error);
    enter_kdebug("rvp");
    return 0;
  } else {
    return PAGE_SIZE;
  }
}

int 
request_dev_page_from_sigma0(dword_t fault_address, dword_t pte_val)
{
  l4_msgdope_t result;
  dword_t send_base, fpage;
  int error;

  fault_address &= L4_DEV_PAGE_MASK;

  /* request area from sigma 0 */
  error = call_pager(L4_DEV_PAGE(pte_val & PAGE_MASK), 0,
		     L4_IPC_MAPMSG(fault_address, L4_LOG2_DEV_PAGE), 
		     &send_base, &fpage,  
		     L4_IPC_NEVER, &result);
  if (error)
    {
      enter_kdebug("error in request_mem_area_from_sigma0");
      return 0;
    }
  else
    return L4_DEV_PAGE_SIZE;
}
  
void
request_zero_page_from_sigma0(void)
{
  dword_t fpage, snd_base, error;
  l4_msgdope_t result;
  error = call_pager(L4_WRITE_ACCESS, 0,
		     L4_IPC_MAPMSG(0, PAGE_SHIFT),
		     &fpage, &snd_base, 
		     L4_IPC_NEVER, &result);
   if (error)
    enter_kdebug("request_zero_page: ipc error");
  if (!l4_ipc_fpage_received(result))
    enter_kdebug("request_zero_page: no mapping");
}
