#include <linux/vmalloc.h>
#include <linux/mm.h>
#include <asm/segment.h>
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>

#include "../include/l4_memory.h"
#include "../include/l4_thread.h"
#include "../include/config.h"
#include "../include/debug.h"

/* #define DEBUG */
#define ROOT_NO_REPLY 0x29051970

static int root_pager_stack[256];

l4_threadid_t root_pager_id, root_preempter_id;
extern volatile int null_pointer_check;
static int null_pointer_check_done = 0;

void 
create_kernel_pager(void)
{
  create_thread(LTHREAD_NO_ROOT_PAGER, root_pager, &root_pager_stack[255]);
}
void change_pager(int thread_no)
{
  l4_threadid_t preempter, pager;
  dword_t dummy;

  /* get thread parameters */
  l4_threadid_t id = l4_myself();
  id.id.lthread = thread_no;

  preempter = L4_INVALID_ID;
  pager = L4_INVALID_ID;

  l4_thread_ex_regs(id, 
		    (dword_t) -1, (dword_t) -1,
		    &preempter,
		    &pager,
		    &dummy,
		    &dummy,
		    &dummy);

  
  /* exchange pager */
  preempter = L4_INVALID_ID;
  pager = id;
  pager.id.lthread = LTHREAD_NO_ROOT_PAGER;

  l4_thread_ex_regs(id,
		    (dword_t) -1, (dword_t) -1,
		    &preempter, /* preempter */
		    &pager,	/* pager */
		    &dummy,	/* old_flags */
		    &dummy,	/* old_eip */
		    &dummy);	/* old_esp */
}


int
page_fault_request(l4_threadid_t src, dword_t fault_address, dword_t faulting_eip)
{
  pte_t pte, *ptep;

 
  ptep = lookup_pte(swapper_pg_dir, fault_address);
#if 0
  herc_printf("page fault from at %lx, eip: %lx, ptep: %p, pte: %lx\n",
	      src.id.task, src.id.lthread, 
	      (unsigned long)fault_address, (unsigned long)faulting_eip,
	      ptep, ptep ? pte_val(*ptep) : 0);
#endif  
  if (! ptep)
    {
      /* Ohh..  the kernel touched a non-present page! */
      herc_printf("Root: pte not present for addr %lx (virt: %lx) "
		  "and pdir %p at eip: %lx\n",
		  (unsigned long)fault_address, 
		  (unsigned long)VMALLOC_VMADDR(fault_address),
		  swapper_pg_dir, (unsigned long)faulting_eip);
      enter_kdebug("root_pager: pte not present");

      return -1;		/* wait for next message */
    }

  pte = *ptep;

  /* vremapped area? */
  if (pte_val(pte) > 0x80000000U)
    {
      /* OK, this is a vremap()'d area. */
      fault_address &= ~(0x400000 - 1);
      if (request_dev_page_from_sigma0(fault_address, pte_val(*ptep)))
	{
	  return 0;
	}
      else
	{
	  enter_kdebug("request region in kernel failed");
	  return -1;
	}
    }
  
  /* vmalloced area? */
  if (fault_address >= VMALLOC_START) {
    /* ok this is vmalloc'ed memory, simply request ist from our underlying pager */
    if (request_virtual_page_from_sigma0(fault_address, pte_val(*ptep))) {
      return 0;
    }
    else
      {
	enter_kdebug("request virtual page in kernel failed");
	return -1;
      }
  }
  if ((fault_address & _PAGE_RW) && !pte_write(pte)) 
    {
      /* access check failed */ 
      herc_printf("Root, Wrong Access in %x.%x, addr: %lx (%s), "
		  "eip; %lx\n",
		  src.id.task, src.id.lthread,
		  (unsigned long)fault_address, 
		  ((fault_address & _PAGE_RW) ? "write" : "read"),
		  (unsigned long)faulting_eip);
      enter_kdebug("wrong access");
      return -1;
    }

  /* check for null pointer */
  if ((unsigned long)fault_address < (unsigned long)0x1000L)
    {
      if (null_pointer_check && !null_pointer_check_done)
	{
 	  enter_kdebug("touch zero page"); 
	  *(int *)0 = 0;
	  null_pointer_check = 0;
	  null_pointer_check_done = 1;
	}
      else
	{
	  herc_printf("Root: null pointer deref in %x.%x "
		      "at addr: %lx and eip: %lx (pte: %lx)\n",
		      src.id.task, src.id.lthread, 
		      (unsigned long)fault_address, 
		      (unsigned long)faulting_eip,  pte_val(*ptep));
	  return -1;
	}
    }
  else
    if ((void *)fault_address < high_memory )
      {
	herc_printf("\nRoot: pfa (%lx) < rp_mem_end (%p)"
		    " in %x.%x at eip %lx\n",
		    (unsigned long)fault_address, 
		    high_memory, 
		    src.id.task, src.id.lthread, 
		    (unsigned long)faulting_eip);
	enter_kdebug("root: ram-pf; why?");
      }
  return 0;
}

#define pager_wait l4_i386_ipc_wait
#define pager_reply_and_wait l4_i386_ipc_reply_and_wait

int send_sig_segv(void) {
  enter_kdebug("pagefault within kernel");
  return 0;
}
void 
root_pager(void)
{
  dword_t error, fault_address, faulting_eip;
  l4_threadid_t src;
  l4_msgdope_t result;
  
  /* Enter endless loop and wait for requests or page faults. The only
     page faults which can happen, are page faults within vmalloc'ed
     areas and they should only be raised by the service thread. Check
     this while handling the page fault. */

  for(;;)
    {
      error = pager_wait(&src, L4_IPC_SHORT_MSG, 
			       &fault_address, &faulting_eip,
			       L4_IPC_NEVER, &result);
      while(!error) 
	{
	  KO('r');
#ifdef DEBUG
	  herc_printf("root: rcvd %lx, %lx\n", fault_address, faulting_eip);
	  if (src.id.lthread != LTHREAD_NO_LINUX_SERVER)
	    herc_printf("root: wrong thread (%x.%x)\n", 
			src.id.taskno, src.id.lthread);
#endif
	  error = page_fault_request(src, fault_address, faulting_eip);
	  if (error)
	    {
	      send_sig_segv();
	      error = ROOT_NO_REPLY;
	    }

	  if (!error)
	    {
	      KO('R');
	      error = pager_reply_and_wait(src,
					   L4_IPC_SHORT_MSG,
					   fault_address, faulting_eip,
					   &src, 
					   L4_IPC_SHORT_MSG, 
					   &fault_address, &faulting_eip,
					   L4_IPC_TIMEOUT(0,1,0,0,0,0), 
					   &result);
	    }
	}
      if (error && (error != ROOT_NO_REPLY))
	enter_kdebug("root: error!?");
    }
}

int
l4_handle_special_region(pte_t pte)
{
  static struct vm_struct * special_map_area = 0;
  static unsigned long special_map_addr = 0;

  if (pte_page(pte) > 0x80000000U)
    {
      printk("pte_page(pte) > 0x80000000U, val: %lx\n", pte_page(pte));

      if (! special_map_addr)
	{

	  /* get an 4MB-aligned empty VM area */
	  special_map_area = get_vm_area(2*L4_DEV_PAGE_SIZE);
	  if (!special_map_area)
	    {
	      /* Oh lord...  couldn't get a temporary mapping area! */
	      enter_kdebug("no map_area");
		  
	      /* XXX */
	      return 0;
	    }
	  special_map_addr = L4_DEV_PAGE_ALIGN((unsigned long)special_map_area->addr);
	}

      /* OK, we have a temporary region.  now request a mapping from
	 sigma0 */
      if (request_dev_page_from_sigma0(special_map_addr, pte_page(pte)))
	{
	  /* OK, temporarily map this area in the kernel, then
	     grant it to the user */
	      
#if 0
	  enter_kdebug("region requested");
#endif
	  return l4_fpage(special_map_addr,
			  L4_LOG2_DEV_PAGE, 
			  pte_write(pte) ? L4_FPAGE_RW : L4_FPAGE_RO, 
			  L4_FPAGE_GRANT).fpage;
	}
      else
	{
	  enter_kdebug("request region failed");
	  /* XXX */
	  return 0;
	}
    }
  else
    {
      enter_kdebug("pte_page(pte) <= 0x80000000U");
      return 0;
    }
}










