#include <asm/page.h>

#include "../include/l4_memory.h"
#include <l4/syscalls.h>
#define MIN(a,b) (((a)<(b))?(a):(b))

/* #define DEBUG_MEMCPY_TOFS */
/* #define DEBUG_MEMCPY_FROMFS */
#define LOG_EFAULT
#define INFORM_USERTASK

extern inline int
__copy_to_user_page(void * to, const void * from, unsigned long n)
{
  unsigned long page, offset;

#ifdef DEBUG_MEMCPY_TOFS
  printk("__copy_to_user_page to: %p, from: %p, len: %lx\n", to, from, n);
#endif 
  page = parse_ptabs_write((unsigned long)to, &offset);
  if (page != -EFAULT)
    {
#ifdef DEBUG_MEMCPY_TOFS
      printk("    __copy_to_user_page writing to: %lx\n", (page + offset));
#endif 
      memcpy((void *)(page + offset), from, n);
      return 0;
    }
#ifdef LOG_EFAULT
  herc_printf("__copy_to_user_page: returning -EFAULT");
  enter_kdebug("__copy_to_user_page");
#endif
  return -EFAULT;
}

int copy_to_user(void *to, const void *from, unsigned long n)
{
  unsigned long copy_size = (unsigned long)to & ~PAGE_MASK;

#ifdef INFORM_USERTASK
  asm volatile (
    "pushl %%ebp\n\t"
    "xorl  %%ebp,%%ebp\n\t"
    "movl  4(%%esi),%%edi\n\t"
    "movl  (%%esi),%%esi\n\t"
    ToLId
    "movl  $2,%%edi\n\t"
    "int   $0x30\n\t"
    "popl  %%ebp"
    :
    : "a"(0),
      "c"(0),
      "d"(to),
      "b"(n),
      "S"(&current->tss.user_thread_id.lh.low)
    : "eax", "ebx", "ecx", "edx", "esi", "edi", "cc"
  );
#endif

#ifdef DEBUG_MEMCPY_TOFS
  printk("copy_to_user called from: %lx to: %p, from: %p, len: %lx\n", 
	 *((unsigned long *)&to - 1), to, from, n);
#endif 
  if (copy_size)
    {
      copy_size = MIN(PAGE_SIZE - copy_size, n);
      if (__copy_to_user_page(to, from, copy_size) == -EFAULT)
	return n;
      n -= copy_size;
    }
  while (n)
    {
      from +=copy_size;
      to += copy_size;
      copy_size = MIN(PAGE_SIZE, n);
      if (__copy_to_user_page(to, from, copy_size) == -EFAULT)
	return n;
      n -= copy_size;
    }
  return 0;
}

extern inline int
__copy_from_user_page(void * to, const void * from, unsigned long n)
{
  unsigned long page, offset;

#ifdef DEBUG_MEMCPY_FROMFS
  printk("__copy_from_user_page to: %p, from: %p, len: %lx\n", to, from, n);
#endif 
  page = parse_ptabs_read((unsigned long)from, &offset);
  if (page != -EFAULT)
    {
#ifdef DEBUG_MEMCPY_FROMFS
      printk("    __copy_from_user_page reading from: %lx\n", (page + offset));
#endif 
      memcpy(to, (void *)(page + offset), n);
      return 0;
    }
#ifdef LOG_EFAULT
  herc_printf("__copy_from_user_page: returning -EFAULT");
  enter_kdebug("__copy_from_user_page");
#endif
  return -EFAULT;
}

int copy_from_user(void *to, const void *from, unsigned long n)
{
  unsigned long copy_size = (unsigned long)from & ~PAGE_MASK;

#ifdef INFORM_USERTASK
  asm volatile (
    "pushl %%ebp\n\t"
    "xorl  %%ebp,%%ebp\n\t"
    "movl  4(%%esi),%%edi\n\t"
    "movl  (%%esi),%%esi\n\t"
    ToLId
    "movl  $1,%%edi\n\t"
    "int   $0x30\n\t"
    "popl  %%ebp"
    :
    : "a"(0),
      "c"(0),
      "d"(from),
      "b"(n),
      "S"(&current->tss.user_thread_id.lh.low)
    : "eax", "ebx", "ecx", "edx", "esi", "edi", "cc"
  );
#endif

#ifdef DEBUG_MEMCPY_FROMFS
  printk("copy_from_user called from: %lx to: %p, from: %p, len: %lx\n", 
	 *((unsigned long *)&to - 1), to, from, n);
#endif 
  if (copy_size)
    {
      copy_size = MIN(PAGE_SIZE - copy_size, n);
      if (__copy_from_user_page(to, from, copy_size) == -EFAULT) {
	memset(to, 0, n);
	return n;
      }
      n -= copy_size;
    }
  while (n)
    {
      from +=copy_size;
      to += copy_size;
      copy_size = MIN(PAGE_SIZE, n);
      if (__copy_from_user_page(to, from, copy_size) == -EFAULT) {
	memset(to, 0, n);
	return n;
      }
      n -= copy_size;
    }
  return 0;
}

extern inline int
__clear_user_page(void * address, unsigned long n)
{
  unsigned long page, offset;

#ifdef DEBUG_MEMCPY_TOFS
  printk("__clear_user_page: %p, len: %lx\n", address, n);
#endif 
  page = parse_ptabs_write((unsigned long)address, &offset);
  if (page != -EFAULT)
    {
#ifdef DEBUG_MEMCPY_TOFS
      printk("    __clear_user_page writing to: %lx\n", (page + offset));
#endif 
      memset((void *)(page + offset), 0, n);
      return 0;
    }
#ifdef LOG_EFAULT
  herc_printf("__clear_user_page: returning -EFAULT");
  enter_kdebug("__clear_user_page");
#endif
  return -EFAULT;
}

unsigned long clear_user(void *address, unsigned long n)
{
  unsigned long clear_size = (unsigned long)address & ~PAGE_MASK;

#ifdef INFORM_USERTASK
  asm volatile (
    "pushl %%ebp\n\t"
    "xorl  %%ebp,%%ebp\n\t"
    "movl  4(%%esi),%%edi\n\t"
    "movl  (%%esi),%%esi\n\t"
    ToLId
    "movl  $3,%%edi\n\t"
    "int   $0x30\n\t"
    "popl  %%ebp"
    :
    : "a"(0),
      "c"(0),
      "d"(address),
      "b"(n),
      "S"(&current->tss.user_thread_id.lh.low)
    : "eax", "ebx", "ecx", "edx", "esi", "edi", "cc"
  );
#endif

#ifdef DEBUG_MEMCPY_TOFS
  printk("clear_user called from: %lx to: %p, len: %lx\n", 
	 *((unsigned long *)&address - 1), address, n);
#endif 
  if (clear_size)
    {
      clear_size = MIN(PAGE_SIZE - clear_size, n);
      if (__clear_user_page(address, clear_size) == -EFAULT)
	return n;
      n -= clear_size;
    }
  while (n)
    {
      address += clear_size;
      clear_size = MIN(PAGE_SIZE, n);
      if (__clear_user_page(address, clear_size) == -EFAULT)
	return n;
      n -= clear_size;
    }
  return 0;
}


/*
 * Copy a null terminated string from userspace.
 */

#define __do_strncpy_from_user_page(dst,src,count)			   \
do {									   \
	int __d0, __d1, __d2;						   \
	__asm__ __volatile__(						   \
		"	testl %0,%0\n"					   \
		"	jz 1f\n"					   \
		"0:	lodsb\n"					   \
		"	stosb\n"					   \
		"	testb %%al,%%al\n"				   \
		"	jz 1f\n"					   \
		"	decl %0\n"					   \
		"	jnz 0b\n"					   \
		"1:\n"							   \
		: "=c"(count), "=&a" (__d0), "=&S" (__d1),		   \
		  "=&D" (__d2)						   \
		: "0"(count), "2"(src), "3"(dst) 			   \
		: "memory");						   \
} while (0)

extern inline int
__strncpy_from_user_page(char * to, const char * from, unsigned long n)
{
  unsigned long page, offset;

#ifdef DEBUG_MEMCPY_FROMFS
  printk("__strncpy_from_user_page to: %p, from: %p, len: %lx\n", to, from, n);
#endif 
  page = parse_ptabs_read((unsigned long)from, &offset);
  if (page != -EFAULT)
    {
#ifdef DEBUG_MEMCPY_FROMFS
      printk("    __strncpy_from_user reading from: %lx\n", (page + offset));
#endif 
      /* after finishing the copy operation count is either 
       * - zero: max number of bytes copied or
       * - non zero: end of sting reached, n containing the number of 
       *             remaining bytes
       */
      __do_strncpy_from_user_page(to, (char *)(page + offset), n);
      return n;
    }
#ifdef LOG_EFAULT
  herc_printf("__strncpy_from_user_page: returning -EFAULT");
  enter_kdebug("__strncpy_from_user_page");
#endif
   return -EFAULT;
}

/* strncpy returns the number of bytes copied. We calculate the number
 * simply by substracting the number of bytes remaining from the
 * maximal length. The number of bytes remaining is (n + res) with n
 * beeing the number of bytes to copy from the next pages and res the
 * number of remaining bytes after reaching the '\0' */

long strncpy_from_user(char *dst, const char *src, long count)
{
  unsigned long copy_size = (unsigned long)src & ~PAGE_MASK;
  int res, n=count;

#ifdef INFORM_USERTASK
  asm volatile (
    "pushl %%ebp\n\t"
    "xorl  %%ebp,%%ebp\n\t"
    "movl  4(%%esi),%%edi\n\t"
    "movl  (%%esi),%%esi\n\t"
    ToLId
    "movl  $4,%%edi\n\t"
    "int   $0x30\n\t"
    "popl  %%ebp"
    :
    : "a"(0),
      "c"(0),
      "d"(src),
      "b"(count),
      "S"(&current->tss.user_thread_id.lh.low)
    : "eax", "ebx", "ecx", "edx", "esi", "edi", "cc"
  );
#endif

#ifdef DEBUG_MEMCPY_FROMFS
  printk("strncpy_from_user called from: %lx to: %p, from: %p, len: %x\n", 
	 *((unsigned long *)&dst - 1), dst, src, n);
#endif 
  if (copy_size)
    {
      copy_size = MIN(PAGE_SIZE - copy_size, n);
      res = __strncpy_from_user_page(dst, src, copy_size);
      n -= copy_size;
      if (res == -EFAULT) {
	return -EFAULT;
      }
      else if (res) 
	return count - (n + res) ;
    }
  while (n)
    {
      src +=copy_size;
      dst += copy_size;
      copy_size = MIN(PAGE_SIZE, n);
      n -= copy_size;
      res = __strncpy_from_user_page(dst, src, copy_size);
      if (res == -EFAULT) {
	return -EFAULT;
      }
      else if (res) 
	return count - (n + res) ;
    }
  return count;
}

extern inline int
__strnlen_from_user_page(const char *from, unsigned long n, unsigned long *len)
{
  unsigned long page, offset;
  int res, end;

#ifdef DEBUG_MEMCPY_FROMFS
  printk("__strnlen_from_user_page  from: %p, len: %lx\n", from, n);
#endif 
  page = parse_ptabs_read((unsigned long)from, &offset);
  if (page != -EFAULT)
    {
#ifdef DEBUG_MEMCPY_FROMFS
      printk("    __strlen_from_user_page reading from: %lx\n", (page + offset));
#endif 
      __asm__ __volatile__(
			   "0:	repne; scasb\n"
			   "	sete %b1\n"
			   :"=c" (res),  "=a" (end)
			   :"0" (n), "1" (0), "D" ((page + offset))
			   );
      /* after finishing the search operation end is either
       * - zero: max number of bytes searched
       * - non zero: end of string reached, res containing the number of 
       *             remaining bytes
       */
      *len += n - res;
      return end;
    }
#ifdef LOG_EFAULT
  herc_printf("__strnlen_from_user_page: returning -EFAULT");
  enter_kdebug("__strnlen_from_user_page");
#endif
  return -EFAULT;
}

/* strncpy returns the number of bytes copied. We calculate the number
 * simply by substracting the number of bytes remaining from the
 * maximal length. The number of bytes remaining is (n + res) with n
 * beeing the number of bytes to copy from the next pages and res the
 * number of remaining bytes after reaching the '\0' */

long strlen_user(const char *src)
{
  unsigned long search_size = PAGE_SIZE - ((unsigned long)src & ~PAGE_MASK);
  int res;
  unsigned long len=0;

#ifdef INFORM_USERTASK
  asm volatile (
    "pushl %%ebp\n\t"
    "xorl  %%ebp,%%ebp\n\t"
    "movl  4(%%esi),%%edi\n\t"
    "movl  (%%esi),%%esi\n\t"
    ToLId
    "movl  $5,%%edi\n\t"
    "int   $0x30\n\t"
    "popl  %%ebp"
    :
    : "a"(0),
      "c"(0),
      "d"(src),
      "b"(0),
      "S"(&current->tss.user_thread_id.lh.low)
    : "eax", "ebx", "ecx", "edx", "esi", "edi", "cc"
  );
#endif

#ifdef DEBUG_MEMCPY_FROMFS
  printk("strlen_user called from: %lx, from: %p\n", 
	 *((unsigned long *)&src - 1), src);
#endif 

  if (!search_size)
    search_size = PAGE_SIZE;

  while (1)
    {
      res = __strnlen_from_user_page(src, search_size, &len);
      if (res == -EFAULT) {
	return -EFAULT;
      }
      else if (res) 
	return len;

      src +=search_size;
      search_size = PAGE_SIZE;
    }
}


