#include <linux/config.h>

#include <linux/mm.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/pgtable.h>

#include "../include/l4_memory.h"
#include "../include/task.h"
#include "../include/config.h"
/* #define DEBUG */
#include "../include/debug.h"

#define SANITY 
/* #define DEBUG */
/* #define PARANOIA */

void flush_page(unsigned long address, int size, unsigned long options)
{
  /* some checks:
   * options & ALL_SPACES: 	address >= high_memory
   * otherwise:			address < high_memory 
   * address > 0x80000000UL: 	no flush operation
   */
  if (options & L4_FP_ALL_SPACES) {
    /* unmap page in all spaces, only allowed for vm pages */
    if (address < (unsigned long)high_memory) {
      herc_printf("trying to flush physical page (%lx) from linux server",
		  address);
      enter_kdebug("phys + all_spaces");
    }
  }
  else if (address > 0x80000000UL) {
    /* don't flush these, they are already flushed since we granted
       device memory to our user */
    return;
  }
  else {  
    if ((address >= (unsigned long)high_memory) && (address < 0x80000000UL)) {
      herc_printf("flushing non physical page (0x%x)\n", address);
      enter_kdebug("flush_page: non physical page");
    }
  }
  
  /* do the real flush */
  l4_fpage_unmap(l4_fpage(address, size, 0, 0), options);
}

void set_pte(pte_t *pteptr, pte_t pteval)
{
  /*
   * Check if any invalidation is necessary
   *
   * Invalidation (flush) necessary if:
   *   old page was present
   *       new page has another physical address OR
   *       new page is not present OR
   *       new page has another protection OR
   *       new page has other access attributes
   */
#ifdef DEBUG
   printk("set_pte called from %lx, pte to be set: %x to entry: %lx\n", 
	  *(((unsigned long *)&pteptr)-1),
	  (unsigned)pteptr, pte_val(pteval));
#endif /* DEBUG */
  if (pte_present(*pteptr))
    {
      if (!pte_present(pteval) || (pte_page(*pteptr) != pte_page(pteval)))
	{
	  /* (old was present && new not) || physical page frame changed */
	  /* flush */
	  flush_page(pte_page(*pteptr), PAGE_SHIFT, 
		     L4_FP_OTHER_SPACES | L4_FP_FLUSH_PAGE);
	}
      else
	{
	  /* both old and new are present and in same physical frame, but
	     access mask or access attributes have changed */
	  if ( (pte_write(*pteptr) && !pte_write(pteval)) ||
	       (pte_dirty(*pteptr) && !pte_dirty(pteval)) )
	    {
	      /* Protection changed from r/w to ro or page now clean */
	      /* XXX do it locally if possible */
	      flush_page(pte_page(*pteptr), PAGE_SHIFT, 
			 L4_FP_OTHER_SPACES | L4_FP_REMAP_PAGE);
	    }
	  else if (pte_young(*pteptr) && !pte_young(pteval))
	    {
	      /* flush page to be able to detect access to this page */
	      flush_page(pte_page(*pteptr), PAGE_SHIFT, 
			 L4_FP_OTHER_SPACES | L4_FP_FLUSH_PAGE);

	    }
	}
    }
  /*
   * Set new value
   */
  *pteptr = pteval;
}
void pte_clear(pte_t *ptep)
{ 
  /*
   * Check if any invalidation is necessary
   * If possible, invalidate locally.
   */
#ifdef DEBUG
  printk("pte_clear called, pte to be cleared: %x\n", (unsigned)ptep);
#endif
  if (pte_present(*ptep))
    {
      /* 
       * Invalidate page
       */
      flush_page(pte_page(*ptep), PAGE_SHIFT, 
		 L4_FP_OTHER_SPACES | L4_FP_FLUSH_PAGE);
    }
  pte_val(*ptep) = 0; 
}

/* vmalloc/vmfree helper functions */
void l4_vmfree_area_pages(unsigned long address, unsigned long size);
void l4_vmalloc_area_pages(unsigned long address, unsigned long size);

void 
l4_vmfree_area_pages(unsigned long address, unsigned long size)
{
  /* unmap memory that the kernel is just freeing */
  //enter_kdebug("l4_vmfree_area_pages");
  for(; size > 0; size-=PAGE_SIZE, address+=PAGE_SIZE) {
    /* check whether we are really flushing a vm page */
    if (address < (unsigned long)high_memory) {
      herc_printf("flushing wrong page, addr: %lx\n", address);
      enter_kdebug("lm_vmfree");
    }
    flush_page(address, PAGE_SHIFT, 
	       L4_FP_ALL_SPACES | L4_FP_FLUSH_PAGE);
  }
}

void 
l4_vmalloc_area_pages(unsigned long address, unsigned long size)
{
  /* touch all memory to ensure it is mapped before returning from vmalloc */
  herc_printf("l4_vmalloc_area_pages: touching %lx upto %lx\n",
	      address, address+size);
  for(; size > 0; size-=PAGE_SIZE, address+=PAGE_SIZE)
    asm("orl $0, (%0)" : /* no output */ : "r" (address));
}


pgd_t *get_pgd_fast(void)
{
	unsigned long *ret;

	if((ret = pgd_quicklist) != NULL) {
		pgd_quicklist = (unsigned long *)(*ret);
		ret[0] = ret[1];
		pgtable_cache_size--;
	} else
		ret = (unsigned long *)get_pgd_slow();
	return (pgd_t *)ret;
}

void free_pgd_fast(pgd_t *pgd)
{
	*(unsigned long *)pgd = (unsigned long) pgd_quicklist;
	pgd_quicklist = (unsigned long *) pgd;
	pgtable_cache_size++;
}
pte_t *get_pte_fast(void)
{
	unsigned long *ret;

	if((ret = (unsigned long *)pte_quicklist) != NULL) {
		pte_quicklist = (unsigned long *)(*ret);
		ret[0] = ret[1];
		pgtable_cache_size--;
	}
#ifdef DEBUG
	if (ret) {
	  pte_t *tmp = (pte_t *)ret;
	  int i, kd=0;
	  printk("get_pte_fast: pte: %p\n", ret);
	  for (i=0; i< PTRS_PER_PTE; i++, tmp++)
	    if (pte_present(*tmp)) {
	      printk("get_pte_fast: pte_present: %p (%lx)\n", 
		     tmp, *(unsigned long*)tmp);
	      kd = 1;
	    }
	  
	  if (ret == (unsigned long *)0x1884000) {
	    printk("get_pte_fast: 0x1884\n");
	    kd = 1;
	  }

	  if (kd)
	    enter_kdebug("get_pte_fast");
	}
#endif
	return (pte_t *)ret;
}

void free_pte_fast(pte_t *pte)
{
#ifdef DEBUG
  	printk("free_pte_fast: %p\n", pte);
	if (pte == (pte_t  *)0x1884000) {
	  printk("free_pte_fast: 0x1884, called from %lx\n",
		 *( (unsigned long *)&pte - 1 ));
	  enter_kdebug("free_pte_fast");
	}
#endif
	*(unsigned long *)pte = (unsigned long) pte_quicklist;
	pte_quicklist = (unsigned long *) pte;
	pgtable_cache_size++;
}
