/*
 *  linux/arch/i386/mm/fault.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 */

#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>

#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/pgtable.h>
#include <asm/hardirq.h>


#include "../include/shared_data.h"
#include "../include/l4_memory.h"
#include "../include/fake_interrupt.h"
#include "../include/debug.h"

extern void die(const char *,struct pt_regs *,long);

/*
 * This routine handles page faults.  It determines the address,
 * and the problem, and then passes it off to one of the appropriate
 * routines.
 *
 * error_code:
 *	bit 0 == 0 means no page found, 1 means protection fault
 *	bit 1 == 0 means read, 1 means write
 *	bit 2 == 0 means kernel, 1 means user-mode
 *
 * adjusted for l4/linux. do_pagefault changed to l4_do_pagefault
 * which simply:
 *	- calls handle_m_fault if there is a valid vma,
 *	- forces a signal if the page fault was raised by a user,
 *	  calls generate_fake_interrupt to force the user into the
 *	  kernel (it still is in an ipc waiting for the reply to its
 *	  page fault) and returns (-1) to signal the error condition.
 *	- returns (-1) if the pagefault happens within the kernel
 *	  context and leaves the rest to the calling function (should
 *	  be a uacess function which in turn will return an EFAULT to
 *	  its calling function)
 *
 */
asmlinkage int l4_do_page_fault(unsigned long address, unsigned long error_code)
{
	struct task_struct *tsk;
	struct mm_struct *mm;
	struct vm_area_struct * vma;
	unsigned long page;
	unsigned long fixup;
	int write;

	tsk = current;
	mm = tsk->mm;

	/*
	 * If we're in an interrupt or have no user
	 * context, we must not take the fault..
	 */
	if (in_interrupt() || mm == &init_mm)
		goto no_context;

	down(&mm->mmap_sem);
#ifdef DEBUG_HANDLE_PF
	herc_printf("l4_do_page_fault(%lx, %lx)\n", address, error_code);
#endif
	vma = find_vma(mm, address);
	if (!vma)
		goto bad_area;
	if (vma->vm_start <= address)
		goto good_area;
	if (!(vma->vm_flags & VM_GROWSDOWN))
		goto bad_area;
#if 0
	/* We don't have access to the stack pointer so we can't check
         * for this specific case 
	 */
	if (error_code & 4) {
		/*
		 * accessing the stack below %esp is always a bug.
		 * The "+ 32" is there due to some instructions (like
		 * pusha) doing post-decrement on the stack and that
		 * doesn't show up until later..
		 */
		if (address + 32 < regs->esp)
			goto bad_area;
	}
#endif
	if (expand_stack(vma, address))
		goto bad_area;
/*
 * Ok, we have a good vm_area for this memory access, so
 * we can handle it..
 */
good_area:
#ifdef DEBUG_HANDLE_PF
	herc_printf("   vma found");
#endif
	write = 0;
	switch (error_code & 3) {
		default:	/* 3: write, present */
#ifdef TEST_VERIFY_AREA
			if (regs->cs == KERNEL_CS)
				printk("WP fault at %08lx\n", regs->eip);
#endif
			/* fall through */
		case 2:		/* write, not present */
			if (!(vma->vm_flags & VM_WRITE))
				goto bad_area;
			write++;
			break;
		case 1:		/* read, present */
			goto bad_area;
		case 0:		/* read, not present */
			if (!(vma->vm_flags & (VM_READ | VM_EXEC)))
				goto bad_area;
	}

	/*
	 * If for any reason at all we couldn't handle the fault,
	 * make sure we exit gracefully rather than endlessly redo
	 * the fault.
	 */
#ifdef DEBUG_HANDLE_PF
	herc_printf("calling handle_mm_fault, ");
#endif
	if (!handle_mm_fault(tsk, vma, address, write))
		goto do_sigbus;

	up(&mm->mmap_sem);
#ifdef DEBUG_HANDLE_PF
	herc_printf("returning 0\n");
#endif
	return 0;

/*
 * Something tried to access memory that isn't in our memory map..
 * Fix it, but check if it's kernel or user first..
 */
bad_area:
	up(&mm->mmap_sem);
	herc_printf("oos, bad area (%lx, %lx)\n", address, error_code);
	enter_kdebug("bad area");

	/* User mode accesses just cause a SIGSEGV */
	if (error_code & 4) {
		tsk->tss.cr2 = address;
		tsk->tss.error_code = error_code;
		tsk->tss.trap_no = 14;
		force_sig(SIGSEGV, tsk);
		generate_fake_interrupt(current);
		return -1;
	}

no_context:
	/* Are we prepared to handle this kernel fault?  Who is
           calling do_pagefault anyway? The only functions should be
           the uaccess functions. So we simply return an error value
           and leave teh rest to our uaccess implementation. */
	return -1;

/*
 * We ran out of memory, or some other thing happened to us that made
 * us unable to handle the page fault gracefully.
 */
do_sigbus:
	up(&mm->mmap_sem);

	/*
	 * Send a sigbus, regardless of whether we were in kernel
	 * or user mode.
	 */
	tsk->tss.cr2 = address;
	tsk->tss.error_code = error_code;
	tsk->tss.trap_no = 14;
	force_sig(SIGBUS, tsk);
	generate_fake_interrupt(current);

	/* Kernel mode? Handle exceptions or die */
	if (!(error_code & 4))
		goto no_context;
	return -1;
}

void seg_fault_above_task_size(unsigned long address)
{
  herc_printf("sfats %lx ", address);
  current->tss.cr2 = address;
  current->tss.error_code = PF_EUSER | PF_ENOTPRESENT | 
    ((address & PF_EWRITE) ? PF_EWRITE : PF_EREAD) ;
  current->tss.trap_no = 14;
  force_sig(SIGSEGV, current);
  generate_fake_interrupt(current);
  herc_printf("sfats_end\n");
}
