/*
 * arch/i386/mm/ioremap.c
 *
 * Re-map IO memory to kernel address space so that we can access it.
 * This is needed for high PCI addresses that aren't mapped in the
 * 640k-1MB IO memory area on PC's
 *
 * (C) Copyright 1995 1996 Linus Torvalds
 */

#include <linux/vmalloc.h>
#include <asm/io.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>

#include "../include/l4_memory.h"

#define MAX_IOREMAP_ENTRIES 20
struct ioremap_table {
  unsigned long real_map_addr;
  unsigned long ioremap_addr;
};
static struct ioremap_table io_table[MAX_IOREMAP_ENTRIES];
static int ioremap_table_initialized=0;

static void init_io_remap(void);
static int set_ioremap_entry(unsigned long real_map_addr, 
			     unsigned long ioremap_addr);
static unsigned long lookup_ioremap_entry(unsigned long ioremap_addr);

static void init_io_remap(void)
{
  int i;
  for (i=0; i<MAX_IOREMAP_ENTRIES; i++)
    io_table[i] = (struct ioremap_table){0,0};
  ioremap_table_initialized=1;
}

static int set_ioremap_entry(unsigned long real_map_addr, 
			     unsigned long ioremap_addr)
{
  int i;
  if (!ioremap_table_initialized)
    init_io_remap();

  for (i=0; i<MAX_IOREMAP_ENTRIES; i++)
    if (io_table[i].real_map_addr == 0) {
      io_table[i] = (struct ioremap_table){real_map_addr, ioremap_addr};
      return 0;
    }
  enter_kdebug("no free entry in ioremaptable");
  return 1;
}

static unsigned long lookup_ioremap_entry(unsigned long ioremap_addr)
{
  int i;
  unsigned long tmp;
  
  if (!ioremap_table_initialized)
    enter_kdebug("oops, ioremaptabel not initialized");

  for (i=0; i<MAX_IOREMAP_ENTRIES; i++)
    if (io_table[i].ioremap_addr == ioremap_addr) {
      tmp = io_table[i].real_map_addr;
      io_table[i].real_map_addr = 0;
      return tmp;
    }
  enter_kdebug("no entry found in ioremaptable");
  return 0;
}


static inline void remap_area_pte(pte_t * pte, unsigned long address, unsigned long size,
	unsigned long phys_addr, unsigned long flags)
{
	unsigned long end;

	address &= ~PMD_MASK;
	end = address + size;
	if (end > PMD_SIZE)
		end = PMD_SIZE;
	do {
		if (!pte_none(*pte)) 
		  {
			printk("remap_area_pte: page already exists\n");
		  }
		set_pte(pte, mk_pte_phys(phys_addr, __pgprot(_PAGE_PRESENT | _PAGE_RW | 
					_PAGE_DIRTY | _PAGE_ACCESSED | flags)));
		address += PAGE_SIZE;
		phys_addr += PAGE_SIZE;
		pte++;
	} while (address < end);
}

static inline int remap_area_pmd(pmd_t * pmd, unsigned long address, unsigned long size,
	unsigned long phys_addr, unsigned long flags)
{
	unsigned long end;

	address &= ~PGDIR_MASK;
	end = address + size;
	if (end > PGDIR_SIZE)
		end = PGDIR_SIZE;
	phys_addr -= address;
	do {
		pte_t * pte = pte_alloc_kernel(pmd, address);
		if (!pte)
			return -ENOMEM;
		remap_area_pte(pte, address, end - address, address + phys_addr, flags);
		address = (address + PMD_SIZE) & PMD_MASK;
		pmd++;
	} while (address < end);
	return 0;
}

static int remap_area_pages(unsigned long address, unsigned long phys_addr,
				 unsigned long size, unsigned long flags)
{
	pgd_t * dir;
	unsigned long end = address + size;

	phys_addr -= address;
	dir = pgd_offset(&init_mm, address);
	flush_cache_all();
	while (address < end) {
		pmd_t *pmd = pmd_alloc_kernel(dir, address);
		if (!pmd)
			return -ENOMEM;
		if (remap_area_pmd(pmd, address, end - address,
					 phys_addr + address, flags))
			return -ENOMEM;
		set_pgdir(address, *dir);
		address = (address + PGDIR_SIZE) & PGDIR_MASK;
		dir++;
	}
	flush_tlb_all();
	return 0;
}

/*
 * Generic mapping function (not visible outside):
 */

/*
 * Remap an arbitrary physical address space into the kernel virtual
 * address space. Needed when the kernel wants to access high addresses
 * directly.
 *
 * NOTE! We need to allow non-page-aligned mappings too: we will obviously
 * have to convert them into an offset in a page-aligned mapping, but the
 * caller shouldn't need to know that small detail.
 */
void * __ioremap(unsigned long phys_addr, unsigned long size, unsigned long flags)
{
	void * addr;
	struct vm_struct * area;
	unsigned long offset;
	pte_t *ptep;
	/*
	 * Don't remap the low PCI/ISA area, it's always mapped..
	 */
	if (phys_addr >= 0xA0000 && (phys_addr+size) <= 0x100000)
		return phys_to_virt(phys_addr);

	/*
	 * Don't allow anybody to remap normal RAM that we're using..
	 */
	if (phys_addr < virt_to_phys(high_memory))
		return NULL;

	/*
	 * Mappings have to be page-aligned
	 */
	offset = phys_addr & ~L4_DEV_PAGE_MASK;
	phys_addr &= L4_DEV_PAGE_MASK;
	size = L4_DEV_PAGE_ALIGN(size + offset);

	/*
	 * Don't allow mappings that wrap..
	 */
	if (!size || size > phys_addr + size)
		return NULL;

	/* Ok, go for it.. We need an additional window of
	 * L4_DEV_PAGE_SIZE to be able to align our mapping according
	 * to the limitations of L4/x86 */
	area = get_vm_area(size + L4_DEV_PAGE_SIZE);
	if (!area)
		return NULL;
	addr = (void *)L4_DEV_PAGE_ALIGN((unsigned long)area->addr);
	herc_printf("remapping phys addr %lx to virt addr %lx, size: %lx\n",
		    phys_addr, VMALLOC_VMADDR(addr), size);
/* 	enter_kdebug("wait"); */
	if (remap_area_pages(VMALLOC_VMADDR(addr), phys_addr, size, flags)) {
		vfree(addr);
		return NULL;
	}
/* 	enter_kdebug("finished"); */

	set_ioremap_entry((unsigned long)area->addr, (unsigned long)(offset + (char *)addr));
	
#if 0
	/* try to remember the address of the vm_area we have just allocated */
	remap_area_pages(VMALLOC_VMADDR(addr-1), (unsigned long)area->addr, PAGE_SIZE, 0);
	ptep = lookup_pte(swapper_pg_dir, VMALLOC_VMADDR(addr-1));
	if (ptep = 0)
	  enter_kdebug("Ooops");
	else
	  *ptep = pte_rdprotect(*ptep);
#endif
	return (void *) (offset + (char *)addr);
}

void iounmap(void *addr)
{
	unsigned long real_vm_addr, real_map_addr;  
	if (addr > high_memory) {
	  	real_vm_addr = lookup_ioremap_entry((unsigned long) addr);
		real_map_addr = L4_DEV_PAGE_ALIGN(real_vm_addr);
		herc_printf("iounmapping addr: %p, real_vm_addr: %lx, real_map_addr: %lx", addr, real_vm_addr, real_map_addr);
		enter_kdebug("unmapping");
		flush_page(real_map_addr, L4_LOG2_DEV_PAGE, 
			   L4_FP_ALL_SPACES | L4_FP_FLUSH_PAGE);
		return vfree((void *) (L4_DEV_PAGE_MASK & (unsigned long) addr));}
}
