/* $Id: irq.c,v 1.1.1.1 1999/04/06 19:36:25 yoonho Exp $ */

/* This file implements the following interfaces:

   declared in:		what:

   <asm/system.h>	synchronization primitives for the IRQ_LOCK
   			case: __global_{cli,sti,{save,restore}_flags}
   <asm/irq.h>		enable_irq, disable_irq
   <linux/sched.h>	request_irq, free_irq
   <linux/interrupt.h>	probe_irq_{on,off}
   init/main.c		init_IRQ

   It also defines entry points for hardware-interrupt exceptions.  In
   our case, these are L4 threads (one per irq) which sleep waiting
   for interrupts and then call the appropriate handler.

   Top-half handlers are executed directly within the interrupt
   thread.  Bottom halves are always executed in a separate interrupt
   thread.

   If __SMP__ is not defined, the usual semantics (top halves cannot
   be preempted by bottom halves and normal kernel code; bottom halves
   cannot be preempted by normal kernel code) are ensured using thread
   priorities.

   If, on the other hand, __SMP__ is defined, the interrupt threads
   and the bottom half thread act as if they were running on a private
   (virtual) CPU dedicated to just interrupt handling. */

#include <linux/sched.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/ioport.h>

#include <asm/processor.h>
#include <asm/atomic.h>
#include <asm/io.h>

#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>

#include "../include/config.h"
#include "../include/l4_memory.h"
#include "../include/rmgr.h"
#include "../include/sched.h"
#include "../include/ids.h"

#include <asm/irq.h>
#include <asm/system.h>
#include <linux/sched.h>
#include <linux/interrupt.h>


/* configuration section */

#ifdef __i386__
#define MANUAL_IRQ_ACK		/* see irq_thread() */
#endif

#define SANITY

/* synchronization primitives used later in this file: */



/*
 * synchronization state
 */

unsigned int local_bh_count[NR_CPUS];
unsigned int local_irq_count[NR_CPUS];

#ifdef __SMP__
atomic_t global_irq_count;
atomic_t global_bh_count;
atomic_t global_bh_lock;
#endif /* __SMP__ */


/*
 * synchronization primitives for the TAME_SERVER case: 
 */

#ifdef TAME_SERVER

/* When TAME_SERVER has been defined, cli() and sti() will use a queued
   lock instead of interrupt control.  Access to the lock queue is
   synchronized using the {lock,unlock}_q() functions, which we're
   going to define next. */

# ifdef __SMP__

static spinlock_t int_lock_lock = SPIN_LOCK_UNLOCKED;

#  define lock_q() spin_lock(&int_lock_lock);
#  define unlock_q() spin_unlock(&int_lock_lock);

# else /* ! __SMP__ */

#  define lock_q() __haifischflosse_cli()
#  define unlock_q() __haifischflosse_sti()

# endif /* ! __SMP__ */

/* The global interrupt lock. */

static l4_threadid_t int_lock_owner = L4_INVALID_ID;

struct next_locker
{
  l4_threadid_t id;
  int queued;
  struct next_locker *next;
};

static struct next_locker *int_lock_queue_head = 0;
static struct next_locker *int_lock_queue_tail = 0;

void 
__global_cli(void)
{
  l4_thread_id me;

#ifdef __SMP__
  /* when called from an interrupt context, do nothing */
  if (local_irq_count[smp_processor_id()])
    return;
#endif

  me = get_l4_id_from_stack();

  lock_q();

  if (int_lock_owner == L4_INVALID_ID)
    {
      /* the lock is free -- grab it */
      int_lock_owner = me;
      unlock_q();
    }
  else if (int_lock_owner == me)
    {
      /* we already are the owner of the lock */
      unlock_q();
      return;
    }
  else
    {
      /* wait for someone to pass the lock to us */

      l4_threadid_t waker;
      struct next_locker locker;
      l4_msgdope_t result;
      dword_t data1, data2;

      locker.id = me;
      locker.queue = 1;
      locker.next = 0;

      if (!int_lock_queue_tail)
	{
	  int_lock_queue_head = &locker;
	  waker = int_lock_owner;
	}
      else
	{
	  int_lock_queue_tail->next = &locker;
	  waker = int_lock_queue_tail->id;
	}
      int_lock_queue_tail = &this;
      
      /* end of critical region */
      unlock_q();

      /* go to sleep */
#ifdef SANITY
      if (waker.lh.low == -1)
	enter_kdebug("waker == -1");
#endif

      if (l4_i386_ipc_receive(waker, L4_IPC_SHORT_MSG,
			      &data1, &data2,
			      L4_IPC_NEVER, &result))
	{
#ifdef SANITY
	  enter_kdebug("cli: receive error");
#endif
	}
#ifdef SANITY
      else
	{
	  if (! (data1 == 0xce11dead && data2 == 0xc001b001))
	    enter_kdebug("cli: wakeup wrong");
	}
#endif
      /* we'll be int_lock_owner when we return */

#ifdef SANITY
      if (this.queued)
	enter_kdebug("cli: still queued");

      if (! thread_equal(int_lock_owner, me))
	enter_kdebug("cli: early wakeup");
#endif
    }

#ifdef __SMP__
  /* we newly acquired the lock.  make sure we don't return before all
     bottom-halves and top-halves have executed to their end.  (of
     course, if we're a bottom-half ourselves, we only have to check
     for top-halves. */

# error "release lock again, wait, try again"

#endif /* __SMP__ */
} /* __global_cli */

void
__global_sti()
{
  if (int_lock_owner != get_l4_id_from_stack())
    return;			/* we're not holding the lock */

#ifdef __SMP__
#ifdef SANITY
  if (local_irq_count[smp_processor_id()])
    enter_kdebug("irq context owns irq lock");
#endif
#endif

  lock_q();
  if (! int_lock_queue_head)
    {
      int_lock_owner = L4_INVALID_ID;
    }
  else
    {
      struct next_locker *n;
      l4_msgdope_t result;

      /* find next thread in queue and make it the irq-lock owner */
      n = int_lock_queue_head;
      if ((int_lock_queue_head = n->next) == 0)
	int_lock_queue_tail = 0;
      
      interrupt_lock_owner = n->id;
      n->queued = 0;
      
      /* give it a wake-up call */
      if (l4_i386_ipc_send(n->id, L4_IPC_SHORT_MSG, 0xce11dead, 0xc001b001,
			   L4_IPC_NEVER, &result))
	{
#ifdef SANITY
	  enter_kdebug("sti: send error");
#endif
	}
    }

  unlock_q();
} /* __global_sti */

unsigned long
__global_save_flags(void)
{
  if (local_irq_count[smp_processor_id()])
    return 2;

  return (int_lock_owner == get_l4_id_from_stack()) ? 1 : 0;
} /* __global_save_flags */


void
__global_restore_flags(unsigned long flags)
{
  switch (flags)
    {
    case 0:
      __global_sti(); 
      break;

    case 1:    
      __global_cli(); 
      break;

    default:			/* nothing */
    }
} /* __global_restore_flags */

#endif /* TAME_SERVER */



/*
 * synchronization of interrupt handlers
 * logic from arch/i386/kernel/irq.c
 */

extern inline void 
irq_enter(int cpu, unsigned int irq)
{
  ++local_irq_count[cpu];
#ifdef __SMP__
  atomic_inc(&global_irq_count);

  /* as long as anyone holds the irq lock, we must not enter the
     interrupt context. */
  while (int_lock_owner != L4_INVALID_ID) 
    mb();			/* do nothing */
#endif /* __SMP__ */
} /* irq_enter */

extern inline void 
irq_exit(int cpu, unsigned int irq)
{
#ifdef __SMP__
  atomic_dec(&global_irq_count);
#endif
  --local_irq_count[cpu];
} /* irq_exit */



/*
 * IRQ line status.
 */
#define IRQ_INPROGRESS	1	/* IRQ handler active - do not enter! */
#define IRQ_DISABLED	2	/* IRQ disabled - do not enter! */
#define IRQ_PENDING	4	/* IRQ pending - replay on enable */
#define IRQ_REPLAY	8	/* IRQ has been replayed but not acked yet */
#define IRQ_AUTODETECT	16	/* IRQ is being autodetected */

/*
 * This is the "IRQ descriptor", which contains various information
 * about the irq, including what kind of hardware handling it has,
 * whether it is disabled etc etc.
 *
 * Pad this out to 32 bytes for cache and indexing reasons.
 */
typedef struct {
	unsigned int status;	/* IRQ status - IRQ_INPROGRESS, IRQ_DISABLED */
	struct irqaction *action; /* IRQ action list */
	unsigned int depth;	/* Disable depth for nested irq disables */
	unsigned _padding;
} irq_desc_t;

/* a handler we install if we can't attach to a given interrupt in
   order to prevent probes and requests for this irq */
static void no_action(int cpl, void *dev_id, struct pt_regs *regs) { }
static struct irqaction irq_unassigned = { no_action, 0, 0, "non-Linux", };

/* the array of irq descriptors */
static irq_desc_t irq_desc[NR_IRQS] 
  = { [0 ... NR_IRQS-1] = {IRQ_DISABLED, &irq_unassigned, 0}};

/* this lock protects access to irq-controlling data */
#ifdef TAME_SERVER
static spinlock_t irq_controller_lock = SPIN_LOCK_UNLOCKED;
#endif

/*
 * code to enable or disable specific irqs 
 */

/* In wise anticipation of a time in which L4/x86 will handle all the
   interrupt controllers itself, we do not mask interrupts in the PIC.
   Instead, we maintain a mask register ourselves which is honored by
   the interrupt threads. */

static unsigned long masked_irqs = ~0L;
static unsigned long blocked_irq_thread = 0;

static void 
disable_irq_hard(unsigned int irq_nr)
{
  set_bit(irq_nr, &masked_irqs);
}

static void 
enable_irq_hard(unsigned int irq_nr)
{
  clear_bit(irq_nr, &masked_irqs);
      
  if (test_bit(irq_nr, &blocked_irq_thread))
    {
      l4_threadid_t irq_thread;
      l4_msgdope_t dummydope;
      
      clear_bit(irq_nr, &blocked_irq_thread);
      
      irq_thread = get_l4_id_from_stack();
      irq_thread.id.lthread = LTHREAD_NO_IRQ(irq_nr);
      
      /* give the irq thread a wakeup call */
      l4_i386_ipc_send(irq_thread,
		       0, 0x46464646, 0x4b4b4b4b,
		       L4_IPC_NEVER,
		       &dummydope);
    }
}

/* A globally-visible routine to disable a given hardware interrupt.
   Calls to the pair disable_irq/enable_irq may be nested. */
void 
disable_irq(unsigned int irq)
{
  unsigned long flags;
  
  spin_lock_irqsave(&irq_controller_lock, flags);
  if (!irq_desc[irq].depth++) 
    {
      irq_desc[irq].status |= IRQ_DISABLED;
      
      /* disable hard */
      disable_irq_hard(irq);
    }
  spin_unlock_irqrestore(&irq_controller_lock, flags);

  if (irq_desc[irq].status & IRQ_INPROGRESS) /* current irq in progress? */
    synchronize_irq();		/* return only when all interrupts are done */
} /* disable_irq */

/* Enable a hardware interrupt which has been disabled.  pay attention
   to nested calls of disable_irq. */
void 
enable_irq(unsigned int irq)
{
  unsigned long flags;
  
  spin_lock_irqsave(&irq_controller_lock, flags);
  switch (irq_desc[irq].depth) 
    {
    case 1:			/* enable hard */
      irq_desc[irq].status &= ~(IRQ_DISABLED | IRQ_INPROGRESS);
      enable_irq_hard(irq);

      /* fall through */
    default:
      irq_desc[irq].depth--;
      break;

    case 0:
      printk("enable_irq() unbalanced from %p\n",
	     __builtin_return_address(0));
    }

  spin_unlock_irqrestore(&irq_controller_lock, flags);
} /* enable_irq */

/* Suspend the current interrupt thread (L4 thread attached to an
   interrupt) until the (currently disabled) interrupt is enabled
   again.  Used in irq_thread() later in this file. */
extern inline void
sleep_while_disabled(unsigned irq)
{
  /* we're entered holding irq_controller_lock */

  if (test_bit(irq, &masked_irqs))
    {
      l4_threadid_t sender;
      l4_msgdope_t dummydope;
      int code;
      dword_t dummy;
      
      set_bit(irq, &blocked_irq_thread); /* indicate we're blocked */
      
#ifndef MANUAL_IRQ_ACK
      /* make sure the kernel doesn't unblock interrupt here */
      unregister_l4_irq_source();
#endif

      /* wait for wake-up message */
      do 
	{
	  spin_unlock(&irq_controller_lock);
	  
	  code = l4_i386_ipc_wait(&sender, 0, &dummy, &dummy, 
				  L4_IPC_NEVER, &dummydope);
#ifdef SANITY
	  if (sender.id.lthread == irq + 1)
	    enter_kdebug("blocked irq hit again");
#endif
	  
	  spin_lock(&irq_controller_lock);
	}
      while (test_bit(irq, &masked_irqs));

#ifndef MANUAL_IRQ_ACK
      /* unblock the interrupt again */
      register_l4_irq_source();
#endif

    }

  /* leave holding irq_controller_lock */

} /* sleep_while_disabled */



/* 
 * request_irq() and free_irq()
 */


/* the following code has been adapted from arch/i386/kernel/irq.[ch] */

/* add an interrupt handler; this routine is globally visible so that
   other l4-i386 modules (in particular, time.c) can use it */
int 
setup_x86_irq(unsigned int irq, struct irqaction * new)
{
	int shared = 0;
	struct irqaction *old, **p;
	unsigned long flags;

	/*
	 * Some drivers like serial.c use request_irq() heavily,
	 * so we have to be careful not to interfere with a
	 * running system.
	 */
	if (new->flags & SA_SAMPLE_RANDOM) {
		/*
		 * This function might sleep, we want to call it first,
		 * outside of the atomic block.
		 * Yes, this might clear the entropy pool if the wrong
		 * driver is attempted to be loaded, without actually
		 * installing a new handler, but is this really a problem,
		 * only the sysadmin is able to do this.
		 */
		rand_initialize_irq(irq);
	}

	/*
	 * The following block of code has to be executed atomically
	 */
	spin_lock_irqsave(&irq_controller_lock,flags);
	p = &irq_desc[irq].action;
	if ((old = *p) != NULL) {
		/* Can't share interrupts unless both agree to */
		if (!(old->flags & new->flags & SA_SHIRQ)) {
			spin_unlock_irqrestore(&irq_controller_lock,flags);
			return -EBUSY;
		}

		/* add new interrupt at end of irq queue */
		do {
			p = &old->next;
			old = *p;
		} while (old);
		shared = 1;
	}

	*p = new;

	if (!shared) {
		irq_desc[irq].depth = 0;
		irq_desc[irq].status &= ~(IRQ_DISABLED | IRQ_INPROGRESS);
		enable_irq_hard(irq);
	}
	spin_unlock_irqrestore(&irq_controller_lock,flags);
	return 0;
}

int request_irq(unsigned int irq, 
		void (*handler)(int, void *, struct pt_regs *),
		unsigned long irqflags, 
		const char * devname,
		void *dev_id)
{
	int retval;
	struct irqaction * action;

	if (irq >= NR_IRQS)
		return -EINVAL;
	if (!handler)
		return -EINVAL;

	action = (struct irqaction *)
			kmalloc(sizeof(struct irqaction), GFP_KERNEL);
	if (!action)
		return -ENOMEM;

	action->handler = handler;
	action->flags = irqflags;
	action->mask = 0;
	action->name = devname;
	action->next = NULL;
	action->dev_id = dev_id;

	retval = setup_x86_irq(irq, action);

	if (retval)
		kfree(action);
	return retval;
}
		
void free_irq(unsigned int irq, void *dev_id)
{
	struct irqaction * action, **p;
	unsigned long flags;

	if (irq >= NR_IRQS)
		return;

	spin_lock_irqsave(&irq_controller_lock,flags);
	for (p = &irq_desc[irq].action; (action = *p) != NULL; p = &action->next) {
		if (action->dev_id != dev_id)
			continue;

		/* Found it - now free it */
		*p = action->next;
		kfree(action);
		if (!irq_desc[irq].action) {
			irq_desc[irq].status |= IRQ_DISABLED;
			disable_irq_hard(irq);
		}
		goto out;
	}
	printk("Trying to free free IRQ%d\n",irq);
out:
	spin_unlock_irqrestore(&irq_controller_lock,flags);
}

/*
 * IRQ autodetection code..
 *
 * This depends on the fact that any interrupt that
 * comes in on to an unassigned handler will get stuck
 * with "IRQ_INPROGRESS" asserted and the interrupt
 * disabled.
 */
unsigned long probe_irq_on(void)
{
	unsigned int i;
	unsigned long delay;

	/*
	 * first, enable any unassigned irqs
	 */
	spin_lock_irq(&irq_controller_lock);
	for (i = NR_IRQS-1; i > 0; i--) {
		if (!irq_desc[i].action) {
			unsigned int status = irq_desc[i].status | IRQ_AUTODETECT;
			irq_desc[i].status = status & ~IRQ_INPROGRESS;
			enable_irq_hard(i);
		}
	}
	spin_unlock_irq(&irq_controller_lock);

	/*
	 * Wait for spurious interrupts to trigger
	 */
	for (delay = jiffies + HZ/10; time_after(delay, jiffies); )
		/* about 100ms delay */ synchronize_irq();

	/*
	 * Now filter out any obviously spurious interrupts
	 */
	spin_lock_irq(&irq_controller_lock);
	for (i=0; i<NR_IRQS; i++) {
		unsigned int status = irq_desc[i].status;

		if (!(status & IRQ_AUTODETECT))
			continue;
		
		/* It triggered already - consider it spurious. */
		if (status & IRQ_INPROGRESS) {
			irq_desc[i].status = status & ~IRQ_AUTODETECT;
			disable_irq_hard(i);
		}
	}
	spin_unlock_irq(&irq_controller_lock);

	return 0x12345678;
}

int probe_irq_off(unsigned long unused)
{
	int i, irq_found, nr_irqs;

	if (unused != 0x12345678)
		printk("Bad IRQ probe from %lx\n", (&unused)[-1]);

	nr_irqs = 0;
	irq_found = 0;
	spin_lock_irq(&irq_controller_lock);
	for (i=0; i<NR_IRQS; i++) {
		unsigned int status = irq_desc[i].status;

		if (!(status & IRQ_AUTODETECT))
			continue;

		if (status & IRQ_INPROGRESS) {
			if (!nr_irqs)
				irq_found = i;
			nr_irqs++;
		}
		irq_desc[i].status = status & ~IRQ_AUTODETECT;
		disable_irq_hard(i);
	}
	spin_unlock_irq(&irq_controller_lock);

	if (nr_irqs > 1)
		irq_found = -irq_found;
	return irq_found;
}


int get_irq_list(char *buf)
{
	int i, j;
	struct irqaction * action;
	char *p = buf;

	p += sprintf(p, "           ");
	for (j=0; j<smp_num_cpus; j++)
		p += sprintf(p, "CPU%d       ",j);
	*p++ = '\n';

	for (i = 0 ; i < NR_IRQS ; i++) {
		action = irq_desc[i].action;
		if (!action) 
			continue;
		p += sprintf(p, "%3d: ",i);
#ifndef __SMP__
		p += sprintf(p, "%10u ", kstat_irqs(i));
#else
		for (j=0; j<smp_num_cpus; j++)
			p += sprintf(p, "%10u ",
				kstat.irqs[cpu_logical_map(j)][i]);
#endif
#if 0  /* ! L4Linux */
		p += sprintf(p, " %14s", irq_desc[i].handler->typename);
#else  /* L4Linux */
		p += sprintf(p, i == TIMER_IRQ ? " L4-IPC-timeout"
			                       : " L4-IPC-IRQ    ");
#endif
		p += sprintf(p, "  %s", action->name);

		for (action=action->next; action; action = action->next) {
			p += sprintf(p, ", %s", action->name);
		}
		*p++ = '\n';
	}
#if 0  /* ! L4Linux */
	p += sprintf(p, "NMI: %10u\n", atomic_read(&nmi_counter));
#endif
#ifdef __SMP__
	p += sprintf(p, "ERR: %10lu\n", ipi_count);
#endif		
	return p - buf;
}


/* 
 * Interrupt and bottom-half handlers and threads 
 */

#ifdef SANITY
static unsigned irqs_in_progress;
#endif

/* bottom half thread.

   Note: this must be run at some higher priority than regular kernel
   threads (but at lower priority than interrupt threads) because
   bottom half handlers expect not to be interrupted by anything but
   interrupt handlers. */

static l4_threadid_t bh_th_id;

static void 
bottom_half_thread(void)
{
  int code;
  dword_t dummy;
  l4_msgdope_t dummydope;
  l4_threadid_t sender;

  set_default_idt();

  /* wait for an activation from an interrupt thread, then try to
     execute bottom halfs */
  
  for (;;)
    {
#ifdef SANITY
      irqs_in_progress &= ~(1 << 16); /* last irq + 1 */
#endif
      code = l4_i386_ipc_wait(&sender, 0,
			      &dummy, &dummy,
			      L4_IPC_NEVER, &dummydope);
      if (code)
	continue;		/* Hmm... */

      /* verify we've been activated from some local thread */
      if (! thread_equal(get_taskid(sender), get_taskid(bh_th_id)))
	continue;		/* Hmm... */

      code = l4_i386_ipc_send(sender, 0,
			      0xaffeaffe, 0xaffeaffe,
			      L4_IPC_NEVER, &dummydope);
#ifdef SANITY
      if (code)
	enter_kdebug("bh: reply failed");

      irqs_in_progress |= (1 << 16); /* last irq + 1 */
#endif

      __cli();

      if (bh_active & bh_mask)
	{
	  do_bottom_half();
	}

      __sti();

      /* check whether we have to wakeup our service thread due to 
	 need_resched */
      wakeup_idle_if_needed();      
    }
}

/* execute bottom halfs */
static inline
void execute_bottom_halves(void) 
{
  /* this code is called from the interrupt thread.  in order to
     prevent preventing the irq thread from receiving more interrupts,
     we can't execute bottom halfs right here.  Instead, notify a
     special bottom half thread. */
  dword_t dummy;
  if (bh_active & bh_mask)
    {
      l4_msgdope_t dummydope;

      l4_i386_ipc_call(bh_th_id, 0,
		       0, 0,
		       0, &dummy, &dummy,
		       L4_IPC_TIMEOUT(0,1,0,0,0,0), /* rcv = inf,
						       snd = 0 */
		       &dummydope);

      /* it is OK if the previous IPC times out.  This just means the
         bottom half thread is still busy, and it will eventually
         handle all outstanding requests */

    }
  else
    {
      /* check whether we have to wakeup our service thread due to 
	 need_resched */
      wakeup_idle_if_needed();      
    }
}


/* low-level interrupt handler thread; must be re-entrant 

   Note: irq handlers expect not to be interrupted except by other
   interrupts.  That's why, this thread needs to run with a relatively
   high priority (when __SMP__ is not defined). */

/* helpers  */

/* attach current L4 thread to an interrupt source.  This is done
   using the corresponding L4 kernel operation.  Before we try to do
   that, however, we try to request ownership of that irq from the
   resources manager we may be running under.  */
__initfunc(
static void
init_irq_thread(unsigned irq, l4_threadid_t sync_with_me,
		l4_threadid_t *irq_th))
{
  dword_t dummy, d1;
  l4_msgdope_t dummydope;
  int code;

  /* first, register to the irq */

  /* in case we're running under a RMGR, fetch the irq nr first from it */
  d1 = rmgr_get_irq(irq);
  if (d1 != 0)
    {
      /* the supervisor thread exists and sent a negative response */
      printk("irq_thread: RMGR denied irq %u: code 0x%x\n", irq, d1);
      goto error;
    }

  irq_th->lh.low = irq + 1;
  irq_th->lh.high = 0;

  code = l4_i386_ipc_receive(*irq_th,
			     0, /* receive descriptor */
			     &dummy,
			     &dummy,
			     L4_IPC_TIMEOUT(0,0,0,1,0,0), /* rcv = 0,
							     snd = inf */
			     &dummydope);
  if (code != L4_IPC_RETIMEOUT)
    {
      printk("irq_thread: can't register to irq %u: error 0x%x\n", irq, 
	     (unsigned) code);

    error:
      l4_i386_ipc_send(sync_with_me, 
		       0, 0xffffffff, 0xffffffff,
		       L4_IPC_NEVER, &dummydope);

      /* XXX: We have a race here: this function's code will be thrown
         away after initialization.  We hope we have already entered
         the sleep-forever IPC below at that point. */

      /* sleep forever */
      l4_i386_ipc_receive(L4_NIL_ID, 0,
			  &dummy, &dummy,
			  L4_IPC_NEVER,
			  &dummydope);
    }

  /* synchronize with the thread that started us */
  l4_i386_ipc_send(sync_with_me, 
		   0, 0, 0,
		   L4_IPC_NEVER, &dummydope);

  /* OK, now get ready to take irq handlers */
  irq_desc[irq].action = 0;
} /* init_irq_thread */

/* this operation unblocks a hardware interrupt source and suspends
   the current L4 thread until it receives an interrupt message.
   Then, it blocks the interrupt and acknowledges it.  Ideally, the L4
   kernel would do the irq unblocking/blocking/acknowledging for us,
   but until it does, we do it ourselves here.  */
extern inline void
receive_irq_message(unsigned irq, l4_threadid_t irq_th)
{
  dword_t dummy;
  l4_msgdope_t dummydope;
  int code;
#ifdef MANUAL_IRQ_ACK
  unsigned char mask;
#endif

#ifdef TAME_SERVER
  __sti();
#endif

  for(;;)
    {
#ifdef MANUAL_IRQ_ACK
      /* this code will eventually go away when L4 will unblock/acknowledge
	 irqs itself */

      __haifischflosse_cli();	/* run with disabled irqs to avoid races
				   when we acknowledge the interrupt later */
      
      // spin_lock(&irq_controller_lock); /* not necessary: we're cli'd */

      mask = ~(1 << (irq & 7));
      if (irq < 8) 
	outb(inb(0x21) & mask, 0x21);
      else
	outb(inb(0xA1) & mask, 0xA1);

      // spin_unlock(&irq_controller_lock); /* not necessary: we're cli'd */
#endif /* MANUAL_IRQ_ACK */
      
#ifdef SANITY
      irqs_in_progress &= ~(1 << irq);
#endif

      code = l4_i386_ipc_receive(irq_th,
				 0, /* receive descriptor */
				 &dummy,
				 &dummy,
				 L4_IPC_NEVER,
				 &dummydope);
      if (code == 0)
	break;

      printk("irq_thread: irq %u (%x:%x) receive failed, code = 0x%x", 
	     irq, irq_th.lh.high, irq_th.lh.low,
	     (unsigned) code);
#ifdef SANITY
      enter_kdebug("receive from intr failed");
#endif
    }

#ifdef MANUAL_IRQ_ACK
  /* this code will eventually go away when L4 will acknowledge
     irqs itself */
  
  /* we assume we're still cli'd here; we definitely don't want
     any other L4 task to manipulate the PIC at the same time */
  
  // spin_lock(&irq_controller_lock); /* not necessary: we're cli'd */
  
  mask = 1 << (irq & 7);
  if (irq < 8)
    {
      outb(inb(0x21) | mask, 0x21);	/* block the irq */
      outb(0x20, 0x20);	/* acknowledge the irq */
    }
  else 
    {
      outb(inb(0xA1) | mask, 0xA1);	/* block */
      outb(0x20, 0xA0);	/* acknowledge */
      outb(0x0B, 0xA0);
      if (inb(0xA0) == 0)
	outb(0x20, 0x20);
    }      
  
  // spin_unlock(&irq_controller_lock); /* not necessary: we're cli'd */
  
#ifdef TAME_SERVER
  __haifischflosse_sti();
  __cli();
#endif
#endif /* MANUAL_IRQ_ACK */
  
#ifdef SANITY
  irqs_in_progress |= (1 << irq);
#endif
} /* receive_irq_message */

/* handle an irq event: look up handlers for it, and if we find any,
   run them.  */
extern inline void
handle_irq(unsigned irq, int cpu)
{
  /* we're entered holding irq_controller_lock, and __cli'd. */

  unsigned int status;
  struct irqaction *action;

  /* see if we have been soft-blocked -- logic from
     arch/i386/kernel/irq.c: do_8259A_IRQ() */
  
  action = (irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS))
    ? 0 : irq_desc[irq].action;
  irq_desc[irq].status = (irq_desc[irq].status & ~IRQ_REPLAY) | IRQ_INPROGRESS;

  spin_unlock(&irq_controller_lock);
  
  if (! action) 
    return;		/* we're soft-blocked, or there is no action */
  
  /* we are not blocked -- call the handler; logic from
     arch/i386/kernel/irq.c: handle_IRQ_event() */
  irq_enter(cpu, irq);
  
  if (! (action->flags & SA_INTERRUPT))
    __sti();

  status = 0;
  do 
    {
      status |= action->flags;
      action->handler(irq, action->dev_id, 0); /* XXX regs = 0 */
      action = action->next;
    }
  while (action);
  
  if (status & SA_SAMPLE_RANDOM) add_interrupt_randomness(irq);
  
  __cli();
  irq_exit(cpu, irq);
  
  spin_lock(&irq_controller_lock);
  irq_desc[irq].status = irq_desc[irq].status & ~IRQ_INPROGRESS;

  /* leave holding irq_controller_lock */

} /* handle_irq */

/* the main irq-thread function */

static void 
irq_thread(unsigned irq, l4_threadid_t sync_with_me)
{
  l4_threadid_t irq_th;
  int cpu = smp_processor_id();

  init_irq_thread(irq, sync_with_me, &irq_th);

  /* initialization complete -- now wait for irq messages and handle
     them appropriately */

  for (;;) 
    {
      receive_irq_message(irq, irq_th); /* unblocks, receives, blocks,
                                           & acknowledges irq */

      spin_lock(&irq_controller_lock);

      kstat.irqs[cpu][irq]++;

      sleep_while_disabled(irq); /* if irq has been hard-disabled, sleep */

      handle_irq(irq, cpu);	/* we're unblocked -- execute handler */

      spin_unlock(&irq_controller_lock);
  
      execute_bottom_halves();
    }
} /* irq_thread */

/* low-level timer-interrupt handler thread */
static void 
timer_irq_thread(void)
{
  /* we can't grab the timer interrupt 0, as L4 needs it for itself.
     So we try to wake up every 1000/HZ = 10 ms. */

#if (HZ != 100)
# error "adapt me...  This code only works for HZ == 100"
#endif

  dword_t dummy;
  l4_msgdope_t dummydope;
  unsigned long total, newtotal;
  unsigned long totallost;

  extern unsigned long lost_ticks; /* in kernel/sched.c */

  int cpu = smp_processor_id();
  const unsigned long jiffie_usecs = 1000000L / HZ;
  
/*   set_default_idt(); */

  totallost = 0;
  total = l4_kernel_info->clock;

#if !defined(TAME_SERVER) && defined(MANUAL_IRQ_ACK)
  __haifischflosse_cli();	/* we run with irq disabled by default */
#endif

  /* OK, now get ready to take irq handlers */
  irq_desc[TIMER_IRQ].action = 0;

  for (;;)
    {
#ifdef SANITY
      irqs_in_progress &= ~(1 << TIMER_IRQ);
#endif

      l4_i386_ipc_receive(L4_NIL_ID, 0,
			  &dummy, &dummy,
			  L4_IPC_TIMEOUT(0,0,4*39,12,0,0), /* ca. 10 ms */
			  &dummydope);
#ifdef SANITY
      irqs_in_progress |= (1 << TIMER_IRQ);
#endif

      /* synchronize jiffies/lost_ticks with real-time clock */
      newtotal = l4_kernel_info->clock;

      totallost += (newtotal - total); /* usecs we slept */
      total = newtotal;

      if (totallost < jiffie_usecs)
	continue;		/* just sleep a little longer in order
				   to prevent negative jiffie updates */
      totallost -= jiffie_usecs; /* account for irq handler's
				    update of jiffies */
      if (totallost > jiffie_usecs)
	{
	  unsigned long jiffies_catchup = totallost / jiffie_usecs;

	  totallost -= jiffies_catchup * jiffie_usecs;
	  jiffies += jiffies_catchup;
	  lost_ticks += jiffies_catchup;
	}

      /* call irq handler */

      spin_lock(&irq_controller_lock);

      kstat.irqs[cpu][TIMER_IRQ]++;

      // sleep_while_disabled(irq); /* if irq has been hard-disabled, sleep */
      
      handle_irq(TIMER_IRQ, cpu); /* we're unblocked -- execute handler */

      spin_unlock(&irq_controller_lock);
  
      execute_bottom_halves();
    }
} /* timer_irq_thread */


/* 
 * Interrupt initialization
 */

/* fake handlers for useless irqs */

/* IRQ2 is cascade interrupt to second interrupt controller */
static struct irqaction irq2  = { no_action, 0, 0, "cascade", NULL, NULL};

/* Note that on a 486, we don't want to do a SIGFPE on a irq13 as the
   irq is unreliable, and exception 16 works correctly (ie as
   explained in the intel literature). */
 
static void math_error_irq(int cpl, void *dev_id, struct pt_regs *regs)
{
  outb(0,0xF0);			/* XXX necessary? */
}

static struct irqaction irq13 = { math_error_irq, 0, 0, "math error", 
				    NULL, NULL };
/* initialize IRQ system */

extern union task_union interrupt_stacks[NR_IRQS + 1]; /* in init_task.c */

__initfunc(
void 
init_IRQ(void))
{
  const unsigned irq_stack_size = sizeof(union task_union);

  unsigned stack_page = (unsigned) interrupt_stacks;

  unsigned i;
  unsigned *sp;
  dword_t dummy, routine;
  l4_msgdope_t dummydope;

  l4_threadid_t preempter, pager, my_preempter, my_pager;

  /* get thread parameters */
  l4_threadid_t me = l4_myself();
  my_preempter = L4_INVALID_ID;
  my_pager = L4_INVALID_ID;
  l4_thread_ex_regs(me, (dword_t) -1, (dword_t) -1,
		    &my_preempter,
		    &my_pager,
		    &dummy,
		    &dummy);

  /* start bottom-half thread */

  /* save thread id */
  bh_th_id = me;
  bh_th_id.id.lthread = LTHREAD_NO_BOTTOM_HALF;

  /* allocate stack page and space for the thread id*/
  sp = (unsigned *) (stack_page + irq_stack_size - sizeof(l4_threadid_t));

  /* put thread id to top of stack */
  put_l4_id_to_stack((unsigned)sp, bh_th_id);

  /* push parameters on the stack */
  *(--sp) = 0;			/* fake return address */


  /* create thread */
  preempter = my_preempter;
  pager = my_pager;
  l4_thread_ex_regs(bh_th_id,
		    (dword_t) bottom_half_thread, /* eip */
		    (dword_t) sp, /* esp */
		    &preempter, /* preempter */
		    &pager,	/* pager */
		    &dummy,	/* old_eip */
		    &dummy);	/* old_esp */

#ifdef USE_PRIOS
  rmgr_set_prio(bh_th_id, PRIO_BOTTOM_HALF);
#endif

  /* set up interrupt vectors, and start interrupt threads */
  for (i = 0; i < 16 ; i++)
    {
      l4_threadid_t newthid;

      /* create thread */
      newthid = me;
      newthid.id.lthread = LTHREAD_NO_IRQ(i);

      /* allocate stack page and space for the thread id*/
      stack_page += irq_stack_size;
      sp = (unsigned *) (stack_page + irq_stack_size - sizeof(l4_threadid_t)); 
            
      /* put thread id on bottom of stack */
      put_l4_id_to_stack((unsigned)sp, newthid);

      /* push parameters on the stack */
      *(--sp) = newthid.lh.high; /* push the thread id */
      *(--sp) = newthid.lh.low;

      if (i == TIMER_IRQ)	/* must be handled differently */
	{
	  routine = (dword_t) timer_irq_thread;
	}
      else
	{
	  *(--sp) = me.lh.high;	/* pass my thread id as arg for sync */
	  *(--sp) = me.lh.low;

	  *(--sp) = i;		/* pass irq number as arg */

	  routine = (dword_t) irq_thread;
	}

      *(--sp) = 0;		/* fake return address */

      preempter = my_preempter;
      pager = my_pager;

      l4_thread_ex_regs(newthid,
			routine, /* eip */
			(dword_t) sp, /* esp */
			&preempter, /* preempter */
			&pager, /* pager */
			&dummy, /* old_eip */
			&dummy); /* old_esp */

#ifdef USE_PRIOS
      /* set prio */
      rmgr_set_prio(newthid, PRIO_IRQ(i));
#endif
  
      /* give the new thread some time to initialize: wait for
         synchronization message */
      if (i != TIMER_IRQ)
	l4_i386_ipc_receive(newthid, 
			    0, &dummy, &dummy,
			    L4_IPC_NEVER,
			    &dummydope);
      
    }
#ifdef USE_PRIOS
  rmgr_set_prio(l4_myself(), PRIO_KERNEL);
#endif

  request_region(0x20,0x20,"pic1");
  request_region(0xa0,0x20,"pic2");
  setup_x86_irq(2, &irq2);
  setup_x86_irq(13, &irq13);
}
