/* $Id: softirq.h,v 1.1.1.1 1999/04/06 19:49:01 yoonho Exp $ */

#ifndef __ASM_L4_I386_SOFTIRQ_H
#define __ASM_L4_I386_SOFTIRQ_H

/* bottom-half interrupt handling synchronization

   we need to define the following interfaces:

   get_active_bhs() returns a bit-vector of currently active (and
   unmasked) botton-halfs.

   clear_active_bhs() removes a given set of bottom-halfs from the
   active set.

   init_bh() initializes a given bottom-half number with a routine to
   be called.

   remove_bh() removes a bottom-half handler.

   mark_bh() adds a bottom-half to the set of active bottom-halves.

   enable_bh() and disable_bh() temporarily mask or unmask a
   bottom-half.  Calls to this pair of routines may be nested.

   start_bh_atomic() increases the bottom-half counter, stopping
   bottom-halves from executing.  if a bottom-half is currently
   executing, waits until it is done.

   end_bh_atomic() decreases the bottom-half counter.  if the counter
   reaches 0, bottom-halves may execute.

   softirq_trylock() acquires the bottom-half lock if the bottom-half
   counter is 0.  If successful, it returns true.  This routine must
   return true before bottom-halves may be executed.

   softirq_endlock() releases the bottom-half lock.

   synchronize_bh() waits until no CPU is executing bottom-halves. */

/* Our implementation looks a lot like the i386 implementation.
   However, I think that <asm-i386/softirq.h> uses the names
   global_bh_lock and global_bh_count in a reversed sense (~lock is
   used as count, and ~count is used as a lock), and I chose to revert
   that ugliness. */

#include <asm/hardirq.h>

extern unsigned int local_bh_count[NR_CPUS];

extern inline unsigned long
get_active_bhs(void)
{
  return (bh_mask & bh_active);
}

extern inline void
clear_active_bhs(unsigned long x)
{
  atomic_clear_mask(x, &bh_active);
}

extern inline void 
init_bh(int nr, void (*routine)(void))
{
  bh_base[nr] = routine;
  atomic_set(&bh_mask_count[nr], 0);
  bh_mask |= 1 << nr;
}

extern inline void 
remove_bh(int nr)
{
  bh_mask &= ~(1 << nr);
  mb();
  bh_base[nr] = NULL;
}

extern inline void 
mark_bh(int nr)
{
  set_bit(nr, &bh_active);
}

#ifdef __SMP__

extern atomic_t global_bh_lock;
extern atomic_t global_bh_count;

extern inline void 
synchronize_bh(void)
{
  if (atomic_read(&global_bh_lock) && !in_interrupt())
    while (atomic_read(&global_bh_lock))
      mb();			/* do nothing -- wait until bh's finish */
}

extern inline void 
start_bh_atomic(void)
{
  atomic_inc(&global_bh_count);
  synchronize_bh();
}

extern inline void 
end_bh_atomic(void)
{
  atomic_dec(&global_bh_count);
}

/* These are for the IRQs testing the lock */
extern inline int 
softirq_trylock(int cpu)
{
  if (!test_and_set_bit(0,&global_bh_lock)) 
    {
      if (atomic_read(&global_bh_count) == 0) 
	{
	  ++local_bh_count[cpu];
	  return 1;
	}
      clear_bit(0,&global_bh_lock);
    }
  return 0;
}

extern inline void 
softirq_endlock(int cpu)
{
  local_bh_count[cpu]--;
  clear_bit(0,&global_bh_lock);
}

#else  /* ! __SMP__ */

extern inline void 
synchronize_bh(void)
{
  barrier();
}

extern inline void 
start_bh_atomic(void)
{
  local_bh_count[smp_processor_id()]++;
  barrier();
}

extern inline void 
end_bh_atomic(void)
{
  barrier();
  local_bh_count[smp_processor_id()]--;
}

/* These are for the irq's testing the lock */
extern inline int 
softirq_trylock(int cpu)
{
  if (local_bh_count[cpu])
    return 0;
  
  local_bh_count[cpu] = 1;
  return 1;
}

extern inline void 
softirq_endlock(int cpu)
{
  local_bh_count[cpu] = 0;
}

#endif /* ! __SMP__ */

/*
 * These use a mask count to correctly handle
 * nested disable/enable calls
 */
extern inline void disable_bh(int nr)
{
  bh_mask &= ~(1 << nr);
  atomic_inc(&bh_mask_count[nr]);
  synchronize_bh();
}

extern inline void enable_bh(int nr)
{
  if (atomic_dec_and_test(&bh_mask_count[nr]))
    bh_mask |= 1 << nr;
}

#endif /* ! __ASM_L4_I386_SOFTIRQ_H */
