#ifndef __L4_i386_UACCESS_H
#define __L4_i386_UACCESS_H

/*
 * User space memory access functions
 */
#include <linux/config.h>
#include <linux/sched.h>
#include <asm/page.h>

#define VERIFY_READ 0
#define VERIFY_WRITE 1

/* we provide this data structure to keep kernel changes minimal */
struct exception_table_entry
{
	unsigned long insn, fixup;
};

/*
 * The fs value determines whether argument validity checking should be
 * performed or not.  If get_fs() == USER_DS, checking is performed, with
 * get_fs() == KERNEL_DS, checking is bypassed.
 *
 * For historical reasons, these macros are grossly misnamed.
 */

/* We have to duplicate the declaration of swapper_pg_dir, so if it
 * changes somehow, make sure you update both declarations. The other
 * (and main) decalaration is in pgtable.h 
 */
extern pgd_t swapper_pg_dir[1024];

#define MAKE_MM_SEG(s)	((mm_segment_t) { (s) })

#define KERNEL_DS	MAKE_MM_SEG(1)
#define USER_DS		MAKE_MM_SEG(0)

#define SWAPPER_PG_DIR	MAKE_MM_SEG((unsigned long)swapper_pg_dir)

#define get_ds()	(KERNEL_DS)
#define get_fs()	(current->tss.fs)

#define set_fs(x)							     \
do { 									     \
       current->tss.fs = (x); 						     \
       current->addr_limit = ( segment_eq((x),KERNEL_DS) ? SWAPPER_PG_DIR :  \
			       MAKE_MM_SEG((unsigned long)current->tss.cr3));\
} while (0);

#define set_tsk_fs(tsk,x) 						     \
do { 									     \
       tsk->tss.fs = (x); 						     \
       tsk->addr_limit = (segment_eq((x),KERNEL_DS) ? SWAPPER_PG_DIR : 	     \
			  MAKE_MM_SEG((unsigned long)tsk->tss.cr3)); 	     \
} while(0);

#define update_pgdir_cache(tsk)						     \
	tsk->addr_limit=(segment_eq(tsk->tss.fs,KERNEL_DS) ? SWAPPER_PG_DIR: \
			 MAKE_MM_SEG((unsigned long)tsk->tss.cr3));

#define segment_eq(a,b)	((a).seg == (b).seg)

#define access_ok(type,addr,size) (1)
#define verify_area(type,addr,size) (0)

/*
 * These are the main single-value transfer routines.  They automatically
 * use the right size if we just have the right pointer type.
 *
 * This gets kind of ugly. We want to return _two_ values in "get_user()"
 * and yet we don't want to do any pointers, because that is too much
 * of a performance impact. Thus we have a few rather ugly macros here,
 * and hide all the uglyness from the user.
 *
 * The "__xxx" versions of the user access functions are versions that
 * do not verify the address space, that must have been done previously
 * with a separate "access_ok()" call (this is used when we do multiple
 * accesses to the same area of user memory).
 */

int __get_user_byte(unsigned char *val,   const void *address);
int __get_user_short(unsigned short *val, const void *address);
int __get_user_long(unsigned long *val,   const void *address);
int __get_user_bad(void);
/* Careful: we have to cast the result to the type of the pointer for sign reasons */
#define get_user(x,ptr)							\
({	long __ret_gu, __gu_val;       					\
	switch(sizeof (*(ptr))) {					\
	case 1:  __ret_gu = __get_user_byte((unsigned char*)&__gu_val,ptr); break;  	\
	case 2:  __ret_gu = __get_user_short((unsigned short*)&__gu_val,ptr); break;	\
	case 4:  __ret_gu = __get_user_long((unsigned long*)&__gu_val,ptr); break;	\
	default: __ret_gu = __get_user_bad(); break;			\
	}								\
	(x) = (__typeof__(*(ptr)))__gu_val;    				\
	__ret_gu;							\
})

int __put_user_byte(unsigned char val,   const void *address);
int __put_user_short(unsigned short val, const void *address);
int __put_user_long(unsigned long val,   const void *address); 
int __put_user_bad(void);

#define put_user(x,ptr)									\
({	int __ret_pu;									\
	switch(sizeof (*(ptr))) {							\
	case 1:  __ret_pu = __put_user_byte((unsigned char)(unsigned long)(__typeof__(*(ptr)))(x),ptr); break;		\
	case 2:  __ret_pu = __put_user_short((unsigned short)(unsigned long)(__typeof__(*(ptr)))(x),ptr); break;		\
	case 4:  __ret_pu = __put_user_long((unsigned long)(__typeof__(*(ptr)))(x),ptr); break;		\
	default: __ret_pu = __put_user_bad(); break;					\
	}										\
	__ret_pu;									\
})

#define __get_user get_user
#define __put_user put_user
/*
 * The "xxx_ret" versions return constant specified in third argument, if
 * something bad happens. These macros can be optimized for the
 * case of just returning from the function xxx_ret is used.
 */

#define put_user_ret(x,ptr,ret) ({ if (put_user(x,ptr)) return ret; })
#define get_user_ret(x,ptr,ret) ({ if (get_user(x,ptr)) return ret; })

#define __put_user_ret put_user_ret
#define __get_user_ret get_user_ret


/*
 * Copy To/From Userspace
 */

int copy_to_user(void *to, const void *from, unsigned long n);
int copy_from_user(void *to, const void *from, unsigned long n);

#define copy_to_user_ret(to,from,n,retval) ({ if (copy_to_user(to,from,n)) return retval; })

#define copy_from_user_ret(to,from,n,retval) ({ if (copy_from_user(to,from,n)) return retval; })

#define __copy_to_user copy_to_user
#define __copy_from_user copy_from_user

long strncpy_from_user(char *dst, const char *src, long count);
#define __strncpy_from_user strncpy_from_user
long strlen_user(const char *str);
unsigned long clear_user(void *mem, unsigned long len);
#define __clear_user clear_user

#endif /* __L4_i386_UACCESS_H */
